/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.vespa.config.server.session;

import com.google.common.collect.HashMultiset;
import com.yahoo.cloud.config.ConfigserverConfig;
import com.yahoo.concurrent.DaemonThreadFactory;
import com.yahoo.concurrent.StripedExecutor;
import com.yahoo.config.FileReference;
import com.yahoo.config.application.api.ApplicationPackage;
import com.yahoo.config.application.api.DeployLogger;
import com.yahoo.config.model.api.ConfigDefinitionRepo;
import com.yahoo.config.model.application.provider.DeployData;
import com.yahoo.config.model.application.provider.FilesApplicationPackage;
import com.yahoo.config.provision.AllocatedHosts;
import com.yahoo.config.provision.ApplicationId;
import com.yahoo.config.provision.TenantName;
import com.yahoo.config.provision.Zone;
import com.yahoo.container.jdisc.secretstore.SecretStore;
import com.yahoo.io.IOUtils;
import com.yahoo.lang.SettableOptional;
import com.yahoo.path.Path;
import com.yahoo.transaction.AbstractTransaction;
import com.yahoo.transaction.NestedTransaction;
import com.yahoo.transaction.Transaction;
import com.yahoo.vespa.config.server.ConfigServerDB;
import com.yahoo.vespa.config.server.TimeoutBudget;
import com.yahoo.vespa.config.server.application.ApplicationSet;
import com.yahoo.vespa.config.server.application.PermanentApplicationPackage;
import com.yahoo.vespa.config.server.application.TenantApplications;
import com.yahoo.vespa.config.server.configchange.ConfigChangeActions;
import com.yahoo.vespa.config.server.deploy.TenantFileSystemDirs;
import com.yahoo.vespa.config.server.filedistribution.FileDirectory;
import com.yahoo.vespa.config.server.modelfactory.ActivatedModelsBuilder;
import com.yahoo.vespa.config.server.modelfactory.ModelFactoryRegistry;
import com.yahoo.vespa.config.server.monitoring.MetricUpdater;
import com.yahoo.vespa.config.server.monitoring.Metrics;
import com.yahoo.vespa.config.server.provision.HostProvisionerProvider;
import com.yahoo.vespa.config.server.session.LocalSession;
import com.yahoo.vespa.config.server.session.PrepareParams;
import com.yahoo.vespa.config.server.session.RemoteSession;
import com.yahoo.vespa.config.server.session.Session;
import com.yahoo.vespa.config.server.session.SessionPreparer;
import com.yahoo.vespa.config.server.session.SessionStateWatcher;
import com.yahoo.vespa.config.server.session.SessionZooKeeperClient;
import com.yahoo.vespa.config.server.tenant.TenantListener;
import com.yahoo.vespa.config.server.tenant.TenantRepository;
import com.yahoo.vespa.config.server.zookeeper.ConfigCurator;
import com.yahoo.vespa.config.server.zookeeper.SessionCounter;
import com.yahoo.vespa.curator.Curator;
import com.yahoo.vespa.curator.transaction.CuratorTransaction;
import com.yahoo.vespa.defaults.Defaults;
import com.yahoo.vespa.flags.FlagSource;
import com.yahoo.yolean.Exceptions;
import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.recipes.cache.ChildData;
import org.apache.curator.framework.recipes.cache.PathChildrenCacheEvent;
import org.apache.zookeeper.KeeperException;

public class SessionRepository {
    private static final Logger log = Logger.getLogger(SessionRepository.class.getName());
    private static final FilenameFilter sessionApplicationsFilter = (dir, name) -> name.matches("\\d+");
    private static final long nonExistingActiveSessionId = 0L;
    private final Object monitor = new Object();
    private final Map<Long, LocalSession> localSessionCache = Collections.synchronizedMap(new HashMap());
    private final Map<Long, RemoteSession> remoteSessionCache = Collections.synchronizedMap(new HashMap());
    private final Map<Long, SessionStateWatcher> sessionStateWatchers = Collections.synchronizedMap(new HashMap());
    private final Duration sessionLifetime;
    private final Clock clock;
    private final Curator curator;
    private final Executor zkWatcherExecutor;
    private final PermanentApplicationPackage permanentApplicationPackage;
    private final FlagSource flagSource;
    private final TenantFileSystemDirs tenantFileSystemDirs;
    private final Metrics metrics;
    private final MetricUpdater metricUpdater;
    private final Curator.DirectoryCache directoryCache;
    private final TenantApplications applicationRepo;
    private final SessionPreparer sessionPreparer;
    private final Path sessionsPath;
    private final TenantName tenantName;
    private final ConfigCurator configCurator;
    private final SessionCounter sessionCounter;
    private final SecretStore secretStore;
    private final HostProvisionerProvider hostProvisionerProvider;
    private final ConfigserverConfig configserverConfig;
    private final ConfigServerDB configServerDB;
    private final Zone zone;
    private final ModelFactoryRegistry modelFactoryRegistry;
    private final ConfigDefinitionRepo configDefinitionRepo;
    private final TenantListener tenantListener;

    public SessionRepository(TenantName tenantName, TenantApplications applicationRepo, SessionPreparer sessionPreparer, Curator curator, Metrics metrics, StripedExecutor<TenantName> zkWatcherExecutor, PermanentApplicationPackage permanentApplicationPackage, FlagSource flagSource, ExecutorService zkCacheExecutor, SecretStore secretStore, HostProvisionerProvider hostProvisionerProvider, ConfigserverConfig configserverConfig, ConfigServerDB configServerDB, Zone zone, Clock clock, ModelFactoryRegistry modelFactoryRegistry, ConfigDefinitionRepo configDefinitionRepo, TenantListener tenantListener) {
        this.tenantName = tenantName;
        this.configCurator = ConfigCurator.create(curator);
        this.sessionCounter = new SessionCounter(this.configCurator, tenantName);
        this.sessionsPath = TenantRepository.getSessionsPath(tenantName);
        this.clock = clock;
        this.curator = curator;
        this.sessionLifetime = Duration.ofSeconds(configserverConfig.sessionLifetime());
        this.zkWatcherExecutor = command -> zkWatcherExecutor.execute((Object)tenantName, command);
        this.permanentApplicationPackage = permanentApplicationPackage;
        this.flagSource = flagSource;
        this.tenantFileSystemDirs = new TenantFileSystemDirs(configServerDB, tenantName);
        this.applicationRepo = applicationRepo;
        this.sessionPreparer = sessionPreparer;
        this.metrics = metrics;
        this.metricUpdater = metrics.getOrCreateMetricUpdater(Metrics.createDimensions(tenantName));
        this.secretStore = secretStore;
        this.hostProvisionerProvider = hostProvisionerProvider;
        this.configserverConfig = configserverConfig;
        this.configServerDB = configServerDB;
        this.zone = zone;
        this.modelFactoryRegistry = modelFactoryRegistry;
        this.configDefinitionRepo = configDefinitionRepo;
        this.tenantListener = tenantListener;
        this.loadSessions();
        this.directoryCache = curator.createDirectoryCache(this.sessionsPath.getAbsolute(), false, false, zkCacheExecutor);
        this.directoryCache.addListener(this::childEvent);
        this.directoryCache.start();
    }

    private void loadSessions() {
        ExecutorService executor = Executors.newFixedThreadPool(Math.max(8, Runtime.getRuntime().availableProcessors()), (ThreadFactory)new DaemonThreadFactory("load-sessions-"));
        this.loadLocalSessions(executor);
        this.loadRemoteSessions(executor);
        try {
            executor.shutdown();
            if (!executor.awaitTermination(1L, TimeUnit.MINUTES)) {
                log.log(Level.INFO, "Executor did not terminate");
            }
        }
        catch (InterruptedException e) {
            log.log(Level.WARNING, "Shutdown of executor for loading sessions failed: " + Exceptions.toMessageString((Throwable)e));
        }
    }

    public void addLocalSession(LocalSession session) {
        long sessionId = session.getSessionId();
        this.localSessionCache.put(sessionId, session);
        if (this.remoteSessionCache.get(sessionId) == null) {
            this.createRemoteSession(sessionId);
        }
    }

    public LocalSession getLocalSession(long sessionId) {
        return this.localSessionCache.get(sessionId);
    }

    public Collection<LocalSession> getLocalSessions() {
        return this.localSessionCache.values();
    }

    private void loadLocalSessions(ExecutorService executor) {
        File[] sessions = this.tenantFileSystemDirs.sessionsPath().listFiles(sessionApplicationsFilter);
        if (sessions == null) {
            return;
        }
        HashMap<Long, Future> futures = new HashMap<Long, Future>();
        for (File session : sessions) {
            long sessionId2 = Long.parseLong(session.getName());
            futures.put(sessionId2, executor.submit(() -> this.createSessionFromId(sessionId2)));
        }
        futures.forEach((sessionId, future) -> {
            try {
                future.get();
                log.log(Level.INFO, () -> "Local session " + sessionId + " loaded");
            }
            catch (InterruptedException | ExecutionException e) {
                log.log(Level.WARNING, "Could not load session " + sessionId, e);
            }
        });
    }

    public ConfigChangeActions prepareLocalSession(Session session, DeployLogger logger, PrepareParams params, Instant now) {
        this.applicationRepo.createApplication(params.getApplicationId());
        logger.log(Level.FINE, "Created application " + params.getApplicationId());
        long sessionId = session.getSessionId();
        SessionZooKeeperClient sessionZooKeeperClient = this.createSessionZooKeeperClient(sessionId);
        Curator.CompletionWaiter waiter = sessionZooKeeperClient.createPrepareWaiter();
        Optional<ApplicationSet> activeApplicationSet = this.getActiveApplicationSet(params.getApplicationId());
        ConfigChangeActions actions = this.sessionPreparer.prepare(this.applicationRepo.getHostValidator(), logger, params, activeApplicationSet, now, this.getSessionAppDir(sessionId), session.getApplicationPackage(), sessionZooKeeperClient).getConfigChangeActions();
        this.setPrepared(session);
        waiter.awaitCompletion(params.getTimeoutBudget().timeLeft());
        return actions;
    }

    public LocalSession createSessionFromExisting(Session existingSession, boolean internalRedeploy, TimeoutBudget timeoutBudget) {
        ApplicationId existingApplicationId = existingSession.getApplicationId();
        File existingApp = this.getSessionAppDir(existingSession.getSessionId());
        LocalSession session = this.createSessionFromApplication(existingApp, existingApplicationId, internalRedeploy, timeoutBudget);
        session.setApplicationId(existingApplicationId);
        session.setApplicationPackageReference(existingSession.getApplicationPackageReference());
        session.setVespaVersion(existingSession.getVespaVersion());
        session.setDockerImageRepository(existingSession.getDockerImageRepository());
        session.setAthenzDomain(existingSession.getAthenzDomain());
        return session;
    }

    public LocalSession createSessionFromApplicationPackage(File applicationDirectory, ApplicationId applicationId, TimeoutBudget timeoutBudget) {
        this.applicationRepo.createApplication(applicationId);
        return this.createSessionFromApplication(applicationDirectory, applicationId, false, timeoutBudget);
    }

    private void createLocalSession(File applicationFile, ApplicationId applicationId, long sessionId) {
        try {
            ApplicationPackage applicationPackage = this.createApplicationPackage(applicationFile, applicationId, sessionId, false);
            this.createLocalSession(sessionId, applicationPackage);
        }
        catch (Exception e) {
            throw new RuntimeException("Error creating session " + sessionId, e);
        }
    }

    public void deleteLocalSession(LocalSession session) {
        long sessionId = session.getSessionId();
        log.log(Level.FINE, () -> "Deleting local session " + sessionId);
        SessionStateWatcher watcher = this.sessionStateWatchers.remove(sessionId);
        if (watcher != null) {
            watcher.close();
        }
        this.localSessionCache.remove(sessionId);
        NestedTransaction transaction = new NestedTransaction();
        transaction.add((Transaction)FileTransaction.from(FileOperations.delete(this.getSessionAppDir(sessionId).getAbsolutePath())), new Class[0]);
        transaction.commit();
    }

    private void deleteAllSessions() {
        ArrayList<LocalSession> sessions = new ArrayList<LocalSession>(this.localSessionCache.values());
        for (LocalSession session : sessions) {
            this.deleteLocalSession(session);
        }
    }

    public RemoteSession getRemoteSession(long sessionId) {
        return this.remoteSessionCache.get(sessionId);
    }

    public List<Long> getRemoteSessionsFromZooKeeper() {
        return this.getSessionList(this.curator.getChildren(this.sessionsPath));
    }

    public RemoteSession createRemoteSession(long sessionId) {
        SessionZooKeeperClient sessionZKClient = this.createSessionZooKeeperClient(sessionId);
        RemoteSession session = new RemoteSession(this.tenantName, sessionId, sessionZKClient);
        RemoteSession newSession = this.loadSessionIfActive(session).orElse(session);
        this.remoteSessionCache.put(sessionId, newSession);
        this.updateSessionStateWatcher(sessionId, newSession);
        return newSession;
    }

    public int deleteExpiredRemoteSessions(Clock clock, Duration expiryTime) {
        int deleted = 0;
        for (long sessionId : this.getRemoteSessionsFromZooKeeper()) {
            Session session = this.remoteSessionCache.get(sessionId);
            if (session == null || session.getStatus() == Session.Status.ACTIVATE || !this.sessionHasExpired(session.getCreateTime(), expiryTime, clock)) continue;
            log.log(Level.FINE, () -> "Remote session " + sessionId + " for " + this.tenantName + " has expired, deleting it");
            this.deleteRemoteSessionFromZooKeeper(session);
            ++deleted;
        }
        return deleted;
    }

    public void deactivateAndUpdateCache(RemoteSession remoteSession) {
        RemoteSession session = remoteSession.deactivated();
        this.remoteSessionCache.put(session.getSessionId(), session);
    }

    public void deleteRemoteSessionFromZooKeeper(Session session) {
        SessionZooKeeperClient sessionZooKeeperClient = this.createSessionZooKeeperClient(session.getSessionId());
        CuratorTransaction transaction = sessionZooKeeperClient.deleteTransaction();
        transaction.commit();
        transaction.close();
    }

    private boolean sessionHasExpired(Instant created, Duration expiryTime, Clock clock) {
        return created.plus(expiryTime).isBefore(clock.instant());
    }

    private List<Long> getSessionListFromDirectoryCache(List<ChildData> children) {
        return this.getSessionList(children.stream().map(child -> Path.fromString((String)child.getPath()).getName()).collect(Collectors.toList()));
    }

    private List<Long> getSessionList(List<String> children) {
        return children.stream().map(Long::parseLong).collect(Collectors.toList());
    }

    private void loadRemoteSessions(ExecutorService executor) throws NumberFormatException {
        HashMap<Long, Future> futures = new HashMap<Long, Future>();
        for (long sessionId2 : this.getRemoteSessionsFromZooKeeper()) {
            futures.put(sessionId2, executor.submit(() -> this.sessionAdded(sessionId2)));
        }
        futures.forEach((sessionId, future) -> {
            try {
                future.get();
                log.log(Level.INFO, () -> "Remote session " + sessionId + " loaded");
            }
            catch (InterruptedException | ExecutionException e) {
                log.log(Level.WARNING, "Could not load session " + sessionId, e);
            }
        });
    }

    public void sessionAdded(long sessionId) {
        if (this.hasStatusDeleted(sessionId)) {
            return;
        }
        log.log(Level.FINE, () -> "Adding remote session " + sessionId);
        RemoteSession session = this.createRemoteSession(sessionId);
        if (session.getStatus() == Session.Status.NEW) {
            log.log(Level.FINE, () -> session.logPre() + "Confirming upload for session " + sessionId);
            this.confirmUpload(session);
        }
        this.createLocalSessionFromDistributedApplicationPackage(sessionId);
    }

    private boolean hasStatusDeleted(long sessionId) {
        SessionZooKeeperClient sessionZKClient = this.createSessionZooKeeperClient(sessionId);
        RemoteSession session = new RemoteSession(this.tenantName, sessionId, sessionZKClient);
        return session.getStatus() == Session.Status.DELETE;
    }

    void activate(RemoteSession session) {
        long sessionId = session.getSessionId();
        Curator.CompletionWaiter waiter = this.createSessionZooKeeperClient(sessionId).getActiveWaiter();
        log.log(Level.FINE, () -> session.logPre() + "Activating " + sessionId);
        this.applicationRepo.activateApplication(this.ensureApplicationLoaded(session), sessionId);
        log.log(Level.FINE, () -> session.logPre() + "Notifying " + waiter);
        this.notifyCompletion(waiter, session);
        log.log(Level.INFO, session.logPre() + "Session activated: " + sessionId);
    }

    public void delete(Session remoteSession) {
        long sessionId = remoteSession.getSessionId();
        log.log(Level.FINE, () -> remoteSession.logPre() + "Deactivating and deleting remote session " + sessionId);
        this.createSetStatusTransaction(remoteSession, Session.Status.DELETE).commit();
        this.deleteRemoteSessionFromZooKeeper(remoteSession);
        this.remoteSessionCache.remove(sessionId);
        LocalSession localSession = this.getLocalSession(sessionId);
        if (localSession != null) {
            log.log(Level.FINE, () -> localSession.logPre() + "Deleting local session " + sessionId);
            this.deleteLocalSession(localSession);
        }
    }

    private Optional<RemoteSession> loadSessionIfActive(RemoteSession session) {
        for (ApplicationId applicationId : this.applicationRepo.activeApplications()) {
            if (this.applicationRepo.requireActiveSessionOf(applicationId) != session.getSessionId()) continue;
            log.log(Level.FINE, () -> "Found active application for session " + session.getSessionId() + " , loading it");
            this.applicationRepo.activateApplication(this.ensureApplicationLoaded(session), session.getSessionId());
            log.log(Level.INFO, session.logPre() + "Application activated successfully: " + applicationId + " (generation " + session.getSessionId() + ")");
            return Optional.ofNullable(this.remoteSessionCache.get(session.getSessionId()));
        }
        return Optional.empty();
    }

    void prepareRemoteSession(RemoteSession session) {
        SessionZooKeeperClient sessionZooKeeperClient = this.createSessionZooKeeperClient(session.getSessionId());
        Curator.CompletionWaiter waiter = sessionZooKeeperClient.getPrepareWaiter();
        this.ensureApplicationLoaded(session);
        this.notifyCompletion(waiter, session);
    }

    public ApplicationSet ensureApplicationLoaded(RemoteSession session) {
        if (session.applicationSet().isPresent()) {
            return session.applicationSet().get();
        }
        Optional<Long> activeSessionId = this.getActiveSessionId(session.getApplicationId());
        Optional<ApplicationSet> previousApplicationSet = activeSessionId.filter(session::isNewerThan).flatMap(this::getApplicationSet);
        ApplicationSet applicationSet = this.loadApplication(session, previousApplicationSet);
        RemoteSession activated = session.activated(applicationSet);
        long sessionId = activated.getSessionId();
        this.remoteSessionCache.put(sessionId, activated);
        this.updateSessionStateWatcher(sessionId, activated);
        return applicationSet;
    }

    void confirmUpload(Session session) {
        Curator.CompletionWaiter waiter = session.getSessionZooKeeperClient().getUploadWaiter();
        long sessionId = session.getSessionId();
        log.log(Level.FINE, "Notifying upload waiter for session " + sessionId);
        this.notifyCompletion(waiter, session);
        log.log(Level.FINE, "Done notifying upload for session " + sessionId);
    }

    void notifyCompletion(Curator.CompletionWaiter completionWaiter, Session session) {
        try {
            completionWaiter.notifyCompletion();
        }
        catch (RuntimeException e) {
            Set<Class<KeeperException.NodeExistsException>> acceptedExceptions = Set.of(KeeperException.NoNodeException.class, KeeperException.NodeExistsException.class);
            Class<?> exceptionClass = e.getCause().getClass();
            if (acceptedExceptions.contains(exceptionClass)) {
                log.log(Level.FINE, "Not able to notify completion for session " + session.getSessionId() + " (" + completionWaiter + "), node " + (exceptionClass.equals(KeeperException.NoNodeException.class) ? "has been deleted" : "already exists"));
            }
            throw e;
        }
    }

    private ApplicationSet loadApplication(Session session, Optional<ApplicationSet> previousApplicationSet) {
        log.log(Level.FINE, () -> "Loading application for " + session);
        SessionZooKeeperClient sessionZooKeeperClient = this.createSessionZooKeeperClient(session.getSessionId());
        ApplicationPackage applicationPackage = sessionZooKeeperClient.loadApplicationPackage();
        ActivatedModelsBuilder builder = new ActivatedModelsBuilder(session.getTenantName(), session.getSessionId(), sessionZooKeeperClient, previousApplicationSet, this.curator, this.metrics, this.permanentApplicationPackage, this.flagSource, this.secretStore, this.hostProvisionerProvider, this.configserverConfig, this.zone, this.modelFactoryRegistry, this.configDefinitionRepo, this.tenantListener);
        SettableOptional allocatedHosts = new SettableOptional(applicationPackage.getAllocatedHosts());
        return ApplicationSet.fromList(builder.buildModels(session.getApplicationId(), sessionZooKeeperClient.readDockerImageRepository(), sessionZooKeeperClient.readVespaVersion(), applicationPackage, (SettableOptional<AllocatedHosts>)allocatedHosts, this.clock.instant()));
    }

    private void nodeChanged() {
        this.zkWatcherExecutor.execute(() -> {
            HashMultiset sessionMetrics = HashMultiset.create();
            for (Session session : this.remoteSessionCache.values()) {
                sessionMetrics.add((Object)session.getStatus());
            }
            this.metricUpdater.setNewSessions(sessionMetrics.count((Object)Session.Status.NEW));
            this.metricUpdater.setPreparedSessions(sessionMetrics.count((Object)Session.Status.PREPARE));
            this.metricUpdater.setActivatedSessions(sessionMetrics.count((Object)Session.Status.ACTIVATE));
            this.metricUpdater.setDeactivatedSessions(sessionMetrics.count((Object)Session.Status.DEACTIVATE));
        });
    }

    private void childEvent(CuratorFramework ignored, PathChildrenCacheEvent event) {
        this.zkWatcherExecutor.execute(() -> {
            log.log(Level.FINE, () -> "Got child event: " + event);
            switch (event.getType()) {
                case CHILD_ADDED: 
                case CHILD_REMOVED: 
                case CONNECTION_RECONNECTED: {
                    this.sessionsChanged();
                    break;
                }
            }
        });
    }

    public void deleteExpiredSessions(Map<ApplicationId, Long> activeSessions) {
        log.log(Level.FINE, () -> "Purging old sessions for tenant '" + this.tenantName + "'");
        HashSet<LocalSession> toDelete = new HashSet<LocalSession>();
        try {
            for (LocalSession candidate : List.copyOf(this.localSessionCache.values())) {
                Long activeSession;
                Optional<ApplicationId> applicationId;
                Instant createTime = candidate.getCreateTime();
                log.log(Level.FINE, () -> "Candidate session for deletion: " + candidate.getSessionId() + ", created: " + createTime);
                if (this.hasExpired(candidate) && !this.isActiveSession(candidate)) {
                    toDelete.add(candidate);
                    continue;
                }
                if (!createTime.plus(Duration.ofDays(1L)).isBefore(this.clock.instant()) || (applicationId = candidate.getOptionalApplicationId()).isEmpty() || (activeSession = activeSessions.get(applicationId.get())) != null && activeSession.longValue() == candidate.getSessionId()) continue;
                toDelete.add(candidate);
                log.log(Level.INFO, "Deleted inactive session " + candidate.getSessionId() + " created " + createTime + " for '" + applicationId + "'");
            }
            toDelete.forEach(this::deleteLocalSession);
        }
        catch (Throwable e) {
            log.log(Level.WARNING, "Error when purging old sessions ", e);
        }
        log.log(Level.FINE, () -> "Done purging old sessions");
    }

    private boolean hasExpired(LocalSession candidate) {
        return candidate.getCreateTime().plus(this.sessionLifetime).isBefore(this.clock.instant());
    }

    private boolean isActiveSession(LocalSession candidate) {
        return candidate.getStatus() == Session.Status.ACTIVATE;
    }

    private void ensureSessionPathDoesNotExist(long sessionId) {
        Path sessionPath = this.getSessionPath(sessionId);
        if (this.configCurator.exists(sessionPath.getAbsolute())) {
            throw new IllegalArgumentException("Path " + sessionPath.getAbsolute() + " already exists in ZooKeeper");
        }
    }

    private ApplicationPackage createApplication(File userDir, File configApplicationDir, ApplicationId applicationId, long sessionId, Optional<Long> currentlyActiveSessionId, boolean internalRedeploy) {
        long deployTimestamp = System.currentTimeMillis();
        String user = System.getenv("USER");
        if (user == null) {
            user = "unknown";
        }
        DeployData deployData = new DeployData(user, userDir.getAbsolutePath(), applicationId, Long.valueOf(deployTimestamp), internalRedeploy, Long.valueOf(sessionId), currentlyActiveSessionId.orElse(0L).longValue());
        return FilesApplicationPackage.fromFileWithDeployData((File)configApplicationDir, (DeployData)deployData);
    }

    private LocalSession createSessionFromApplication(File applicationFile, ApplicationId applicationId, boolean internalRedeploy, TimeoutBudget timeoutBudget) {
        long sessionId = this.getNextSessionId();
        try {
            this.ensureSessionPathDoesNotExist(sessionId);
            ApplicationPackage app = this.createApplicationPackage(applicationFile, applicationId, sessionId, internalRedeploy);
            log.log(Level.FINE, () -> TenantRepository.logPre(this.tenantName) + "Creating session " + sessionId + " in ZooKeeper");
            SessionZooKeeperClient sessionZKClient = this.createSessionZooKeeperClient(sessionId);
            sessionZKClient.createNewSession(this.clock.instant());
            Curator.CompletionWaiter waiter = sessionZKClient.getUploadWaiter();
            LocalSession session = new LocalSession(this.tenantName, sessionId, app, sessionZKClient);
            waiter.awaitCompletion(timeoutBudget.timeLeft());
            this.addLocalSession(session);
            return session;
        }
        catch (Exception e) {
            throw new RuntimeException("Error creating session " + sessionId, e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private ApplicationPackage createApplicationPackage(File applicationFile, ApplicationId applicationId, long sessionId, boolean internalRedeploy) throws IOException {
        Object object = this.monitor;
        synchronized (object) {
            Optional<Long> activeSessionId = this.getActiveSessionId(applicationId);
            File userApplicationDir = this.getSessionAppDir(sessionId);
            this.copyApp(applicationFile, userApplicationDir);
            ApplicationPackage applicationPackage = this.createApplication(applicationFile, userApplicationDir, applicationId, sessionId, activeSessionId, internalRedeploy);
            applicationPackage.writeMetaData();
            return applicationPackage;
        }
    }

    public Optional<ApplicationSet> getActiveApplicationSet(ApplicationId appId) {
        return this.applicationRepo.activeSessionOf(appId).flatMap(this::getApplicationSet);
    }

    private Optional<ApplicationSet> getApplicationSet(long sessionId) {
        Optional<ApplicationSet> applicationSet = Optional.empty();
        try {
            RemoteSession session = this.getRemoteSession(sessionId);
            applicationSet = Optional.ofNullable(this.ensureApplicationLoaded(session));
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
        return applicationSet;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void copyApp(File sourceDir, File destinationDir) throws IOException {
        if (destinationDir.exists()) {
            log.log(Level.INFO, "Destination dir " + destinationDir + " already exists, app has already been copied");
            return;
        }
        if (!sourceDir.isDirectory()) {
            throw new IllegalArgumentException(sourceDir.getAbsolutePath() + " is not a directory");
        }
        java.nio.file.Path tempDestinationDir = null;
        try {
            tempDestinationDir = Files.createTempDirectory(destinationDir.getParentFile().toPath(), "app-package", new FileAttribute[0]);
            log.log(Level.FINE, "Copying dir " + sourceDir.getAbsolutePath() + " to " + tempDestinationDir.toFile().getAbsolutePath());
            IOUtils.copyDirectory((File)sourceDir, (File)tempDestinationDir.toFile());
            log.log(Level.FINE, "Moving " + tempDestinationDir + " to " + destinationDir.getAbsolutePath());
            Files.move(tempDestinationDir, destinationDir.toPath(), StandardCopyOption.ATOMIC_MOVE);
        }
        finally {
            if (tempDestinationDir != null) {
                IOUtils.recursiveDeleteDir((File)tempDestinationDir.toFile());
            }
        }
    }

    void createSessionFromId(long sessionId) {
        File sessionDir = this.getAndValidateExistingSessionAppDir(sessionId);
        FilesApplicationPackage applicationPackage = FilesApplicationPackage.fromFile((File)sessionDir);
        this.createLocalSession(sessionId, (ApplicationPackage)applicationPackage);
    }

    void createLocalSession(long sessionId, ApplicationPackage applicationPackage) {
        SessionZooKeeperClient sessionZKClient = this.createSessionZooKeeperClient(sessionId);
        LocalSession session = new LocalSession(this.tenantName, sessionId, applicationPackage, sessionZKClient);
        this.addLocalSession(session);
    }

    public void createLocalSessionFromDistributedApplicationPackage(long sessionId) {
        if (this.applicationRepo.sessionExistsInFileSystem(sessionId)) {
            log.log(Level.FINE, () -> "Local session for session id " + sessionId + " already exists");
            this.createSessionFromId(sessionId);
            return;
        }
        SessionZooKeeperClient sessionZKClient = this.createSessionZooKeeperClient(sessionId);
        FileReference fileReference = sessionZKClient.readApplicationPackageReference();
        log.log(Level.FINE, () -> "File reference for session id " + sessionId + ": " + fileReference);
        if (fileReference != null) {
            File sessionDir;
            File rootDir = new File(Defaults.getDefaults().underVespaHome(this.configserverConfig.fileReferencesDir()));
            FileDirectory fileDirectory = new FileDirectory(rootDir);
            try {
                sessionDir = fileDirectory.getFile(fileReference);
            }
            catch (IllegalArgumentException e) {
                log.log(Level.FINE, "File reference for session id " + sessionId + ": " + fileReference + " not found in " + fileDirectory);
                return;
            }
            ApplicationId applicationId = sessionZKClient.readApplicationId().orElseThrow(() -> new RuntimeException("Could not find application id for session " + sessionId));
            log.log(Level.FINE, () -> "Creating local session for tenant '" + this.tenantName + "' with session id " + sessionId);
            this.createLocalSession(sessionDir, applicationId, sessionId);
        }
    }

    private Optional<Long> getActiveSessionId(ApplicationId applicationId) {
        List<ApplicationId> applicationIds = this.applicationRepo.activeApplications();
        return applicationIds.contains(applicationId) ? Optional.of(this.applicationRepo.requireActiveSessionOf(applicationId)) : Optional.empty();
    }

    private long getNextSessionId() {
        return this.sessionCounter.nextSessionId();
    }

    public Path getSessionPath(long sessionId) {
        return this.sessionsPath.append(String.valueOf(sessionId));
    }

    Path getSessionStatePath(long sessionId) {
        return this.getSessionPath(sessionId).append("/sessionState");
    }

    private SessionZooKeeperClient createSessionZooKeeperClient(long sessionId) {
        String serverId = this.configserverConfig.serverId();
        return new SessionZooKeeperClient(this.curator, this.configCurator, this.tenantName, sessionId, serverId);
    }

    private File getAndValidateExistingSessionAppDir(long sessionId) {
        File appDir = this.getSessionAppDir(sessionId);
        if (!appDir.exists() || !appDir.isDirectory()) {
            throw new IllegalArgumentException("Unable to find correct application directory for session " + sessionId);
        }
        return appDir;
    }

    private File getSessionAppDir(long sessionId) {
        return new TenantFileSystemDirs(this.configServerDB, this.tenantName).getUserApplicationDir(sessionId);
    }

    private void updateSessionStateWatcher(long sessionId, RemoteSession remoteSession) {
        SessionStateWatcher sessionStateWatcher = this.sessionStateWatchers.get(sessionId);
        if (sessionStateWatcher == null) {
            Curator.FileCache fileCache = this.curator.createFileCache(this.getSessionStatePath(sessionId).getAbsolute(), false);
            fileCache.addListener(this::nodeChanged);
            this.sessionStateWatchers.put(sessionId, new SessionStateWatcher(fileCache, remoteSession, this.metricUpdater, this.zkWatcherExecutor, this));
        } else {
            sessionStateWatcher.updateRemoteSession(remoteSession);
        }
    }

    public String toString() {
        return this.getLocalSessions().toString();
    }

    public Clock clock() {
        return this.clock;
    }

    public void close() {
        this.deleteAllSessions();
        this.tenantFileSystemDirs.delete();
        try {
            if (this.directoryCache != null) {
                this.directoryCache.close();
            }
        }
        catch (Exception e) {
            log.log(Level.WARNING, "Exception when closing path cache", e);
        }
        finally {
            this.checkForRemovedSessions(new ArrayList<Long>());
        }
    }

    private void sessionsChanged() throws NumberFormatException {
        List<Long> sessions = this.getSessionListFromDirectoryCache(this.directoryCache.getCurrentData());
        this.checkForRemovedSessions(sessions);
        this.checkForAddedSessions(sessions);
    }

    private void checkForRemovedSessions(List<Long> existingSessions) {
        Iterator<RemoteSession> it = this.remoteSessionCache.values().iterator();
        while (it.hasNext()) {
            long sessionId = it.next().sessionId;
            if (existingSessions.contains(sessionId)) continue;
            SessionStateWatcher watcher = this.sessionStateWatchers.remove(sessionId);
            if (watcher != null) {
                watcher.close();
            }
            it.remove();
            this.metricUpdater.incRemovedSessions();
        }
    }

    private void checkForAddedSessions(List<Long> sessions) {
        for (Long sessionId : sessions) {
            if (this.remoteSessionCache.get(sessionId) != null) continue;
            this.sessionAdded(sessionId);
        }
    }

    public Transaction createActivateTransaction(Session session) {
        Transaction transaction = this.createSetStatusTransaction(session, Session.Status.ACTIVATE);
        transaction.add(this.applicationRepo.createPutTransaction(session.getApplicationId(), session.getSessionId()).operations());
        return transaction;
    }

    public Transaction createSetStatusTransaction(Session session, Session.Status status) {
        return session.sessionZooKeeperClient.createWriteStatusTransaction(status);
    }

    void setPrepared(Session session) {
        session.setStatus(Session.Status.PREPARE);
    }

    private static class DeleteOperation
    implements FileOperation {
        private final String pathToDelete;

        DeleteOperation(String pathToDelete) {
            this.pathToDelete = pathToDelete;
        }

        @Override
        public void commit() {
            IOUtils.recursiveDeleteDir((File)new File(this.pathToDelete));
        }
    }

    private static interface FileOperation
    extends Transaction.Operation {
        public void commit();
    }

    private static class FileOperations {
        private FileOperations() {
        }

        public static DeleteOperation delete(String pathToDelete) {
            return new DeleteOperation(pathToDelete);
        }
    }

    private static class FileTransaction
    extends AbstractTransaction {
        private FileTransaction() {
        }

        public static FileTransaction from(FileOperation operation) {
            FileTransaction transaction = new FileTransaction();
            transaction.add(operation);
            return transaction;
        }

        public void prepare() {
        }

        public void commit() {
            for (Transaction.Operation operation : this.operations()) {
                ((FileOperation)operation).commit();
            }
        }
    }
}

