/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.vespa.config.server.tenant;

import com.google.common.collect.ImmutableSet;
import com.google.inject.Inject;
import com.yahoo.cloud.config.ConfigserverConfig;
import com.yahoo.concurrent.StripedExecutor;
import com.yahoo.config.provision.ApplicationId;
import com.yahoo.config.provision.TenantName;
import com.yahoo.path.Path;
import com.yahoo.vespa.config.server.GlobalComponentRegistry;
import com.yahoo.vespa.config.server.monitoring.MetricUpdater;
import com.yahoo.vespa.config.server.tenant.Tenant;
import com.yahoo.vespa.config.server.tenant.TenantBuilder;
import com.yahoo.vespa.config.server.tenant.TenantListener;
import com.yahoo.vespa.curator.Curator;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.recipes.cache.PathChildrenCacheEvent;
import org.apache.curator.framework.state.ConnectionState;
import org.apache.zookeeper.KeeperException;

public class TenantRepository {
    public static final TenantName HOSTED_VESPA_TENANT = TenantName.from((String)"hosted-vespa");
    private static final TenantName DEFAULT_TENANT = TenantName.defaultName();
    private static final Path tenantsPath = Path.fromString((String)"/config/v2/tenants/");
    private static final Path locksPath = Path.fromString((String)"/config/v2/locks/");
    private static final Path vespaPath = Path.fromString((String)"/vespa");
    private static final Duration checkForRemovedApplicationsInterval = Duration.ofMinutes(1L);
    private static final Logger log = Logger.getLogger(TenantRepository.class.getName());
    private final Map<TenantName, Tenant> tenants = Collections.synchronizedMap(new LinkedHashMap());
    private final GlobalComponentRegistry globalComponentRegistry;
    private final List<TenantListener> tenantListeners = Collections.synchronizedList(new ArrayList());
    private final Curator curator;
    private final MetricUpdater metricUpdater;
    private final ExecutorService zkCacheExecutor;
    private final StripedExecutor<TenantName> zkWatcherExecutor;
    private final ExecutorService bootstrapExecutor;
    private final ScheduledExecutorService checkForRemovedApplicationsService = new ScheduledThreadPoolExecutor(1);
    private final Optional<Curator.DirectoryCache> directoryCache;
    private final boolean throwExceptionIfBootstrappingFails;

    @Inject
    public TenantRepository(GlobalComponentRegistry globalComponentRegistry) {
        this(globalComponentRegistry, true);
    }

    public TenantRepository(GlobalComponentRegistry globalComponentRegistry, boolean useZooKeeperWatchForTenantChanges) {
        this.globalComponentRegistry = globalComponentRegistry;
        ConfigserverConfig configserverConfig = globalComponentRegistry.getConfigserverConfig();
        this.bootstrapExecutor = Executors.newFixedThreadPool(configserverConfig.numParallelTenantLoaders());
        this.throwExceptionIfBootstrappingFails = configserverConfig.throwIfBootstrappingTenantRepoFails();
        this.curator = globalComponentRegistry.getCurator();
        this.metricUpdater = globalComponentRegistry.getMetrics().getOrCreateMetricUpdater(Collections.emptyMap());
        this.tenantListeners.add(globalComponentRegistry.getTenantListener());
        this.zkCacheExecutor = globalComponentRegistry.getZkCacheExecutor();
        this.zkWatcherExecutor = globalComponentRegistry.getZkWatcherExecutor();
        this.curator.framework().getConnectionStateListenable().addListener(this::stateChanged);
        this.curator.create(tenantsPath);
        this.curator.create(locksPath);
        this.createSystemTenants(configserverConfig);
        this.curator.create(vespaPath);
        if (useZooKeeperWatchForTenantChanges) {
            this.directoryCache = Optional.of(this.curator.createDirectoryCache(tenantsPath.getAbsolute(), false, false, this.zkCacheExecutor));
            this.directoryCache.get().start();
            this.directoryCache.get().addListener(this::childEvent);
        } else {
            this.directoryCache = Optional.empty();
        }
        log.log(Level.FINE, "Creating all tenants");
        this.bootstrapTenants();
        this.notifyTenantsLoaded();
        log.log(Level.FINE, "All tenants created");
        this.checkForRemovedApplicationsService.scheduleWithFixedDelay(this::removeUnusedApplications, checkForRemovedApplicationsInterval.getSeconds(), checkForRemovedApplicationsInterval.getSeconds(), TimeUnit.SECONDS);
    }

    private void notifyTenantsLoaded() {
        for (TenantListener tenantListener : this.tenantListeners) {
            tenantListener.onTenantsLoaded();
        }
    }

    public synchronized void addTenant(TenantName tenantName) {
        this.addTenant(TenantBuilder.create(this.globalComponentRegistry, tenantName));
    }

    public synchronized void addTenant(TenantBuilder builder) {
        this.writeTenantPath(builder.getTenantName());
        this.createTenant(builder);
    }

    private static Set<TenantName> readTenantsFromZooKeeper(Curator curator) {
        return curator.getChildren(tenantsPath).stream().map(TenantName::from).collect(Collectors.toSet());
    }

    public synchronized void updateTenants() {
        Set<TenantName> allTenants = TenantRepository.readTenantsFromZooKeeper(this.curator);
        log.log(Level.FINE, "Create tenants, tenants found in zookeeper: " + allTenants);
        for (TenantName tenantName : Set.copyOf(this.tenants.keySet())) {
            if (allTenants.contains(tenantName)) continue;
            this.zkWatcherExecutor.execute((Object)tenantName, () -> this.closeTenant(tenantName));
        }
        for (TenantName tenantName : allTenants) {
            if (this.tenants.containsKey(tenantName)) continue;
            this.zkWatcherExecutor.execute((Object)tenantName, () -> this.createTenant(tenantName));
        }
        this.metricUpdater.setTenants(this.tenants.size());
    }

    private void bootstrapTenants() {
        HashMap futures = new HashMap();
        TenantRepository.readTenantsFromZooKeeper(this.curator).forEach(t -> futures.put(t, this.bootstrapExecutor.submit(() -> this.createTenant((TenantName)t))));
        HashSet<TenantName> failed = new HashSet<TenantName>();
        for (Map.Entry f : futures.entrySet()) {
            TenantName tenantName = (TenantName)f.getKey();
            try {
                ((Future)f.getValue()).get();
            }
            catch (ExecutionException e) {
                log.log(Level.WARNING, "Failed to create tenant " + tenantName, e);
                failed.add(tenantName);
            }
            catch (InterruptedException e) {
                log.log(Level.WARNING, "Interrupted while creating tenant '" + tenantName + "'", e);
            }
        }
        if (failed.size() > 0 && this.throwExceptionIfBootstrappingFails) {
            throw new RuntimeException("Could not create all tenants when bootstrapping, failed to create: " + failed);
        }
        this.metricUpdater.setTenants(this.tenants.size());
        this.bootstrapExecutor.shutdown();
        try {
            this.bootstrapExecutor.awaitTermination(365L, TimeUnit.DAYS);
        }
        catch (InterruptedException e) {
            throw new RuntimeException("Executor for creating tenants did not terminate within timeout");
        }
    }

    private void createTenant(TenantName tenantName) {
        this.createTenant(TenantBuilder.create(this.globalComponentRegistry, tenantName));
    }

    protected void createTenant(TenantBuilder builder) {
        TenantName tenantName = builder.getTenantName();
        if (this.tenants.containsKey(tenantName)) {
            return;
        }
        log.log(Level.INFO, "Creating tenant '" + tenantName + "'");
        Tenant tenant = builder.build();
        this.notifyNewTenant(tenant);
        this.tenants.putIfAbsent(tenantName, tenant);
    }

    public synchronized Tenant defaultTenant() {
        return this.tenants.get(DEFAULT_TENANT);
    }

    private void removeUnusedApplications() {
        this.getAllTenants().forEach(tenant -> tenant.getApplicationRepo().removeUnusedApplications());
    }

    private void notifyNewTenant(Tenant tenant) {
        for (TenantListener listener : this.tenantListeners) {
            listener.onTenantCreate(tenant.getName(), tenant);
        }
    }

    private void notifyRemovedTenant(TenantName name) {
        for (TenantListener listener : this.tenantListeners) {
            listener.onTenantDelete(name);
        }
    }

    private synchronized void createSystemTenants(ConfigserverConfig configserverConfig) {
        ArrayList<TenantName> systemTenants = new ArrayList<TenantName>();
        systemTenants.add(DEFAULT_TENANT);
        if (configserverConfig.hostedVespa()) {
            systemTenants.add(HOSTED_VESPA_TENANT);
        }
        for (TenantName tenantName : systemTenants) {
            try {
                this.writeTenantPath(tenantName);
            }
            catch (RuntimeException e) {
                if (e.getCause().getClass() == KeeperException.NodeExistsException.class) continue;
                throw e;
            }
        }
    }

    private synchronized void writeTenantPath(TenantName name) {
        this.curator.createAtomically(new Path[]{TenantRepository.getTenantPath(name), TenantRepository.getSessionsPath(name), TenantRepository.getApplicationsPath(name), TenantRepository.getLocksPath(name)});
    }

    public synchronized void deleteTenant(TenantName name) {
        if (name.equals((Object)DEFAULT_TENANT)) {
            throw new IllegalArgumentException("Deleting 'default' tenant is not allowed");
        }
        if (!this.tenants.containsKey(name)) {
            throw new IllegalArgumentException("Deleting '" + name + "' failed, tenant does not exist");
        }
        log.log(Level.INFO, "Deleting tenant '" + name + "'");
        this.tenants.get(name).delete();
    }

    private synchronized void closeTenant(TenantName name) {
        Tenant tenant = this.tenants.remove(name);
        if (tenant == null) {
            throw new IllegalArgumentException("Closing '" + name + "' failed, tenant does not exist");
        }
        log.log(Level.INFO, "Closing tenant '" + name + "'");
        this.notifyRemovedTenant(name);
        tenant.close();
    }

    String tenantZkPath(TenantName tenant) {
        return TenantRepository.getTenantPath(tenant).getAbsolute();
    }

    public static String logPre(ApplicationId app) {
        if (DEFAULT_TENANT.equals((Object)app.tenant())) {
            return "";
        }
        StringBuilder ret = new StringBuilder().append(TenantRepository.logPre(app.tenant())).append("app:" + app.application().value()).append(":" + app.instance().value()).append(" ");
        return ret.toString();
    }

    public static String logPre(TenantName tenant) {
        if (DEFAULT_TENANT.equals((Object)tenant)) {
            return "";
        }
        StringBuilder ret = new StringBuilder().append("tenant:" + tenant.value()).append(" ");
        return ret.toString();
    }

    private void stateChanged(CuratorFramework framework, ConnectionState connectionState) {
        switch (connectionState) {
            case CONNECTED: {
                this.metricUpdater.incZKConnected();
                break;
            }
            case SUSPENDED: {
                this.metricUpdater.incZKSuspended();
                break;
            }
            case RECONNECTED: {
                this.metricUpdater.incZKReconnected();
                break;
            }
            case LOST: {
                this.metricUpdater.incZKConnectionLost();
                break;
            }
        }
    }

    private void childEvent(CuratorFramework framework, PathChildrenCacheEvent event) {
        switch (event.getType()) {
            case CHILD_ADDED: 
            case CHILD_REMOVED: {
                this.updateTenants();
            }
        }
    }

    public void close() {
        this.directoryCache.ifPresent(Curator.DirectoryCache::close);
        try {
            this.zkCacheExecutor.shutdown();
            this.checkForRemovedApplicationsService.shutdown();
            this.zkWatcherExecutor.shutdownAndWait();
            this.zkCacheExecutor.awaitTermination(50L, TimeUnit.SECONDS);
            this.checkForRemovedApplicationsService.awaitTermination(50L, TimeUnit.SECONDS);
        }
        catch (InterruptedException e) {
            log.log(Level.WARNING, "Interrupted while shutting down.", e);
            Thread.currentThread().interrupt();
        }
    }

    public boolean checkThatTenantExists(TenantName tenant) {
        return this.tenants.containsKey(tenant);
    }

    public Tenant getTenant(TenantName tenantName) {
        return this.tenants.get(tenantName);
    }

    public Set<TenantName> getAllTenantNames() {
        return ImmutableSet.copyOf(this.tenants.keySet());
    }

    public Collection<Tenant> getAllTenants() {
        return ImmutableSet.copyOf(this.tenants.values());
    }

    public static Path getTenantPath(TenantName tenantName) {
        return tenantsPath.append(tenantName.value());
    }

    public static Path getSessionsPath(TenantName tenantName) {
        return TenantRepository.getTenantPath(tenantName).append("sessions");
    }

    public static Path getApplicationsPath(TenantName tenantName) {
        return TenantRepository.getTenantPath(tenantName).append("applications");
    }

    public static Path getLocksPath(TenantName tenantName) {
        return locksPath.append(tenantName.value());
    }
}

