/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.vespa.config.server.session;

import com.google.common.collect.HashMultiset;
import com.yahoo.concurrent.StripedExecutor;
import com.yahoo.config.provision.ApplicationId;
import com.yahoo.config.provision.TenantName;
import com.yahoo.path.Path;
import com.yahoo.vespa.config.server.GlobalComponentRegistry;
import com.yahoo.vespa.config.server.ReloadHandler;
import com.yahoo.vespa.config.server.application.TenantApplications;
import com.yahoo.vespa.config.server.monitoring.MetricUpdater;
import com.yahoo.vespa.config.server.monitoring.Metrics;
import com.yahoo.vespa.config.server.session.RemoteSession;
import com.yahoo.vespa.config.server.session.RemoteSessionFactory;
import com.yahoo.vespa.config.server.session.RemoteSessionStateWatcher;
import com.yahoo.vespa.config.server.session.Session;
import com.yahoo.vespa.config.server.session.SessionRepo;
import com.yahoo.vespa.config.server.tenant.TenantRepository;
import com.yahoo.vespa.curator.Curator;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executor;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.recipes.cache.ChildData;
import org.apache.curator.framework.recipes.cache.PathChildrenCacheEvent;

public class RemoteSessionRepo
extends SessionRepo<RemoteSession> {
    private static final Logger log = Logger.getLogger(RemoteSessionRepo.class.getName());
    private final GlobalComponentRegistry componentRegistry;
    private final Curator curator;
    private final Path sessionsPath;
    private final RemoteSessionFactory remoteSessionFactory;
    private final Map<Long, RemoteSessionStateWatcher> sessionStateWatchers = new HashMap<Long, RemoteSessionStateWatcher>();
    private final ReloadHandler reloadHandler;
    private final TenantName tenantName;
    private final MetricUpdater metrics;
    private final Curator.DirectoryCache directoryCache;
    private final TenantApplications applicationRepo;
    private final Executor zkWatcherExecutor;

    public RemoteSessionRepo(GlobalComponentRegistry componentRegistry, RemoteSessionFactory remoteSessionFactory, ReloadHandler reloadHandler, TenantName tenantName, TenantApplications applicationRepo) {
        this.componentRegistry = componentRegistry;
        this.curator = componentRegistry.getCurator();
        this.sessionsPath = TenantRepository.getSessionsPath(tenantName);
        this.applicationRepo = applicationRepo;
        this.remoteSessionFactory = remoteSessionFactory;
        this.reloadHandler = reloadHandler;
        this.tenantName = tenantName;
        this.metrics = componentRegistry.getMetrics().getOrCreateMetricUpdater(Metrics.createDimensions(tenantName));
        StripedExecutor<TenantName> zkWatcherExecutor = componentRegistry.getZkWatcherExecutor();
        this.zkWatcherExecutor = command -> zkWatcherExecutor.execute((Object)tenantName, command);
        this.initializeSessions();
        this.directoryCache = this.curator.createDirectoryCache(this.sessionsPath.getAbsolute(), false, false, componentRegistry.getZkCacheExecutor());
        this.directoryCache.addListener(this::childEvent);
        this.directoryCache.start();
    }

    public List<Long> getSessions() {
        return this.getSessionList(this.curator.getChildren(this.sessionsPath));
    }

    public int deleteExpiredSessions(Duration expiryTime) {
        int deleted = 0;
        for (long sessionId : this.getSessions()) {
            Instant created;
            RemoteSession session = (RemoteSession)this.getSession(sessionId);
            if (session == null || !this.sessionHasExpired(created = Instant.ofEpochSecond(session.getCreateTime()), expiryTime)) continue;
            log.log(Level.INFO, "Remote session " + sessionId + " for " + this.tenantName + " has expired, deleting it");
            session.delete();
            ++deleted;
        }
        return deleted;
    }

    private boolean sessionHasExpired(Instant created, Duration expiryTime) {
        return created.plus(expiryTime).isBefore(Instant.now());
    }

    private List<Long> getSessionListFromDirectoryCache(List<ChildData> children) {
        return this.getSessionList(children.stream().map(child -> Path.fromString((String)child.getPath()).getName()).collect(Collectors.toList()));
    }

    private List<Long> getSessionList(List<String> children) {
        return children.stream().map(Long::parseLong).collect(Collectors.toList());
    }

    private void initializeSessions() throws NumberFormatException {
        this.getSessions().forEach(this::sessionAdded);
    }

    private synchronized void sessionsChanged() throws NumberFormatException {
        List<Long> sessions = this.getSessionListFromDirectoryCache(this.directoryCache.getCurrentData());
        this.checkForRemovedSessions(sessions);
        this.checkForAddedSessions(sessions);
    }

    private void checkForRemovedSessions(List<Long> sessions) {
        for (RemoteSession session : this.listSessions()) {
            if (sessions.contains(session.getSessionId())) continue;
            this.sessionRemoved(session.getSessionId());
        }
    }

    private void checkForAddedSessions(List<Long> sessions) {
        for (Long sessionId : sessions) {
            if (this.getSession(sessionId) != null) continue;
            this.sessionAdded(sessionId);
        }
    }

    private void sessionAdded(long sessionId) {
        log.log(Level.FINE, () -> "Adding session to RemoteSessionRepo: " + sessionId);
        try {
            RemoteSession session = this.remoteSessionFactory.createSession(sessionId);
            Path sessionPath = this.sessionsPath.append(String.valueOf(sessionId));
            Curator.FileCache fileCache = this.curator.createFileCache(sessionPath.append("/sessionState").getAbsolute(), false);
            fileCache.addListener(this::nodeChanged);
            this.loadSessionIfActive(session);
            this.addSession(session);
            this.metrics.incAddedSessions();
            this.sessionStateWatchers.put(sessionId, new RemoteSessionStateWatcher(fileCache, this.reloadHandler, session, this.metrics, this.zkWatcherExecutor));
        }
        catch (Exception e) {
            if (this.componentRegistry.getConfigserverConfig().throwIfActiveSessionCannotBeLoaded()) {
                throw e;
            }
            log.log(Level.WARNING, "Failed loading session " + sessionId + ": No config for this session can be served", e);
        }
    }

    private void sessionRemoved(long sessionId) {
        RemoteSessionStateWatcher watcher = this.sessionStateWatchers.remove(sessionId);
        if (watcher != null) {
            watcher.close();
        }
        this.removeSession(sessionId);
        this.metrics.incRemovedSessions();
    }

    private void loadSessionIfActive(RemoteSession session) {
        for (ApplicationId applicationId : this.applicationRepo.activeApplications()) {
            if (this.applicationRepo.requireActiveSessionOf(applicationId) != session.getSessionId()) continue;
            log.log(Level.FINE, () -> "Found active application for session " + session.getSessionId() + " , loading it");
            this.reloadHandler.reloadConfig(session.ensureApplicationLoaded());
            log.log(Level.INFO, session.logPre() + "Application activated successfully: " + applicationId + " (generation " + session.getSessionId() + ")");
            return;
        }
    }

    public synchronized void close() {
        try {
            if (this.directoryCache != null) {
                this.directoryCache.close();
            }
        }
        catch (Exception e) {
            log.log(Level.WARNING, "Exception when closing path cache", e);
        }
        finally {
            this.checkForRemovedSessions(new ArrayList<Long>());
        }
    }

    private void nodeChanged() {
        this.zkWatcherExecutor.execute(() -> {
            HashMultiset sessionMetrics = HashMultiset.create();
            for (RemoteSession session : this.listSessions()) {
                sessionMetrics.add((Object)session.getStatus());
            }
            this.metrics.setNewSessions(sessionMetrics.count((Object)Session.Status.NEW));
            this.metrics.setPreparedSessions(sessionMetrics.count((Object)Session.Status.PREPARE));
            this.metrics.setActivatedSessions(sessionMetrics.count((Object)Session.Status.ACTIVATE));
            this.metrics.setDeactivatedSessions(sessionMetrics.count((Object)Session.Status.DEACTIVATE));
        });
    }

    private void childEvent(CuratorFramework ignored, PathChildrenCacheEvent event) {
        this.zkWatcherExecutor.execute(() -> {
            log.log(Level.FINE, () -> "Got child event: " + event);
            switch (event.getType()) {
                case CHILD_ADDED: {
                    this.sessionsChanged();
                    this.synchronizeOnNew(this.getSessionListFromDirectoryCache(Collections.singletonList(event.getData())));
                    break;
                }
                case CHILD_REMOVED: {
                    this.sessionsChanged();
                    break;
                }
                case CONNECTION_RECONNECTED: {
                    this.sessionsChanged();
                }
            }
        });
    }

    private void synchronizeOnNew(List<Long> sessionList) {
        for (long sessionId : sessionList) {
            RemoteSession session = (RemoteSession)this.getSession(sessionId);
            if (session == null) continue;
            log.log(Level.FINE, () -> session.logPre() + "Confirming upload for session " + sessionId);
            session.confirmUpload();
        }
    }
}

