/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.vespa.config.server;

import com.google.inject.Inject;
import com.yahoo.cloud.config.ConfigserverConfig;
import com.yahoo.component.AbstractComponent;
import com.yahoo.concurrent.DaemonThreadFactory;
import com.yahoo.config.provision.ApplicationId;
import com.yahoo.config.provision.Deployment;
import com.yahoo.config.provision.TransientException;
import com.yahoo.container.handler.VipStatus;
import com.yahoo.container.jdisc.state.StateMonitor;
import com.yahoo.log.LogLevel;
import com.yahoo.vespa.config.server.ApplicationRepository;
import com.yahoo.vespa.config.server.rpc.RpcServer;
import com.yahoo.vespa.config.server.version.VersionState;
import com.yahoo.yolean.Exceptions;
import java.time.Duration;
import java.time.Instant;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;

public class ConfigServerBootstrap
extends AbstractComponent
implements Runnable {
    private static final Logger log = Logger.getLogger(ConfigServerBootstrap.class.getName());
    private final ApplicationRepository applicationRepository;
    private final RpcServer server;
    private final Optional<Thread> serverThread;
    private final VersionState versionState;
    private final StateMonitor stateMonitor;
    private final VipStatus vipStatus;
    private final ConfigserverConfig configserverConfig;
    private final Duration maxDurationOfRedeployment;
    private final Duration sleepTimeWhenRedeployingFails;
    private final RedeployingApplicationsFails exitIfRedeployingApplicationsFails;
    private final ExecutorService rpcServerExecutor;

    @Inject
    public ConfigServerBootstrap(ApplicationRepository applicationRepository, RpcServer server, VersionState versionState, StateMonitor stateMonitor, VipStatus vipStatus) {
        this(applicationRepository, server, versionState, stateMonitor, vipStatus, Mode.BOOTSTRAP_IN_CONSTRUCTOR, RedeployingApplicationsFails.EXIT_JVM, applicationRepository.configserverConfig().hostedVespa() ? VipStatusMode.VIP_STATUS_FILE : VipStatusMode.VIP_STATUS_PROGRAMMATICALLY);
    }

    ConfigServerBootstrap(ApplicationRepository applicationRepository, RpcServer server, VersionState versionState, StateMonitor stateMonitor, VipStatus vipStatus, Mode mode, VipStatusMode vipStatusMode) {
        this(applicationRepository, server, versionState, stateMonitor, vipStatus, mode, RedeployingApplicationsFails.CONTINUE, vipStatusMode);
    }

    private ConfigServerBootstrap(ApplicationRepository applicationRepository, RpcServer server, VersionState versionState, StateMonitor stateMonitor, VipStatus vipStatus, Mode mode, RedeployingApplicationsFails exitIfRedeployingApplicationsFails, VipStatusMode vipStatusMode) {
        this.applicationRepository = applicationRepository;
        this.server = server;
        this.versionState = versionState;
        this.stateMonitor = stateMonitor;
        this.vipStatus = vipStatus;
        this.configserverConfig = applicationRepository.configserverConfig();
        this.maxDurationOfRedeployment = Duration.ofSeconds(this.configserverConfig.maxDurationOfBootstrap());
        this.sleepTimeWhenRedeployingFails = Duration.ofSeconds(this.configserverConfig.sleepTimeWhenRedeployingFails());
        this.exitIfRedeployingApplicationsFails = exitIfRedeployingApplicationsFails;
        this.rpcServerExecutor = Executors.newSingleThreadExecutor((ThreadFactory)new DaemonThreadFactory("config server RPC server"));
        log.log(LogLevel.INFO, "Bootstrap mode: " + mode + ", VIP status mode: " + vipStatusMode);
        this.initializing(vipStatusMode);
        switch (mode) {
            case BOOTSTRAP_IN_SEPARATE_THREAD: {
                this.serverThread = Optional.of(new Thread((Runnable)this, "config server bootstrap thread"));
                this.serverThread.get().start();
                break;
            }
            case BOOTSTRAP_IN_CONSTRUCTOR: {
                this.serverThread = Optional.empty();
                this.start();
                break;
            }
            case INITIALIZE_ONLY: {
                this.serverThread = Optional.empty();
                break;
            }
            default: {
                throw new IllegalArgumentException("Unknown bootstrap mode " + mode + ", legal values: " + Arrays.toString((Object[])Mode.values()));
            }
        }
    }

    public void deconstruct() {
        log.log(LogLevel.INFO, "Stopping config server");
        this.down();
        this.server.stop();
        log.log(LogLevel.INFO, "RPC server stopped");
        this.rpcServerExecutor.shutdown();
        this.serverThread.ifPresent(thread -> {
            try {
                thread.join();
            }
            catch (InterruptedException e) {
                log.log(LogLevel.WARNING, "Error joining server thread on shutdown: " + e.getMessage());
            }
        });
    }

    @Override
    public void run() {
        this.start();
        do {
            try {
                Thread.sleep(1000L);
            }
            catch (InterruptedException e) {
                log.log((Level)LogLevel.ERROR, "Got interrupted", e);
                break;
            }
        } while (this.server.isRunning());
        this.down();
    }

    public void start() {
        if (this.versionState.isUpgraded()) {
            log.log(LogLevel.INFO, "Config server upgrading from " + this.versionState.storedVersion() + " to " + this.versionState.currentVersion() + ". Redeploying all applications");
            try {
                if (!this.redeployAllApplications()) {
                    this.redeployingApplicationsFailed();
                    return;
                }
                this.versionState.saveNewVersion();
                log.log(LogLevel.INFO, "All applications redeployed successfully");
            }
            catch (Exception e) {
                log.log((Level)LogLevel.ERROR, "Redeployment of applications failed", e);
                this.redeployingApplicationsFailed();
                return;
            }
        }
        this.startRpcServer();
        this.up();
    }

    StateMonitor.Status status() {
        return this.stateMonitor.status();
    }

    private void up() {
        this.vipStatus.setInRotation(Boolean.valueOf(true));
    }

    private void down() {
        this.vipStatus.setInRotation(Boolean.valueOf(false));
    }

    private void initializing(VipStatusMode vipStatusMode) {
        this.stateMonitor.status(StateMonitor.Status.initializing);
        if (vipStatusMode == VipStatusMode.VIP_STATUS_PROGRAMMATICALLY) {
            this.vipStatus.setInRotation(Boolean.valueOf(false));
        }
    }

    private void startRpcServer() {
        log.log(LogLevel.INFO, "Starting RPC server");
        this.rpcServerExecutor.execute(this.server);
        Instant end = Instant.now().plus(Duration.ofSeconds(10L));
        while (!this.server.isRunning() && Instant.now().isBefore(end)) {
            try {
                Thread.sleep(10L);
            }
            catch (InterruptedException e) {
                log.log((Level)LogLevel.ERROR, "Got interrupted", e);
                break;
            }
        }
        if (!this.server.isRunning()) {
            throw new RuntimeException("RPC server not started in 10 seconds");
        }
        log.log(LogLevel.INFO, "RPC server started");
    }

    private void redeployingApplicationsFailed() {
        if (this.exitIfRedeployingApplicationsFails == RedeployingApplicationsFails.EXIT_JVM) {
            System.exit(1);
        }
    }

    private boolean redeployAllApplications() throws InterruptedException {
        Instant end = Instant.now().plus(this.maxDurationOfRedeployment);
        Set<ApplicationId> applicationsNotRedeployed = this.applicationRepository.listApplications();
        do {
            if ((applicationsNotRedeployed = this.redeployApplications(applicationsNotRedeployed)).isEmpty()) continue;
            log.log(LogLevel.INFO, "Redeployment of " + applicationsNotRedeployed + " failed, will retry in " + this.sleepTimeWhenRedeployingFails);
            Thread.sleep(this.sleepTimeWhenRedeployingFails.toMillis());
        } while (!applicationsNotRedeployed.isEmpty() && Instant.now().isBefore(end));
        if (!applicationsNotRedeployed.isEmpty()) {
            log.log((Level)LogLevel.ERROR, "Redeploying applications not finished after " + this.maxDurationOfRedeployment + ", exiting, applications that failed redeployment: " + applicationsNotRedeployed);
            return false;
        }
        return true;
    }

    private Set<ApplicationId> redeployApplications(Set<ApplicationId> applicationIds) throws InterruptedException {
        ExecutorService executor = Executors.newFixedThreadPool(this.configserverConfig.numParallelTenantLoaders(), (ThreadFactory)new DaemonThreadFactory("redeploy apps"));
        HashMap futures = new HashMap();
        HashSet<ApplicationId> failedDeployments = new HashSet<ApplicationId>();
        for (ApplicationId applicationId : applicationIds) {
            Optional<Deployment> deploymentOptional = this.applicationRepository.deployFromLocalActive(applicationId, true);
            if (!deploymentOptional.isPresent()) continue;
            futures.put(applicationId, executor.submit(() -> ((Deployment)deploymentOptional.get()).activate()));
        }
        for (Map.Entry entry : futures.entrySet()) {
            ApplicationId app = (ApplicationId)entry.getKey();
            try {
                ((Future)entry.getValue()).get();
            }
            catch (TransientException e) {
                log.log(LogLevel.INFO, "Redeploying " + app + " failed with transient error, will retry after bootstrap: " + Exceptions.toMessageString((Throwable)e));
            }
            catch (ExecutionException e) {
                log.log(LogLevel.WARNING, "Redeploying " + app + " failed, will retry", e);
                failedDeployments.add(app);
            }
        }
        executor.shutdown();
        executor.awaitTermination(365L, TimeUnit.DAYS);
        return failedDeployments;
    }

    static enum VipStatusMode {
        VIP_STATUS_FILE,
        VIP_STATUS_PROGRAMMATICALLY;

    }

    static enum RedeployingApplicationsFails {
        EXIT_JVM,
        CONTINUE;

    }

    static enum Mode {
        BOOTSTRAP_IN_CONSTRUCTOR,
        BOOTSTRAP_IN_SEPARATE_THREAD,
        INITIALIZE_ONLY;

    }
}

