/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.config.subscription.impl;

import com.yahoo.config.ConfigInstance;
import com.yahoo.config.ConfigurationRuntimeException;
import com.yahoo.config.subscription.ConfigSourceSet;
import com.yahoo.config.subscription.impl.ConfigSubscription;
import com.yahoo.config.subscription.impl.JRTConfigSubscription;
import com.yahoo.jrt.Request;
import com.yahoo.jrt.RequestWaiter;
import com.yahoo.log.LogLevel;
import com.yahoo.vespa.config.Connection;
import com.yahoo.vespa.config.ConnectionPool;
import com.yahoo.vespa.config.ErrorCode;
import com.yahoo.vespa.config.ErrorType;
import com.yahoo.vespa.config.TimingValues;
import com.yahoo.vespa.config.protocol.JRTClientConfigRequest;
import com.yahoo.vespa.config.protocol.JRTConfigRequestFactory;
import com.yahoo.vespa.config.protocol.Trace;
import com.yahoo.yolean.Exceptions;
import java.time.Duration;
import java.time.Instant;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;

public class JRTConfigRequester
implements RequestWaiter {
    private static final Logger log = Logger.getLogger(JRTConfigRequester.class.getName());
    public static final ConfigSourceSet defaultSourceSet = ConfigSourceSet.createDefault();
    private static final int TRACELEVEL = 6;
    private final TimingValues timingValues;
    private int fatalFailures = 0;
    private int transientFailures = 0;
    private final ScheduledThreadPoolExecutor scheduler = new ScheduledThreadPoolExecutor(1, new JRTSourceThreadFactory());
    private Instant suspendWarningLogged = Instant.MIN;
    private Instant noApplicationWarningLogged = Instant.MIN;
    private static final Duration delayBetweenWarnings = Duration.ofSeconds(60L);
    private final ConnectionPool connectionPool;
    static final float randomFraction = 0.2f;
    private static final Double additionalTimeForClientTimeout = 10.0;

    public static JRTConfigRequester get(ConnectionPool connectionPool, TimingValues timingValues) {
        return new JRTConfigRequester(connectionPool, timingValues);
    }

    JRTConfigRequester(ConnectionPool connectionPool, TimingValues timingValues) {
        this.connectionPool = connectionPool;
        this.timingValues = timingValues;
    }

    public <T extends ConfigInstance> void request(JRTConfigSubscription<T> sub) {
        JRTClientConfigRequest req = JRTConfigRequestFactory.createFromSub(sub);
        this.doRequest(sub, req);
    }

    private <T extends ConfigInstance> void doRequest(JRTConfigSubscription<T> sub, JRTClientConfigRequest req) {
        Connection connection = this.connectionPool.getCurrent();
        req.getRequest().setContext((Object)new RequestContext(sub, req, connection));
        if (!req.validateParameters()) {
            throw new ConfigurationRuntimeException("Error in parameters for config request: " + req);
        }
        double jrtClientTimeout = this.getClientTimeout(req);
        if (log.isLoggable((Level)LogLevel.DEBUG)) {
            log.log((Level)LogLevel.DEBUG, "Requesting config for " + sub + " on connection " + connection + " with client timeout " + jrtClientTimeout + (String)(log.isLoggable((Level)LogLevel.SPAM) ? ",defcontent=" + req.getDefContent().asString() : ""));
        }
        connection.invokeAsync(req.getRequest(), jrtClientTimeout, this);
    }

    public void handleRequestDone(Request req) {
        JRTConfigSubscription sub = null;
        try {
            RequestContext context = (RequestContext)req.getContext();
            sub = context.sub;
            this.doHandle(sub, context.jrtReq, context.connection);
        }
        catch (RuntimeException e) {
            if (sub != null) {
                sub.setException(e);
            }
            log.log(Level.SEVERE, "Failed to get subscription object from JRT config callback: " + Exceptions.toMessageString((Throwable)e));
        }
    }

    private void doHandle(JRTConfigSubscription<ConfigInstance> sub, JRTClientConfigRequest jrtReq, Connection connection) {
        boolean validResponse = jrtReq.validateResponse();
        log.log((Level)LogLevel.DEBUG, () -> "Request callback " + (validResponse ? "valid" : "invalid") + ". Req: " + jrtReq + "\nSpec: " + connection);
        if (sub.getState() == ConfigSubscription.State.CLOSED) {
            return;
        }
        Trace trace = jrtReq.getResponseTrace();
        trace.trace(6, "JRTConfigRequester.doHandle()");
        log.log((Level)LogLevel.SPAM, trace::toString);
        if (validResponse) {
            this.handleOKRequest(jrtReq, sub, connection);
        } else {
            this.logWhenErrorResponse(jrtReq, connection);
            this.handleFailedRequest(jrtReq, sub, connection);
        }
    }

    private void logWhenErrorResponse(JRTClientConfigRequest jrtReq, Connection connection) {
        switch (jrtReq.errorCode()) {
            case 104: {
                log.log((Level)LogLevel.DEBUG, () -> "Request callback failed: " + jrtReq.errorMessage() + "\nConnection spec: " + connection);
                break;
            }
            case 100005: 
            case 100300: {
                if (!this.noApplicationWarningLogged.isBefore(Instant.now().minus(delayBetweenWarnings))) break;
                log.log(LogLevel.WARNING, "Request callback failed: " + ErrorCode.getName(jrtReq.errorCode()) + ". Connection spec: " + connection.getAddress() + ", error message: " + jrtReq.errorMessage());
                this.noApplicationWarningLogged = Instant.now();
                break;
            }
            default: {
                log.log(LogLevel.WARNING, "Request callback failed. Req: " + jrtReq + "\nSpec: " + connection.getAddress() + " . Req error message: " + jrtReq.errorMessage());
            }
        }
    }

    private void handleFailedRequest(JRTClientConfigRequest jrtReq, JRTConfigSubscription<ConfigInstance> sub, Connection connection) {
        boolean configured;
        boolean bl = configured = sub.getConfigState().getConfig() != null;
        if (configured) {
            log.log(LogLevel.INFO, "Failure of config subscription, clients will keep existing config until resolved: " + sub);
        }
        ErrorType errorType = ErrorType.getErrorType(jrtReq.errorCode());
        this.connectionPool.setError(connection, jrtReq.errorCode());
        long delay = JRTConfigRequester.calculateFailedRequestDelay(errorType, this.transientFailures, this.fatalFailures, this.timingValues, configured);
        if (errorType == ErrorType.TRANSIENT) {
            this.handleTransientlyFailed(jrtReq, sub, delay, connection);
        } else {
            this.handleFatallyFailed(jrtReq, sub, delay);
        }
    }

    static long calculateFailedRequestDelay(ErrorType errorCode, int transientFailures, int fatalFailures, TimingValues timingValues, boolean configured) {
        long delay = configured ? timingValues.getConfiguredErrorDelay() : timingValues.getUnconfiguredDelay();
        if (errorCode == ErrorType.TRANSIENT) {
            delay *= (long)Math.min(transientFailures + 1, timingValues.getMaxDelayMultiplier());
        } else {
            delay = timingValues.getFixedDelay() + delay * (long)Math.min(fatalFailures, timingValues.getMaxDelayMultiplier());
            delay = timingValues.getPlusMinusFractionRandom(delay, 0.2f);
        }
        return delay;
    }

    private void handleTransientlyFailed(JRTClientConfigRequest jrtReq, JRTConfigSubscription<ConfigInstance> sub, long delay, Connection connection) {
        ++this.transientFailures;
        if (this.suspendWarningLogged.isBefore(Instant.now().minus(delayBetweenWarnings))) {
            log.log(LogLevel.INFO, "Connection to " + connection.getAddress() + " failed or timed out, clients will keep existing config, will keep trying.");
            this.suspendWarningLogged = Instant.now();
        }
        if (sub.getState() != ConfigSubscription.State.OPEN) {
            return;
        }
        this.scheduleNextRequest(jrtReq, sub, delay, this.calculateErrorTimeout());
    }

    private long calculateErrorTimeout() {
        return this.timingValues.getPlusMinusFractionRandom(this.timingValues.getErrorTimeout(), 0.2f);
    }

    private void handleFatallyFailed(JRTClientConfigRequest jrtReq, JRTConfigSubscription<ConfigInstance> sub, long delay) {
        if (sub.getState() != ConfigSubscription.State.OPEN) {
            return;
        }
        ++this.fatalFailures;
        Object logLevel = sub.getConfigState().getConfig() == null ? LogLevel.DEBUG : LogLevel.INFO;
        String logMessage = "Request for config " + jrtReq.getShortDescription() + "' failed with error code " + jrtReq.errorCode() + " (" + jrtReq.errorMessage() + "), scheduling new connect  in " + delay + " ms";
        log.log((Level)logLevel, logMessage);
        this.scheduleNextRequest(jrtReq, sub, delay, this.calculateErrorTimeout());
    }

    private void handleOKRequest(JRTClientConfigRequest jrtReq, JRTConfigSubscription<ConfigInstance> sub, Connection connection) {
        this.fatalFailures = 0;
        this.transientFailures = 0;
        this.suspendWarningLogged = Instant.MIN;
        this.noApplicationWarningLogged = Instant.MIN;
        connection.setSuccess();
        sub.setLastCallBackOKTS(System.currentTimeMillis());
        if (jrtReq.hasUpdatedGeneration()) {
            sub.getReqQueue().clear();
            boolean putOK = sub.getReqQueue().offer(jrtReq);
            if (!putOK) {
                sub.setException((RuntimeException)new ConfigurationRuntimeException("Could not put returned request on queue of subscription " + sub));
            }
        }
        if (sub.getState() != ConfigSubscription.State.OPEN) {
            return;
        }
        this.scheduleNextRequest(jrtReq, sub, this.calculateSuccessDelay(), this.calculateSuccessTimeout());
    }

    private long calculateSuccessTimeout() {
        return this.timingValues.getPlusMinusFractionRandom(this.timingValues.getSuccessTimeout(), 0.2f);
    }

    private long calculateSuccessDelay() {
        return this.timingValues.getPlusMinusFractionRandom(this.timingValues.getFixedDelay(), 0.2f);
    }

    private void scheduleNextRequest(JRTClientConfigRequest jrtReq, JRTConfigSubscription<?> sub, long delay, long timeout) {
        long delayBeforeSendingRequest = delay < 0L ? 0L : delay;
        JRTClientConfigRequest jrtReqNew = jrtReq.nextRequest(timeout);
        log.log((Level)LogLevel.DEBUG, this.timingValues::toString);
        log.log((Level)LogLevel.DEBUG, () -> "Scheduling new request " + delayBeforeSendingRequest + " millis from now for " + jrtReqNew.getConfigKey());
        this.scheduler.schedule(new GetConfigTask(jrtReqNew, sub), delayBeforeSendingRequest, TimeUnit.MILLISECONDS);
    }

    public void close() {
        this.suspendWarningLogged = Instant.now();
        this.noApplicationWarningLogged = Instant.now();
        this.connectionPool.close();
        this.scheduler.shutdown();
    }

    int getTransientFailures() {
        return this.transientFailures;
    }

    int getFatalFailures() {
        return this.fatalFailures;
    }

    public ConnectionPool getConnectionPool() {
        return this.connectionPool;
    }

    private Double getClientTimeout(JRTClientConfigRequest request) {
        return (double)request.getTimeout() / 1000.0 + additionalTimeForClientTimeout;
    }

    private static class RequestContext {
        final JRTConfigSubscription sub;
        final JRTClientConfigRequest jrtReq;
        final Connection connection;

        private RequestContext(JRTConfigSubscription sub, JRTClientConfigRequest jrtReq, Connection connection) {
            this.sub = sub;
            this.jrtReq = jrtReq;
            this.connection = connection;
        }
    }

    private static class JRTSourceThreadFactory
    implements ThreadFactory {
        private JRTSourceThreadFactory() {
        }

        @Override
        public Thread newThread(Runnable runnable) {
            Thread t = new Thread(runnable, String.format("jrt-config-requester-%d", System.currentTimeMillis()));
            t.setDaemon(true);
            return t;
        }
    }

    private class GetConfigTask
    implements Runnable {
        private final JRTClientConfigRequest jrtReq;
        private final JRTConfigSubscription<?> sub;

        GetConfigTask(JRTClientConfigRequest jrtReq, JRTConfigSubscription<?> sub) {
            this.jrtReq = jrtReq;
            this.sub = sub;
        }

        @Override
        public void run() {
            JRTConfigRequester.this.doRequest(this.sub, this.jrtReq);
        }
    }
}

