/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.vespa.config.proxy.filedistribution;

import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.SettableFuture;
import com.yahoo.config.FileReference;
import com.yahoo.log.LogLevel;
import com.yahoo.vespa.config.ConnectionPool;
import com.yahoo.vespa.config.proxy.filedistribution.FileReferenceDownload;
import com.yahoo.vespa.config.proxy.filedistribution.FileReferenceDownloader;
import com.yahoo.vespa.defaults.Defaults;
import com.yahoo.yolean.Exceptions;
import java.io.File;
import java.time.Duration;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import java.util.logging.Logger;

public class FileDownloader {
    private static final Logger log = Logger.getLogger(FileDownloader.class.getName());
    private final File downloadDirectory;
    private final Duration timeout;
    private final FileReferenceDownloader fileReferenceDownloader;

    public FileDownloader(ConnectionPool connectionPool) {
        this(connectionPool, new File(Defaults.getDefaults().underVespaHome("var/db/vespa/filedistribution")), Duration.ofMinutes(15L));
    }

    FileDownloader(ConnectionPool connectionPool, File downloadDirectory, Duration timeout) {
        this.downloadDirectory = downloadDirectory;
        this.timeout = timeout;
        this.fileReferenceDownloader = new FileReferenceDownloader(downloadDirectory, connectionPool, timeout);
    }

    public Optional<File> getFile(FileReference fileReference) {
        Objects.requireNonNull(fileReference, "file reference cannot be null");
        File directory = new File(this.downloadDirectory, fileReference.value());
        log.log((Level)LogLevel.DEBUG, "Checking if there is a file in '" + directory.getAbsolutePath() + "' ");
        Optional<File> file = this.getFileFromFileSystem(fileReference, directory);
        if (file.isPresent()) {
            return file;
        }
        log.log(LogLevel.INFO, "File reference '" + fileReference.value() + "' not found in " + directory.getAbsolutePath() + ", starting download");
        return this.queueForDownload(fileReference, this.timeout);
    }

    public void queueForDownload(List<FileReference> fileReferences) {
        fileReferences.forEach(this::queueForDownload);
    }

    public void receiveFile(FileReference fileReference, String filename, byte[] content, long xxHash) {
        this.fileReferenceDownloader.receiveFile(fileReference, filename, content, xxHash);
    }

    double downloadStatus(FileReference fileReference) {
        return this.fileReferenceDownloader.downloadStatus(fileReference.value());
    }

    public Map<FileReference, Double> downloadStatus() {
        return this.fileReferenceDownloader.downloadStatus();
    }

    File downloadDirectory() {
        return this.downloadDirectory;
    }

    private Optional<File> getFileFromFileSystem(FileReference fileReference, File directory) {
        File[] files = directory.listFiles();
        if (directory.exists() && directory.isDirectory() && files != null && files.length > 0) {
            if (files.length != 1) {
                throw new RuntimeException("More than one file in  '" + fileReference.value() + "', expected only one, unable to proceed");
            }
            File file = files[0];
            if (!file.exists()) {
                throw new RuntimeException("File with reference '" + fileReference.value() + "' does not exist");
            }
            if (!file.canRead()) {
                throw new RuntimeException("File with reference '" + fileReference.value() + "'exists, but unable to read it");
            }
            this.fileReferenceDownloader.setDownloadStatus(fileReference.value(), 100.0);
            return Optional.of(file);
        }
        return Optional.empty();
    }

    private synchronized Optional<File> queueForDownload(FileReference fileReference, Duration timeout) {
        if (this.fileReferenceDownloader.isDownloading(fileReference)) {
            log.log(LogLevel.INFO, "Already downloading '" + fileReference.value() + "'");
            ListenableFuture<Optional<File>> future = this.fileReferenceDownloader.addDownloadListener(fileReference, () -> this.getFile(fileReference));
            try {
                return (Optional)future.get();
            }
            catch (InterruptedException | ExecutionException e) {
                throw new RuntimeException("Failed downloading file reference '" + fileReference.value() + "': " + Exceptions.toMessageString((Throwable)e));
            }
        }
        SettableFuture future = SettableFuture.create();
        this.queueForDownload(new FileReferenceDownload(fileReference, (SettableFuture<Optional<File>>)future));
        log.log(LogLevel.INFO, "Queued '" + fileReference.value() + "' for download with timeout " + timeout);
        try {
            Optional fileDownloaded;
            try {
                log.log(LogLevel.INFO, "Waiting for '" + fileReference.value() + "' to download");
                fileDownloaded = (Optional)future.get(timeout.getSeconds() - 1L, TimeUnit.SECONDS);
                log.log(LogLevel.INFO, "'" + fileReference.value() + "' downloaded");
            }
            catch (TimeoutException e) {
                log.log(LogLevel.WARNING, "Downloading '" + fileReference.value() + "' timed out");
                return Optional.empty();
            }
            return fileDownloaded;
        }
        catch (InterruptedException | ExecutionException e) {
            throw new RuntimeException("Could not download '" + fileReference.value() + "'");
        }
    }

    private synchronized void queueForDownload(FileReference fileReference) {
        this.queueForDownload(new FileReferenceDownload(fileReference, (SettableFuture<Optional<File>>)SettableFuture.create()));
    }

    private synchronized void queueForDownload(FileReferenceDownload fileReferenceDownload) {
        this.fileReferenceDownloader.addToDownloadQueue(fileReferenceDownload);
    }

    Set<FileReference> queuedDownloads() {
        return this.fileReferenceDownloader.queuedDownloads();
    }
}

