/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.vespa.model.application.validation;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import com.google.common.base.Joiner;
import com.yahoo.tensor.TensorType;
import java.io.IOException;
import java.io.Reader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

public class ConstantTensorJsonValidator {
    private static final String FIELD_CELLS = "cells";
    private static final String FIELD_ADDRESS = "address";
    private static final String FIELD_VALUE = "value";
    private static final JsonFactory jsonFactory = new JsonFactory();
    private JsonParser parser;
    private Map<String, TensorType.Dimension> tensorDimensions;

    public void validate(String fileName, TensorType type, Reader tensorData) {
        if (fileName.endsWith(".json")) {
            this.validateTensor(type, tensorData);
        } else if (!fileName.endsWith(".json.lz4")) {
            throw new IllegalArgumentException("Ranking constant file names must end with either '.json' or '.json.lz4'");
        }
    }

    private void validateTensor(TensorType type, Reader tensorData) {
        this.wrapIOException(() -> {
            this.parser = jsonFactory.createParser(tensorData);
            this.tensorDimensions = type.dimensions().stream().collect(Collectors.toMap(TensorType.Dimension::name, Function.identity()));
            this.assertNextTokenIs(JsonToken.START_OBJECT);
            this.assertNextTokenIs(JsonToken.FIELD_NAME);
            this.assertFieldNameIs(FIELD_CELLS);
            this.assertNextTokenIs(JsonToken.START_ARRAY);
            while (this.parser.nextToken() != JsonToken.END_ARRAY) {
                this.validateTensorCell();
            }
            this.assertNextTokenIs(JsonToken.END_OBJECT);
        });
    }

    private void validateTensorCell() {
        this.wrapIOException(() -> {
            this.assertCurrentTokenIs(JsonToken.START_OBJECT);
            ArrayList<String> fieldNameCandidates = new ArrayList<String>(Arrays.asList(FIELD_ADDRESS, FIELD_VALUE));
            for (int i = 0; i < 2; ++i) {
                this.assertNextTokenIs(JsonToken.FIELD_NAME);
                String fieldName = this.parser.getCurrentName();
                if (fieldNameCandidates.contains(fieldName)) {
                    fieldNameCandidates.remove(fieldName);
                    if (fieldName.equals(FIELD_ADDRESS)) {
                        this.validateTensorAddress();
                        continue;
                    }
                    if (!fieldName.equals(FIELD_VALUE)) continue;
                    this.validateTensorValue();
                    continue;
                }
                throw new InvalidConstantTensor(this.parser, "Only \"address\" or \"value\" fields are permitted within a cell object");
            }
            this.assertNextTokenIs(JsonToken.END_OBJECT);
        });
    }

    private void validateTensorAddress() throws IOException {
        this.assertNextTokenIs(JsonToken.START_OBJECT);
        HashSet<String> cellDimensions = new HashSet<String>(this.tensorDimensions.keySet());
        while (this.parser.nextToken() != JsonToken.END_OBJECT) {
            this.assertCurrentTokenIs(JsonToken.FIELD_NAME);
            String dimensionName = this.parser.getCurrentName();
            TensorType.Dimension dimension = this.tensorDimensions.get(dimensionName);
            if (dimension == null) {
                throw new InvalidConstantTensor(this.parser, String.format("Tensor dimension \"%s\" does not exist", this.parser.getCurrentName()));
            }
            if (!cellDimensions.contains(dimensionName)) {
                throw new InvalidConstantTensor(this.parser, String.format("Duplicate tensor dimension \"%s\"", this.parser.getCurrentName()));
            }
            cellDimensions.remove(dimensionName);
            this.validateTensorCoordinate(dimension);
        }
        if (!cellDimensions.isEmpty()) {
            throw new InvalidConstantTensor(this.parser, String.format("Tensor address missing dimension(s): %s", Joiner.on((String)", ").join(cellDimensions)));
        }
    }

    private void validateTensorCoordinate(TensorType.Dimension dimension) throws IOException {
        JsonToken token = this.parser.nextToken();
        if (token != JsonToken.VALUE_STRING) {
            throw new InvalidConstantTensor(this.parser, String.format("Tensor coordinate is not a string (%s)", token.toString()));
        }
        if (dimension instanceof TensorType.IndexedBoundDimension) {
            this.validateBoundedCoordinate((TensorType.IndexedBoundDimension)dimension);
        } else if (dimension instanceof TensorType.IndexedUnboundDimension) {
            this.validateUnboundedCoordinate(dimension);
        }
    }

    private void validateBoundedCoordinate(TensorType.IndexedBoundDimension dimension) {
        this.wrapIOException(() -> {
            try {
                int value = Integer.parseInt(this.parser.getValueAsString());
                if ((long)value >= (Long)dimension.size().get()) {
                    throw new InvalidConstantTensor(this.parser, String.format("Coordinate \"%s\" not within limits of bounded dimension %s", value, dimension.name()));
                }
            }
            catch (NumberFormatException e) {
                this.throwCoordinateIsNotInteger(this.parser.getValueAsString(), dimension.name());
            }
        });
    }

    private void validateUnboundedCoordinate(TensorType.Dimension dimension) {
        this.wrapIOException(() -> {
            try {
                Integer.parseInt(this.parser.getValueAsString());
            }
            catch (NumberFormatException e) {
                this.throwCoordinateIsNotInteger(this.parser.getValueAsString(), dimension.name());
            }
        });
    }

    private void throwCoordinateIsNotInteger(String value, String dimensionName) {
        throw new InvalidConstantTensor(this.parser, String.format("Coordinate \"%s\" for dimension %s is not an integer", value, dimensionName));
    }

    private void validateTensorValue() throws IOException {
        JsonToken token = this.parser.nextToken();
        if (token != JsonToken.VALUE_NUMBER_FLOAT && token != JsonToken.VALUE_NUMBER_INT) {
            throw new InvalidConstantTensor(this.parser, String.format("Tensor value is not a number (%s)", token.toString()));
        }
    }

    private void assertCurrentTokenIs(JsonToken wantedToken) {
        this.assertTokenIs(this.parser.getCurrentToken(), wantedToken);
    }

    private void assertNextTokenIs(JsonToken wantedToken) throws IOException {
        this.assertTokenIs(this.parser.nextToken(), wantedToken);
    }

    private void assertTokenIs(JsonToken token, JsonToken wantedToken) {
        if (token != wantedToken) {
            throw new InvalidConstantTensor(this.parser, String.format("Expected JSON token %s, but got %s", wantedToken.toString(), token.toString()));
        }
    }

    private void assertFieldNameIs(String wantedFieldName) throws IOException {
        String actualFieldName = this.parser.getCurrentName();
        if (!actualFieldName.equals(wantedFieldName)) {
            throw new InvalidConstantTensor(this.parser, String.format("Expected field name \"%s\", got \"%s\"", wantedFieldName, actualFieldName));
        }
    }

    private void wrapIOException(SubroutineThrowingIOException lambda) {
        try {
            lambda.invoke();
        }
        catch (IOException e) {
            throw new InvalidConstantTensor(this.parser, e);
        }
    }

    @FunctionalInterface
    private static interface SubroutineThrowingIOException {
        public void invoke() throws IOException;
    }

    static class InvalidConstantTensor
    extends RuntimeException {
        InvalidConstantTensor(JsonParser parser, String message) {
            super(message + " " + parser.getCurrentLocation().toString());
        }

        InvalidConstantTensor(JsonParser parser, Exception base) {
            super("Failed to parse JSON stream " + parser.getCurrentLocation().toString(), base);
        }
    }
}

