/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.vespa.clustercontroller.core;

import com.yahoo.vdslib.distribution.ConfiguredNode;
import com.yahoo.vdslib.state.ClusterState;
import com.yahoo.vdslib.state.Node;
import com.yahoo.vdslib.state.NodeState;
import com.yahoo.vdslib.state.NodeType;
import com.yahoo.vdslib.state.State;
import com.yahoo.vespa.clustercontroller.core.ClusterEvent;
import com.yahoo.vespa.clustercontroller.core.ContentCluster;
import com.yahoo.vespa.clustercontroller.core.EventLogInterface;
import com.yahoo.vespa.clustercontroller.core.FleetControllerContext;
import com.yahoo.vespa.clustercontroller.core.FleetControllerOptions;
import com.yahoo.vespa.clustercontroller.core.NodeEvent;
import com.yahoo.vespa.clustercontroller.core.NodeInfo;
import com.yahoo.vespa.clustercontroller.core.Timer;
import com.yahoo.vespa.clustercontroller.core.database.DatabaseHandler;
import com.yahoo.vespa.clustercontroller.core.listeners.NodeListener;
import java.io.Serializable;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.logging.Level;
import java.util.logging.Logger;

public class StateChangeHandler {
    private static final Logger log = Logger.getLogger(StateChangeHandler.class.getName());
    private final FleetControllerContext context;
    private final Timer timer;
    private final EventLogInterface eventLog;
    private boolean stateMayHaveChanged = false;
    private boolean isMaster = false;
    private Map<NodeType, Integer> maxTransitionTime = new TreeMap<NodeType, Integer>();
    private int maxInitProgressTime = 5000;
    private int maxPrematureCrashes = 4;
    private long stableStateTimePeriod = 3600000L;
    private int maxSlobrokDisconnectGracePeriod = 1000;
    private static final boolean disableUnstableNodes = true;

    public StateChangeHandler(FleetControllerContext context, Timer timer, EventLogInterface eventLog) {
        this.context = context;
        this.timer = timer;
        this.eventLog = eventLog;
        this.maxTransitionTime.put(NodeType.DISTRIBUTOR, 5000);
        this.maxTransitionTime.put(NodeType.STORAGE, 5000);
    }

    public void handleAllDistributorsInSync(ClusterState currentState, Set<ConfiguredNode> nodes, DatabaseHandler database, DatabaseHandler.DatabaseContext dbContext) {
        int startTimestampsReset = 0;
        this.context.log(log, Level.FINE, "handleAllDistributorsInSync invoked for state version %d", currentState.getVersion(), new Object[0]);
        for (NodeType nodeType : NodeType.getTypes()) {
            for (ConfiguredNode configuredNode : nodes) {
                Node node = new Node(nodeType, configuredNode.index());
                NodeInfo nodeInfo = dbContext.getCluster().getNodeInfo(node);
                NodeState nodeState = currentState.getNodeState(node);
                if (nodeInfo != null && nodeState != null) {
                    if (nodeState.getStartTimestamp() > nodeInfo.getStartTimestamp()) {
                        log.log(Level.FINE, () -> String.format("Storing away new start timestamp for node %s (%d)", node, nodeState.getStartTimestamp()));
                        nodeInfo.setStartTimestamp(nodeState.getStartTimestamp());
                    }
                    if (nodeState.getStartTimestamp() <= 0L) continue;
                    log.log(Level.FINE, "Resetting timestamp in cluster state for node %s", node);
                    ++startTimestampsReset;
                    continue;
                }
                if (!log.isLoggable(Level.FINE)) continue;
                log.log(Level.FINE, node + ": " + (Serializable)(nodeInfo == null ? "null" : Long.valueOf(nodeInfo.getStartTimestamp())) + ", " + (Serializable)(nodeState == null ? "null" : Long.valueOf(nodeState.getStartTimestamp())));
            }
        }
        if (startTimestampsReset > 0) {
            this.eventLog.add(new ClusterEvent(ClusterEvent.Type.SYSTEMSTATE, "Reset " + startTimestampsReset + " start timestamps as all available distributors have seen newest cluster state.", this.timer.getCurrentTimeInMillis()));
            this.stateMayHaveChanged = true;
            database.saveStartTimestamps(dbContext);
        } else {
            log.log(Level.FINE, "Found no start timestamps to reset in cluster state.");
        }
    }

    public boolean stateMayHaveChanged() {
        return this.stateMayHaveChanged;
    }

    public void setStateChangedFlag() {
        this.stateMayHaveChanged = true;
    }

    public void unsetStateChangedFlag() {
        this.stateMayHaveChanged = false;
    }

    public void setMaster(boolean isMaster) {
        this.isMaster = isMaster;
    }

    public void setMaxTransitionTime(Map<NodeType, Integer> map) {
        this.maxTransitionTime = map;
    }

    public void setMaxInitProgressTime(int millisecs) {
        this.maxInitProgressTime = millisecs;
    }

    public void setMaxSlobrokDisconnectGracePeriod(int millisecs) {
        this.maxSlobrokDisconnectGracePeriod = millisecs;
    }

    public void setStableStateTimePeriod(long millisecs) {
        this.stableStateTimePeriod = millisecs;
    }

    public void setMaxPrematureCrashes(int count) {
        this.maxPrematureCrashes = count;
    }

    public void handleNewReportedNodeState(ClusterState currentClusterState, NodeInfo node, NodeState reportedState, NodeListener nodeListener) {
        NodeState currentState = currentClusterState.getNodeState(node.getNode());
        Level level = currentState.equals((Object)reportedState) && node.getVersion() == 0 ? Level.FINEST : Level.FINE;
        log.log(level, () -> String.format("Got nodestate reply from %s: %s (Current state is %s)", node, node.getReportedState().getTextualDifference(reportedState), currentState.toString(true)));
        long currentTime = this.timer.getCurrentTimeInMillis();
        if (reportedState.getState().equals((Object)State.DOWN)) {
            node.setTimeOfFirstFailingConnectionAttempt(currentTime);
        }
        if (!reportedState.similarTo((Object)node.getReportedState())) {
            if (reportedState.getState().equals((Object)State.DOWN)) {
                this.eventLog.addNodeOnlyEvent(NodeEvent.forBaseline(node, "Failed to get node state: " + reportedState.toString(true), NodeEvent.Type.REPORTED, currentTime), Level.INFO);
            } else {
                this.eventLog.addNodeOnlyEvent(NodeEvent.forBaseline(node, "Now reporting state " + reportedState.toString(true), NodeEvent.Type.REPORTED, currentTime), Level.FINE);
            }
        }
        if (reportedState.equals((Object)node.getReportedState()) && !reportedState.getState().equals((Object)State.INITIALIZING)) {
            return;
        }
        this.updateNodeInfoFromReportedState(node, currentState, reportedState, nodeListener);
        if (reportedState.getMinUsedBits() != currentState.getMinUsedBits()) {
            int oldCount = currentState.getMinUsedBits();
            int newCount = reportedState.getMinUsedBits();
            log.log(Level.FINE, () -> String.format("Altering node state to reflect that min distribution bit count has changed from %d to %d", oldCount, newCount));
            this.eventLog.add(NodeEvent.forBaseline(node, String.format("Altered min distribution bit count from %d to %d", oldCount, newCount), NodeEvent.Type.CURRENT, currentTime), this.isMaster);
        } else {
            log.log(Level.FINE, () -> String.format("Not altering state of %s in cluster state because new state is too similar: %s", node, currentState.getTextualDifference(reportedState)));
        }
        this.stateMayHaveChanged = true;
    }

    public void handleNewNode(NodeInfo node) {
        String message = "Found new node " + node + " in slobrok at " + node.getRpcAddress();
        this.eventLog.add(NodeEvent.forBaseline(node, message, NodeEvent.Type.REPORTED, this.timer.getCurrentTimeInMillis()), this.isMaster);
    }

    public void handleMissingNode(ClusterState currentClusterState, NodeInfo node, NodeListener nodeListener) {
        long timeNow = this.timer.getCurrentTimeInMillis();
        if (node.getLatestNodeStateRequestTime() != null) {
            this.eventLog.add(NodeEvent.forBaseline(node, "Node is no longer in slobrok, but we still have a pending state request.", NodeEvent.Type.REPORTED, timeNow), this.isMaster);
        } else {
            this.eventLog.add(NodeEvent.forBaseline(node, "Node is no longer in slobrok. No pending state request to node.", NodeEvent.Type.REPORTED, timeNow), this.isMaster);
        }
        if (node.getReportedState().getState().equals((Object)State.STOPPING)) {
            log.log(Level.FINE, () -> "Node " + node.getNode() + " is no longer in slobrok. Was in stopping state, so assuming it has shut down normally. Setting node down");
            NodeState ns = node.getReportedState().clone();
            ns.setState(State.DOWN);
            this.handleNewReportedNodeState(currentClusterState, node, ns.clone(), nodeListener);
        } else {
            log.log(Level.FINE, () -> "Node " + node.getNode() + " no longer in slobrok was in state " + node.getReportedState() + ". Waiting to see if it reappears in slobrok");
        }
        this.stateMayHaveChanged = true;
    }

    public void proposeNewNodeState(ClusterState currentClusterState, NodeInfo node, NodeState proposedState) {
        NodeState currentState = currentClusterState.getNodeState(node.getNode());
        if (currentState.getState().equals((Object)proposedState.getState())) {
            return;
        }
        this.stateMayHaveChanged = true;
        log.log(Level.FINE, () -> String.format("Got new wanted nodestate for %s: %s", node, currentState.getTextualDifference(proposedState)));
        assert (proposedState.getState().validWantedNodeState(node.getNode().getType()));
        long timeNow = this.timer.getCurrentTimeInMillis();
        NodeState currentReported = node.getReportedState();
        if (proposedState.above(currentReported)) {
            this.eventLog.add(NodeEvent.forBaseline(node, String.format("Wanted state %s, but we cannot force node into that state yet as it is currently in %s", proposedState, currentReported), NodeEvent.Type.REPORTED, timeNow), this.isMaster);
            return;
        }
        if (!proposedState.similarTo((Object)currentState)) {
            this.eventLog.add(NodeEvent.forBaseline(node, String.format("Node state set to %s.", proposedState), NodeEvent.Type.WANTED, timeNow), this.isMaster);
        }
    }

    public void handleNewRpcAddress(NodeInfo node) {
        String message = "Node " + node + " has a new address in slobrok: " + node.getRpcAddress();
        this.eventLog.add(NodeEvent.forBaseline(node, message, NodeEvent.Type.REPORTED, this.timer.getCurrentTimeInMillis()), this.isMaster);
    }

    public void handleReturnedRpcAddress(NodeInfo node) {
        String message = "Node got back into slobrok with same address as before: " + node.getRpcAddress();
        this.eventLog.add(NodeEvent.forBaseline(node, message, NodeEvent.Type.REPORTED, this.timer.getCurrentTimeInMillis()), this.isMaster);
    }

    void reconfigureFromOptions(FleetControllerOptions options) {
        this.setMaxPrematureCrashes(options.maxPrematureCrashes());
        this.setStableStateTimePeriod(options.stableStateTimePeriod());
        this.setMaxInitProgressTime(options.maxInitProgressTime());
        this.setMaxSlobrokDisconnectGracePeriod(options.maxSlobrokDisconnectGracePeriod());
        this.setMaxTransitionTime(options.maxTransitionTime());
    }

    public boolean watchTimers(ContentCluster cluster, ClusterState currentClusterState, NodeListener nodeListener) {
        boolean triggeredAnyTimers = false;
        long currentTime = this.timer.getCurrentTimeInMillis();
        for (NodeInfo node : cluster.getNodeInfos()) {
            triggeredAnyTimers |= this.handleTimeDependentOpsForNode(currentClusterState, nodeListener, currentTime, node);
        }
        if (triggeredAnyTimers) {
            this.stateMayHaveChanged = true;
        }
        return triggeredAnyTimers;
    }

    private boolean handleTimeDependentOpsForNode(ClusterState currentClusterState, NodeListener nodeListener, long currentTime, NodeInfo node) {
        NodeState currentStateInSystem = currentClusterState.getNodeState(node.getNode());
        NodeState lastReportedState = node.getReportedState();
        boolean triggeredAnyTimers = this.reportDownIfOutdatedSlobrokNode(currentClusterState, nodeListener, currentTime, node, lastReportedState);
        if (this.nodeStillUnavailableAfterTransitionTimeExceeded(currentTime, node, currentStateInSystem, lastReportedState)) {
            triggeredAnyTimers = true;
        }
        if (this.nodeInitProgressHasTimedOut(currentTime, node, currentStateInSystem, lastReportedState)) {
            this.eventLog.add(NodeEvent.forBaseline(node, String.format("%d milliseconds without initialize progress. Marking node down. Premature crash count is now %d.", currentTime - node.getInitProgressTime(), node.getPrematureCrashCount() + 1), NodeEvent.Type.CURRENT, currentTime), this.isMaster);
            this.handlePrematureCrash(node, nodeListener);
            triggeredAnyTimers = true;
        }
        if (this.mayResetCrashCounterOnStableUpNode(currentTime, node, lastReportedState)) {
            node.setPrematureCrashCount(0);
            log.log(Level.FINE, () -> "Resetting premature crash count on node " + node + " as it has been up for a long time.");
            triggeredAnyTimers = true;
        } else if (this.mayResetCrashCounterOnStableDownNode(currentTime, node, lastReportedState)) {
            node.setPrematureCrashCount(0);
            log.log(Level.FINE, () -> "Resetting premature crash count on node " + node + " as it has been down for a long time.");
            triggeredAnyTimers = true;
        }
        return triggeredAnyTimers;
    }

    private boolean nodeInitProgressHasTimedOut(long currentTime, NodeInfo node, NodeState currentStateInSystem, NodeState lastReportedState) {
        return !currentStateInSystem.getState().equals((Object)State.DOWN) && node.getWantedState().above(new NodeState(node.getNode().getType(), State.DOWN)) && lastReportedState.getState().equals((Object)State.INITIALIZING) && this.maxInitProgressTime != 0 && node.getInitProgressTime() + (long)this.maxInitProgressTime <= currentTime && node.getNode().getType().equals((Object)NodeType.STORAGE);
    }

    private boolean mayResetCrashCounterOnStableDownNode(long currentTime, NodeInfo node, NodeState lastReportedState) {
        return node.getDownStableStateTime() + this.stableStateTimePeriod <= currentTime && lastReportedState.getState().equals((Object)State.DOWN) && node.getPrematureCrashCount() <= this.maxPrematureCrashes && node.getPrematureCrashCount() != 0;
    }

    private boolean mayResetCrashCounterOnStableUpNode(long currentTime, NodeInfo node, NodeState lastReportedState) {
        return node.getUpStableStateTime() + this.stableStateTimePeriod <= currentTime && lastReportedState.getState().equals((Object)State.UP) && node.getPrematureCrashCount() <= this.maxPrematureCrashes && node.getPrematureCrashCount() != 0;
    }

    private boolean nodeStillUnavailableAfterTransitionTimeExceeded(long currentTime, NodeInfo node, NodeState currentStateInSystem, NodeState lastReportedState) {
        return currentStateInSystem.getState().equals((Object)State.MAINTENANCE) && node.getWantedState().above(new NodeState(node.getNode().getType(), State.DOWN)) && (lastReportedState.getState().equals((Object)State.DOWN) || node.isNotInSlobrok()) && node.getTransitionTime() + (long)this.maxTransitionTime.get(node.getNode().getType()).intValue() < currentTime;
    }

    private boolean reportDownIfOutdatedSlobrokNode(ClusterState currentClusterState, NodeListener nodeListener, long currentTime, NodeInfo node, NodeState lastReportedState) {
        if (node.isNotInSlobrok() && !lastReportedState.getState().equals((Object)State.DOWN) && node.lastSeenInSlobrok() + (long)this.maxSlobrokDisconnectGracePeriod <= currentTime) {
            String desc = String.format("Set node down as it has been out of slobrok for %d ms which is more than the max limit of %d ms.", currentTime - node.lastSeenInSlobrok(), this.maxSlobrokDisconnectGracePeriod);
            node.abortCurrentNodeStateRequests();
            NodeState state = lastReportedState.clone();
            state.setState(State.DOWN);
            if (!state.hasDescription()) {
                state.setDescription(desc);
            }
            this.eventLog.add(NodeEvent.forBaseline(node, desc, NodeEvent.Type.CURRENT, currentTime), this.isMaster);
            this.handleNewReportedNodeState(currentClusterState, node, state.clone(), nodeListener);
            node.setReportedState(state, currentTime);
            return true;
        }
        return false;
    }

    private boolean isNotControlledShutdown(NodeState state) {
        return !this.isControlledShutdown(state);
    }

    private boolean isControlledShutdown(NodeState state) {
        return state.getState() == State.STOPPING && List.of("Received signal 15 (SIGTERM - Termination signal)", "controlled shutdown").contains(state.getDescription());
    }

    private void updateNodeInfoFromReportedState(NodeInfo node, NodeState currentState, NodeState reportedState, NodeListener nodeListener) {
        long timeNow = this.timer.getCurrentTimeInMillis();
        log.log(Level.FINE, () -> String.format("Finding new cluster state entry for %s switching state %s", node, currentState.getTextualDifference(reportedState)));
        if (this.handleReportedNodeCrashEdge(node, currentState, reportedState, nodeListener, timeNow)) {
            return;
        }
        if (this.initializationProgressHasIncreased(currentState, reportedState)) {
            node.setInitProgressTime(timeNow);
            log.log(Level.FINEST, () -> "Reset initialize timer on " + node + " to " + node.getInitProgressTime());
        }
        if (this.handleImplicitCrashEdgeFromReverseInitProgress(node, currentState, reportedState, nodeListener, timeNow)) {
            return;
        }
        this.markNodeUnstableIfDownEdgeDuringInit(node, currentState, reportedState, nodeListener, timeNow);
    }

    private void markNodeUnstableIfDownEdgeDuringInit(NodeInfo node, NodeState currentState, NodeState reportedState, NodeListener nodeListener, long timeNow) {
        if (currentState.getState().equals((Object)State.INITIALIZING) && reportedState.getState().oneOf("ds") && this.isNotControlledShutdown(reportedState)) {
            this.eventLog.add(NodeEvent.forBaseline(node, String.format("Stop or crash during initialization. Premature crash count is now %d.", node.getPrematureCrashCount() + 1), NodeEvent.Type.CURRENT, timeNow), this.isMaster);
            this.handlePrematureCrash(node, nodeListener);
        }
    }

    private boolean handleImplicitCrashEdgeFromReverseInitProgress(NodeInfo node, NodeState currentState, NodeState reportedState, NodeListener nodeListener, long timeNow) {
        if (currentState.getState().equals((Object)State.INITIALIZING) && reportedState.getState().equals((Object)State.INITIALIZING) && reportedState.getInitProgress() < currentState.getInitProgress()) {
            this.eventLog.add(NodeEvent.forBaseline(node, String.format("Stop or crash during initialization detected from reverse initializing progress. Progress was %g but is now %g. Premature crash count is now %d.", currentState.getInitProgress(), reportedState.getInitProgress(), node.getPrematureCrashCount() + 1), NodeEvent.Type.CURRENT, timeNow), this.isMaster);
            node.setRecentlyObservedUnstableDuringInit(true);
            this.handlePrematureCrash(node, nodeListener);
            return true;
        }
        return false;
    }

    private boolean handleReportedNodeCrashEdge(NodeInfo node, NodeState currentState, NodeState reportedState, NodeListener nodeListener, long timeNow) {
        if (this.nodeUpToDownEdge(node, currentState, reportedState)) {
            node.setTransitionTime(timeNow);
            if (node.getUpStableStateTime() + this.stableStateTimePeriod > timeNow && this.isNotControlledShutdown(reportedState)) {
                log.log(Level.FINE, () -> "Stable state: " + node.getUpStableStateTime() + " + " + this.stableStateTimePeriod + " > " + timeNow);
                this.eventLog.add(NodeEvent.forBaseline(node, String.format("Stopped or possibly crashed after %d ms, which is before stable state time period. Premature crash count is now %d.", timeNow - node.getUpStableStateTime(), node.getPrematureCrashCount() + 1), NodeEvent.Type.CURRENT, timeNow), this.isMaster);
                return this.handlePrematureCrash(node, nodeListener);
            }
        }
        return false;
    }

    private boolean initializationProgressHasIncreased(NodeState currentState, NodeState reportedState) {
        return reportedState.getState().equals((Object)State.INITIALIZING) && (!currentState.getState().equals((Object)State.INITIALIZING) || reportedState.getInitProgress() > currentState.getInitProgress());
    }

    private boolean nodeUpToDownEdge(NodeInfo node, NodeState currentState, NodeState reportedState) {
        return currentState.getState().oneOf("ur") && reportedState.getState().oneOf("dis") && (node.getWantedState().getState().equals((Object)State.RETIRED) || !reportedState.getState().equals((Object)State.INITIALIZING));
    }

    private boolean handlePrematureCrash(NodeInfo node, NodeListener changeListener) {
        node.setPrematureCrashCount(node.getPrematureCrashCount() + 1);
        if (node.getPrematureCrashCount() > this.maxPrematureCrashes) {
            NodeState wantedState = new NodeState(node.getNode().getType(), State.DOWN).setDescription("Disabled by fleet controller as it prematurely shut down " + node.getPrematureCrashCount() + " times in a row");
            NodeState oldState = node.getWantedState();
            node.setWantedState(wantedState);
            if (!oldState.equals((Object)wantedState)) {
                changeListener.handleNewWantedNodeState(node, wantedState);
            }
            return true;
        }
        return false;
    }
}

