/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.vespa.clustercontroller.core;

import com.yahoo.lang.MutableBoolean;
import com.yahoo.lang.SettableOptional;
import com.yahoo.vdslib.distribution.ConfiguredNode;
import com.yahoo.vdslib.distribution.Group;
import com.yahoo.vdslib.state.ClusterState;
import com.yahoo.vdslib.state.Node;
import com.yahoo.vdslib.state.NodeState;
import com.yahoo.vdslib.state.NodeType;
import com.yahoo.vdslib.state.State;
import com.yahoo.vespa.clustercontroller.core.ClusterInfo;
import com.yahoo.vespa.clustercontroller.core.DistributorNodeInfo;
import com.yahoo.vespa.clustercontroller.core.HierarchicalGroupVisiting;
import com.yahoo.vespa.clustercontroller.core.NodeInfo;
import com.yahoo.vespa.clustercontroller.core.StorageNodeInfo;
import com.yahoo.vespa.clustercontroller.core.hostinfo.HostInfo;
import com.yahoo.vespa.clustercontroller.core.hostinfo.Metrics;
import com.yahoo.vespa.clustercontroller.core.hostinfo.StorageNode;
import com.yahoo.vespa.clustercontroller.utils.staterestapi.requests.SetUnitStateRequest;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

public class NodeStateChangeChecker {
    public static final String BUCKETS_METRIC_NAME = "vds.datastored.bucket_space.buckets_total";
    public static final Map<String, String> BUCKETS_METRIC_DIMENSIONS = Map.of("bucketSpace", "default");
    private final int requiredRedundancy;
    private final HierarchicalGroupVisiting groupVisiting;
    private final ClusterInfo clusterInfo;
    private final boolean inMoratorium;

    public NodeStateChangeChecker(int requiredRedundancy, HierarchicalGroupVisiting groupVisiting, ClusterInfo clusterInfo, boolean inMoratorium) {
        this.requiredRedundancy = requiredRedundancy;
        this.groupVisiting = groupVisiting;
        this.clusterInfo = clusterInfo;
        this.inMoratorium = inMoratorium;
    }

    public Result evaluateTransition(Node node, ClusterState clusterState, SetUnitStateRequest.Condition condition, NodeState oldWantedState, NodeState newWantedState) {
        if (condition == SetUnitStateRequest.Condition.FORCE) {
            return Result.allowSettingOfWantedState();
        }
        if (this.inMoratorium) {
            return Result.createDisallowed("Master cluster controller is bootstrapping and in moratorium");
        }
        if (condition != SetUnitStateRequest.Condition.SAFE) {
            return Result.createDisallowed("Condition not implemented: " + condition.name());
        }
        if (node.getType() != NodeType.STORAGE) {
            return Result.createDisallowed("Safe-set of node state is only supported for storage nodes! Requested node type: " + node.getType().toString());
        }
        StorageNodeInfo nodeInfo = this.clusterInfo.getStorageNodeInfo(node.getIndex());
        if (nodeInfo == null) {
            return Result.createDisallowed("Unknown node " + node);
        }
        if (newWantedState.getState().equals((Object)oldWantedState.getState()) && Objects.equals(newWantedState.getDescription(), oldWantedState.getDescription())) {
            return Result.createAlreadySet();
        }
        return switch (newWantedState.getState()) {
            case State.UP -> this.canSetStateUp(nodeInfo, oldWantedState);
            case State.MAINTENANCE -> this.canSetStateMaintenanceTemporarily(nodeInfo, clusterState, newWantedState.getDescription());
            case State.DOWN -> this.canSetStateDownPermanently(nodeInfo, clusterState, newWantedState.getDescription());
            default -> Result.createDisallowed("Destination node state unsupported in safe mode: " + newWantedState);
        };
    }

    private Result canSetStateDownPermanently(NodeInfo nodeInfo, ClusterState clusterState, String newDescription) {
        NodeState oldWantedState = nodeInfo.getUserWantedState();
        if (oldWantedState.getState() != State.UP && !oldWantedState.getDescription().equals(newDescription)) {
            return Result.createDisallowed("A conflicting wanted state is already set: " + oldWantedState.getState() + ": " + oldWantedState.getDescription());
        }
        State reportedState = nodeInfo.getReportedState().getState();
        if (reportedState != State.UP) {
            return Result.createDisallowed("Reported state (" + reportedState + ") is not UP, so no bucket data is available");
        }
        State currentState = clusterState.getNodeState(nodeInfo.getNode()).getState();
        if (currentState != State.RETIRED) {
            return Result.createDisallowed("Only retired nodes are allowed to be set to DOWN in safe mode - is " + currentState);
        }
        HostInfo hostInfo = nodeInfo.getHostInfo();
        Integer hostInfoNodeVersion = hostInfo.getClusterStateVersionOrNull();
        int clusterControllerVersion = clusterState.getVersion();
        if (hostInfoNodeVersion == null || hostInfoNodeVersion != clusterControllerVersion) {
            return Result.createDisallowed("Cluster controller at version " + clusterControllerVersion + " got info for storage node " + nodeInfo.getNodeIndex() + " at a different version " + hostInfoNodeVersion);
        }
        Optional<Metrics.Value> bucketsMetric = hostInfo.getMetrics().getValueAt(BUCKETS_METRIC_NAME, BUCKETS_METRIC_DIMENSIONS);
        if (bucketsMetric.isEmpty() || bucketsMetric.get().getLast() == null) {
            return Result.createDisallowed("Missing last value of the vds.datastored.bucket_space.buckets_total metric for storage node " + nodeInfo.getNodeIndex());
        }
        long lastBuckets = bucketsMetric.get().getLast();
        if (lastBuckets > 0L) {
            return Result.createDisallowed("The storage node manages " + lastBuckets + " buckets");
        }
        return Result.allowSettingOfWantedState();
    }

    private Result canSetStateUp(NodeInfo nodeInfo, NodeState oldWantedState) {
        if (oldWantedState.getState() == State.UP) {
            return Result.createAlreadySet();
        }
        if (nodeInfo.getReportedState().getState() != State.UP) {
            return Result.createDisallowed("Refuse to set wanted state to UP, since the reported state is not UP (" + nodeInfo.getReportedState().getState() + ")");
        }
        return Result.allowSettingOfWantedState();
    }

    private Result canSetStateMaintenanceTemporarily(StorageNodeInfo nodeInfo, ClusterState clusterState, String newDescription) {
        NodeState oldWantedState = nodeInfo.getUserWantedState();
        if (oldWantedState.getState() != State.UP && !oldWantedState.getDescription().equals(newDescription)) {
            return Result.createDisallowed("A conflicting wanted state is already set: " + oldWantedState.getState() + ": " + oldWantedState.getDescription());
        }
        Result otherGroupCheck = this.anotherNodeInAnotherGroupHasWantedState(nodeInfo);
        if (!otherGroupCheck.settingWantedStateIsAllowed()) {
            return otherGroupCheck;
        }
        if (clusterState.getNodeState(nodeInfo.getNode()).getState() == State.DOWN) {
            return Result.allowSettingOfWantedState();
        }
        if (this.anotherNodeInGroupAlreadyAllowed(nodeInfo, newDescription)) {
            return Result.allowSettingOfWantedState();
        }
        Result allNodesAreUpCheck = this.checkAllNodesAreUp(clusterState);
        if (!allNodesAreUpCheck.settingWantedStateIsAllowed()) {
            return allNodesAreUpCheck;
        }
        Result checkDistributorsResult = this.checkDistributors(nodeInfo.getNode(), clusterState.getVersion());
        if (!checkDistributorsResult.settingWantedStateIsAllowed()) {
            return checkDistributorsResult;
        }
        return Result.allowSettingOfWantedState();
    }

    private Result anotherNodeInAnotherGroupHasWantedState(StorageNodeInfo nodeInfo) {
        if (this.groupVisiting.isHierarchical()) {
            SettableOptional anotherNodeHasWantedState = new SettableOptional();
            this.groupVisiting.visit(group -> {
                Result result;
                if (!NodeStateChangeChecker.groupContainsNode(group, nodeInfo.getNode()) && !(result = this.otherNodeInGroupHasWantedState(group)).settingWantedStateIsAllowed()) {
                    anotherNodeHasWantedState.set((Object)result);
                    return false;
                }
                return true;
            });
            return anotherNodeHasWantedState.asOptional().orElseGet(Result::allowSettingOfWantedState);
        }
        return this.otherNodeHasWantedState(nodeInfo);
    }

    private Result otherNodeInGroupHasWantedState(Group group) {
        for (ConfiguredNode configuredNode : group.getNodes()) {
            int index = configuredNode.index();
            StorageNodeInfo storageNodeInfo = this.clusterInfo.getStorageNodeInfo(index);
            if (storageNodeInfo == null) continue;
            State storageNodeWantedState = storageNodeInfo.getUserWantedState().getState();
            if (storageNodeWantedState != State.UP) {
                return Result.createDisallowed("At most one group can have wanted state: Other storage node " + index + " in group " + group.getIndex() + " has wanted state " + storageNodeWantedState);
            }
            State distributorWantedState = this.clusterInfo.getDistributorNodeInfo(index).getUserWantedState().getState();
            if (distributorWantedState == State.UP) continue;
            return Result.createDisallowed("At most one group can have wanted state: Other distributor " + index + " in group " + group.getIndex() + " has wanted state " + distributorWantedState);
        }
        return Result.allowSettingOfWantedState();
    }

    private Result otherNodeHasWantedState(StorageNodeInfo nodeInfo) {
        for (ConfiguredNode configuredNode : this.clusterInfo.getConfiguredNodes().values()) {
            int index = configuredNode.index();
            if (index == nodeInfo.getNodeIndex()) continue;
            State storageNodeWantedState = this.clusterInfo.getStorageNodeInfo(index).getUserWantedState().getState();
            if (storageNodeWantedState != State.UP) {
                return Result.createDisallowed("At most one node can have a wanted state when #groups = 1: Other storage node " + index + " has wanted state " + storageNodeWantedState);
            }
            State distributorWantedState = this.clusterInfo.getDistributorNodeInfo(index).getUserWantedState().getState();
            if (distributorWantedState == State.UP) continue;
            return Result.createDisallowed("At most one node can have a wanted state when #groups = 1: Other distributor " + index + " has wanted state " + distributorWantedState);
        }
        return Result.allowSettingOfWantedState();
    }

    private boolean anotherNodeInGroupAlreadyAllowed(StorageNodeInfo nodeInfo, String newDescription) {
        MutableBoolean alreadyAllowed = new MutableBoolean(false);
        this.groupVisiting.visit(group -> {
            if (!NodeStateChangeChecker.groupContainsNode(group, nodeInfo.getNode())) {
                return true;
            }
            alreadyAllowed.set(this.anotherNodeInGroupAlreadyAllowed(group, nodeInfo.getNode(), newDescription));
            return false;
        });
        return alreadyAllowed.get();
    }

    private boolean anotherNodeInGroupAlreadyAllowed(Group group, Node node, String newDescription) {
        return group.getNodes().stream().filter(configuredNode -> configuredNode.index() != node.getIndex()).map(configuredNode -> this.clusterInfo.getStorageNodeInfo(configuredNode.index())).filter(Objects::nonNull).map(NodeInfo::getUserWantedState).anyMatch(userWantedState -> userWantedState.getState() == State.MAINTENANCE && Objects.equals(userWantedState.getDescription(), newDescription));
    }

    private static boolean groupContainsNode(Group group, Node node) {
        for (ConfiguredNode configuredNode : group.getNodes()) {
            if (configuredNode.index() != node.getIndex()) continue;
            return true;
        }
        return false;
    }

    private Result checkAllNodesAreUp(ClusterState clusterState) {
        State state;
        State wantedState;
        for (NodeInfo nodeInfo : this.clusterInfo.getStorageNodeInfos()) {
            wantedState = nodeInfo.getUserWantedState().getState();
            if (wantedState != State.UP && wantedState != State.RETIRED) {
                return Result.createDisallowed("Another storage node wants state " + wantedState.toString().toUpperCase() + ": " + nodeInfo.getNodeIndex());
            }
            state = clusterState.getNodeState(nodeInfo.getNode()).getState();
            if (state == State.UP || state == State.RETIRED) continue;
            return Result.createDisallowed("Another storage node has state " + state.toString().toUpperCase() + ": " + nodeInfo.getNodeIndex());
        }
        for (NodeInfo nodeInfo : this.clusterInfo.getDistributorNodeInfos()) {
            wantedState = nodeInfo.getUserWantedState().getState();
            if (wantedState != State.UP && wantedState != State.RETIRED) {
                return Result.createDisallowed("Another distributor wants state " + wantedState.toString().toUpperCase() + ": " + nodeInfo.getNodeIndex());
            }
            state = clusterState.getNodeState(nodeInfo.getNode()).getState();
            if (state == State.UP || state == State.RETIRED) continue;
            return Result.createDisallowed("Another distributor has state " + state.toString().toUpperCase() + ": " + nodeInfo.getNodeIndex());
        }
        return Result.allowSettingOfWantedState();
    }

    private Result checkStorageNodesForDistributor(DistributorNodeInfo distributorNodeInfo, List<StorageNode> storageNodes, Node node) {
        for (StorageNode storageNode : storageNodes) {
            if (storageNode.getIndex().intValue() != node.getIndex()) continue;
            Integer minReplication = storageNode.getMinCurrentReplicationFactorOrNull();
            if (minReplication != null && minReplication < this.requiredRedundancy) {
                return Result.createDisallowed("Distributor " + distributorNodeInfo.getNodeIndex() + " says storage node " + node.getIndex() + " has buckets with redundancy as low as " + storageNode.getMinCurrentReplicationFactorOrNull() + ", but we require at least " + this.requiredRedundancy);
            }
            return Result.allowSettingOfWantedState();
        }
        return Result.allowSettingOfWantedState();
    }

    private Result checkDistributors(Node node, int clusterStateVersion) {
        if (this.clusterInfo.getDistributorNodeInfos().isEmpty()) {
            return Result.createDisallowed("Not aware of any distributors, probably not safe to upgrade?");
        }
        for (DistributorNodeInfo distributorNodeInfo : this.clusterInfo.getDistributorNodeInfos()) {
            Integer distributorClusterStateVersion = distributorNodeInfo.getHostInfo().getClusterStateVersionOrNull();
            if (distributorClusterStateVersion == null) {
                return Result.createDisallowed("Distributor node " + distributorNodeInfo.getNodeIndex() + " has not reported any cluster state version yet.");
            }
            if (distributorClusterStateVersion != clusterStateVersion) {
                return Result.createDisallowed("Distributor node " + distributorNodeInfo.getNodeIndex() + " does not report same version (" + distributorNodeInfo.getHostInfo().getClusterStateVersionOrNull() + ") as fleetcontroller (" + clusterStateVersion + ")");
            }
            List<StorageNode> storageNodes = distributorNodeInfo.getHostInfo().getDistributor().getStorageNodes();
            Result storageNodesResult = this.checkStorageNodesForDistributor(distributorNodeInfo, storageNodes, node);
            if (storageNodesResult.settingWantedStateIsAllowed()) continue;
            return storageNodesResult;
        }
        return Result.allowSettingOfWantedState();
    }

    public static class Result {
        private final Action action;
        private final String reason;

        private Result(Action action, String reason) {
            this.action = action;
            this.reason = reason;
        }

        public static Result createDisallowed(String reason) {
            return new Result(Action.DISALLOWED, reason);
        }

        public static Result allowSettingOfWantedState() {
            return new Result(Action.MUST_SET_WANTED_STATE, "Preconditions fulfilled and new state different");
        }

        public static Result createAlreadySet() {
            return new Result(Action.ALREADY_SET, "Basic preconditions fulfilled and new state is already effective");
        }

        public boolean settingWantedStateIsAllowed() {
            return this.action == Action.MUST_SET_WANTED_STATE;
        }

        public boolean wantedStateAlreadySet() {
            return this.action == Action.ALREADY_SET;
        }

        public String getReason() {
            return this.reason;
        }

        public String toString() {
            return "action " + this.action + ": " + this.reason;
        }

        public static enum Action {
            MUST_SET_WANTED_STATE,
            ALREADY_SET,
            DISALLOWED;

        }
    }
}

