/*
 * Copyright (c) 2016 riontech-xten
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.xtensolution.support.async;

import android.os.AsyncTask;
import android.os.Handler;
import android.os.Looper;
import android.util.Log;

/**
 * A abstract {@link AsyncTask} subclass.
 * To perform the background process use this {@link BaseAsyncTask} class.
 * Contain method return the result after the completed
 * the background process
 */
public abstract class BaseAsyncTask<Params, Progress, Result> extends AsyncTask<Params, Progress, Result> {
    // to identify the class name when on debug mode
    private static final String TAG = BaseAsyncTask.class.getSimpleName();

    // Handle the Exception on UI mode
    private static final Handler mainThreadHandler = new Handler(Looper.getMainLooper());
    // to check exception occurred
    private boolean isExceptionOccurred;

    @Override
    protected final Result doInBackground(Params... params) {
        try {
            return performInBackground(params);
        } catch (final Exception e) {
            isExceptionOccurred = true;
            mainThreadHandler.post(new Runnable() {
                @Override
                public void run() {
                    onException(e);
                }
            });
            return null;
        }
    }

    @Override
    protected final void onPostExecute(Result result) {
        if (!isExceptionOccurred) {
            onResult(result);
        }
    }

    /**
     * A factory method perform background process,
     * must override to extended class.
     *
     * @return result
     */
    public abstract Result performInBackground(Params... params) throws Exception;

    /**
     * A factory method return the result after background
     * process completed, must override
     *
     * @return void
     */
    public abstract void onResult(Result result);

    /**
     * A factory method handle the exception on UI mode
     * @return void
     */
    public void onException(Exception e) {
        Log.w(TAG, e);
    }
}

