/*
 * Decompiled with CFR 0.152.
 */
package com.xceptance.xlt.report.scorecard;

import com.google.common.io.Files;
import com.thoughtworks.xstream.XStream;
import com.thoughtworks.xstream.io.HierarchicalStreamDriver;
import com.xceptance.common.util.ParameterCheckUtils;
import com.xceptance.xlt.report.scorecard.Configuration;
import com.xceptance.xlt.report.scorecard.GroupDefinition;
import com.xceptance.xlt.report.scorecard.RatingDefinition;
import com.xceptance.xlt.report.scorecard.RuleDefinition;
import com.xceptance.xlt.report.scorecard.Scorecard;
import com.xceptance.xlt.report.scorecard.SelectorDefinition;
import com.xceptance.xlt.report.scorecard.Status;
import com.xceptance.xlt.report.scorecard.ValidationException;
import com.xceptance.xlt.report.util.xstream.SanitizingDomDriver;
import dev.harrel.jsonschema.JsonNodeFactory;
import dev.harrel.jsonschema.Validator;
import dev.harrel.jsonschema.ValidatorFactory;
import dev.harrel.jsonschema.providers.OrgJsonNode;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.Writer;
import java.net.URI;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;
import net.sf.saxon.s9api.Processor;
import net.sf.saxon.s9api.SaxonApiException;
import net.sf.saxon.s9api.UnprefixedElementMatchingPolicy;
import net.sf.saxon.s9api.XPathCompiler;
import net.sf.saxon.s9api.XdmItem;
import net.sf.saxon.s9api.XdmNode;
import net.sf.saxon.s9api.XdmValue;
import org.apache.commons.lang3.StringUtils;
import org.json.JSONException;
import org.json.JSONObject;
import org.json.JSONTokener;

public class Evaluator {
    private static final String SCHEMA_RESOURCE_PATH = "configuration-schema.json";
    private final File configFile;
    private final Processor processor;

    public Evaluator(File configFile) {
        ParameterCheckUtils.isReadableFile(configFile, "configFile");
        this.configFile = configFile;
        this.processor = new Processor(false);
    }

    public Scorecard evaluate(File documentFile) {
        ParameterCheckUtils.isNotNull(documentFile, "documentFile");
        try {
            Configuration config = this.parseConfiguration();
            return this.doEvaluate(config, documentFile);
        }
        catch (Exception ex) {
            return Scorecard.error(ex);
        }
    }

    public void writeScorecardToFile(Scorecard scorecard, File outputFile) throws IOException {
        try (OutputStreamWriter osw = new OutputStreamWriter((OutputStream)new FileOutputStream(outputFile), "UTF-8");){
            this.writeScorecard(scorecard, osw);
        }
    }

    public void writeScorecard(Scorecard scorecard, Writer writer) throws IOException {
        writer.write("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n");
        XStream xstream = new XStream((HierarchicalStreamDriver)new SanitizingDomDriver());
        xstream.autodetectAnnotations(true);
        xstream.aliasSystemAttribute(null, "class");
        xstream.setMode(1001);
        xstream.toXML((Object)scorecard, writer);
    }

    protected Configuration parseConfiguration() throws ValidationException, FileNotFoundException, IOException {
        JSONObject configJSON;
        JSONObject schemaJSON;
        try (InputStream is = this.getClass().getResourceAsStream(SCHEMA_RESOURCE_PATH);){
            schemaJSON = new JSONObject(new JSONTokener(is));
        }
        catch (JSONException je) {
            throw new ValidationException("Failed to parse JSON schema file", je);
        }
        try (BufferedReader reader = Files.newReader((File)this.configFile, (Charset)StandardCharsets.UTF_8);){
            configJSON = new JSONObject(new JSONTokener((Reader)reader));
        }
        catch (JSONException je) {
            throw new ValidationException("Could not parse configuration file '" + this.configFile.getName() + "' as JSON", je);
        }
        Validator validator = new ValidatorFactory().withJsonNodeFactory((JsonNodeFactory)new OrgJsonNode.Factory()).createValidator();
        URI schemaURI = validator.registerSchema((Object)schemaJSON);
        Validator.Result validationResult = validator.validate(schemaURI, (Object)configJSON);
        if (!validationResult.isValid()) {
            throw new ValidationException("Configuration file '" + this.configFile.getName() + "' is malformed -> " + validationResult.getErrors().stream().map(e -> e.getInstanceLocation() + ": " + e.getError()).collect(Collectors.joining(", ")));
        }
        try {
            return Configuration.fromJSON(configJSON);
        }
        catch (Exception e2) {
            throw new ValidationException("Configuration file '" + this.configFile.getName() + "' is malformed", e2);
        }
    }

    protected Scorecard doEvaluate(Configuration config, File documentFile) throws SaxonApiException {
        boolean evaluationFailed;
        XdmNode docNode = this.processor.newDocumentBuilder().build(documentFile);
        XPathCompiler xpathCompiler = this.processor.newXPathCompiler();
        xpathCompiler.setUnprefixedElementMatchingPolicy(UnprefixedElementMatchingPolicy.DEFAULT_NAMESPACE);
        xpathCompiler.setCaching(true);
        Integer points = 0;
        Integer totalPoints = 0;
        boolean testFailed = false;
        Scorecard scorecard = new Scorecard(config);
        Scorecard.Result result = scorecard.result;
        ArrayList<Scorecard.Group> erroneousGroups = new ArrayList<Scorecard.Group>();
        for (GroupDefinition groupDef : config.getGroups()) {
            Scorecard.Group group = new Scorecard.Group(groupDef);
            for (String ruleId : groupDef.getRuleIds()) {
                RuleDefinition ruleDef = config.getRule(ruleId);
                Scorecard.Rule rule = new Scorecard.Rule(ruleDef, groupDef.isEnabled());
                this.evaluateRule(rule, xpathCompiler, docNode, config::getSelector);
                group.addRule(rule);
            }
            testFailed = this.conclude(group) | testFailed;
            points = points + group.getPoints();
            totalPoints = totalPoints + group.getTotalPoints();
            if (group.getStatus().isError()) {
                erroneousGroups.add(group);
            }
            result.addGroup(group);
        }
        boolean bl = evaluationFailed = !erroneousGroups.isEmpty();
        if (evaluationFailed) {
            String errorMessagesJoined;
            if (StringUtils.isBlank((CharSequence)result.getError()) && StringUtils.isNotBlank((CharSequence)(errorMessagesJoined = erroneousGroups.stream().flatMap(g -> g.getRules().stream()).filter(r -> r.getStatus().isError() && StringUtils.isNotBlank((CharSequence)r.getMessage())).map(r -> String.format("Error evaluating rule '%s': %s", r.getId(), r.getMessage())).collect(Collectors.joining("\n\n"))))) {
                result.setError(errorMessagesJoined);
            }
        } else {
            result.setPoints(points);
            result.setTotalPoints(totalPoints);
            double pointsPercentage = Evaluator.getPercentage(points, totalPoints);
            String rating = null;
            for (RatingDefinition ratingDef : config.getRatings()) {
                if (!ratingDef.isEnabled() || !(pointsPercentage <= ratingDef.getValue())) continue;
                rating = ratingDef.getId();
                testFailed = testFailed || ratingDef.isFailsTest();
                break;
            }
            result.setTestFailed(testFailed);
            result.setPointsPercentage(pointsPercentage);
            result.setRating(rating);
        }
        return scorecard;
    }

    private void evaluateRule(Scorecard.Rule rule, XPathCompiler compiler, XdmNode document, Function<String, SelectorDefinition> selectorLookup) {
        for (RuleDefinition.Check check : rule.getDefinition().getChecks()) {
            Scorecard.Rule.Check ruleCheck = new Scorecard.Rule.Check(check, rule.isEnabled());
            if (ruleCheck.isEnabled()) {
                this.evaluateRuleCheck(ruleCheck, compiler, document, selectorLookup);
            }
            rule.addCheck(ruleCheck);
        }
        this.conclude(rule);
    }

    private void evaluateRuleCheck(Scorecard.Rule.Check check, XPathCompiler compiler, XdmNode document, Function<String, SelectorDefinition> selectorLookup) {
        String selectorId = check.getDefinition().getSelectorId();
        String selector = selectorId != null ? selectorLookup.apply(selectorId).getExpression() : check.getDefinition().getSelector();
        Status status = Status.FAILED;
        Object message = null;
        String value = null;
        try {
            XdmValue result = compiler.evaluate(selector, (XdmItem)document);
            if (result.isEmpty()) {
                status = Status.ERROR;
                message = "No item found for selector '" + selector + "'";
            } else if (result.size() > 1) {
                status = Status.ERROR;
                message = "Selector must match a single item but found " + result.size() + " items instead";
            } else {
                XdmItem node = result.itemAt(0);
                if (!node.isAtomicValue() && !node.isNode()) {
                    status = Status.ERROR;
                    message = "Selected item is neither a node nor an atomic value";
                } else {
                    value = node.getStringValue();
                    boolean matches = this.evaluateConditionSafe(check.getDefinition().getCondition(), compiler, (XdmValue)node);
                    if (matches) {
                        status = Status.PASSED;
                    }
                }
            }
        }
        catch (SaxonApiException sae) {
            status = Status.ERROR;
            message = sae.getMessage();
        }
        check.setStatus(status);
        check.setErrorMessage((String)message);
        if (check.getDefinition().isDisplayValue()) {
            check.setValue(value);
        }
    }

    private boolean evaluateConditionSafe(String condition, XPathCompiler compiler, XdmValue contextValue) {
        Object expr = StringUtils.strip((String)condition);
        if (StringUtils.startsWithAny((CharSequence)expr, (CharSequence[])new CharSequence[]{"=", "<", ">", "!="})) {
            expr = ". " + (String)expr;
        }
        try {
            return contextValue.select(compiler.compile((String)expr).asStep()).asAtomic().getBooleanValue();
        }
        catch (ClassCastException | SaxonApiException e) {
            return false;
        }
    }

    private void conclude(Scorecard.Rule rule) {
        if (!rule.isEnabled()) {
            return;
        }
        Status lastStatus = Status.PASSED;
        for (Scorecard.Rule.Check c : rule.getChecks()) {
            Status checkStatus = c.getStatus();
            if (checkStatus.isSkipped() || checkStatus.isPassed()) continue;
            lastStatus = checkStatus;
            if (!checkStatus.isError()) continue;
            rule.setMessage(String.format("[Check #%d] %s", c.getIndex(), c.getErrorMessage()));
            break;
        }
        if (rule.getDefinition().isNegateResult()) {
            lastStatus = lastStatus.negate();
        }
        rule.setStatus(lastStatus);
        if (lastStatus.isPassed()) {
            rule.setMessage(rule.getDefinition().getSuccessMessage());
            rule.setPoints(rule.getDefinition().getPoints());
        } else if (lastStatus.isFailed()) {
            rule.setMessage(rule.getDefinition().getFailMessage());
        }
    }

    private boolean conclude(Scorecard.Group group) {
        String groupMessage;
        boolean testFailed;
        List<Scorecard.Rule> rulesThatMayFailTest;
        int totalPoints;
        int points;
        Status groupStatus;
        if (!group.isEnabled()) {
            return false;
        }
        Scorecard.Rule firstMatch = null;
        Scorecard.Rule lastMatch = null;
        int maxPoints = 0;
        int sumPointsTotal = 0;
        int sumPointsMatching = 0;
        boolean somePassed = false;
        boolean someFailed = false;
        boolean someError = false;
        List<Scorecard.Rule> rules = group.getRules();
        for (Scorecard.Rule rule2 : rules) {
            if (!rule2.getDefinition().isEnabled()) continue;
            int pointsAchieved = rule2.getPoints();
            int rulePoints = rule2.getDefinition().getPoints();
            Status ruleStatus = rule2.getStatus();
            if (ruleStatus.isError()) {
                someError = true;
                break;
            }
            maxPoints = Math.max(maxPoints, rulePoints);
            sumPointsTotal += rulePoints;
            if (ruleStatus.isPassed()) {
                somePassed = true;
                if (firstMatch == null) {
                    firstMatch = rule2;
                }
                lastMatch = rule2;
                sumPointsMatching += pointsAchieved;
                continue;
            }
            if (!ruleStatus.isFailed()) continue;
            someFailed = true;
        }
        if (someError) {
            group.setStatus(Status.ERROR);
            group.setPoints(0);
            group.setTotalPoints(0);
            return false;
        }
        GroupDefinition.Mode mode = group.getDefinition().getMode();
        if (mode == GroupDefinition.Mode.allPassed) {
            groupStatus = someFailed ? Status.FAILED : (somePassed ? Status.PASSED : Status.SKIPPED);
            points = sumPointsMatching;
            totalPoints = sumPointsTotal;
            rulesThatMayFailTest = rules;
        } else if (mode == GroupDefinition.Mode.firstPassed || mode == GroupDefinition.Mode.lastPassed) {
            int idx;
            Scorecard.Rule triggerRule = mode == GroupDefinition.Mode.firstPassed ? firstMatch : lastMatch;
            int n = idx = triggerRule != null ? rules.indexOf(triggerRule) : -1;
            groupStatus = somePassed ? Status.PASSED : (someFailed ? Status.FAILED : Status.SKIPPED);
            points = Optional.ofNullable(triggerRule).map(Scorecard.Rule::getPoints).orElse(0);
            totalPoints = maxPoints;
            rulesThatMayFailTest = idx < 0 ? rules : rules.subList(0, idx + 1);
        } else {
            groupStatus = Status.ERROR;
            points = 0;
            totalPoints = 0;
            rulesThatMayFailTest = Collections.emptyList();
        }
        boolean bl = testFailed = rulesThatMayFailTest.stream().filter(rule -> {
            boolean ruleFailedTest = rule.mayFailTest();
            if (ruleFailedTest) {
                rule.setTestFailed();
            }
            return ruleFailedTest;
        }).count() > 0L;
        if (group.mayFailTest()) {
            group.setTestFailed();
            testFailed = true;
        }
        group.setStatus(groupStatus);
        group.setPoints(points);
        group.setTotalPoints(totalPoints);
        String string = groupStatus.isPassed() ? group.getDefinition().getSuccessMessage() : (groupMessage = groupStatus.isFailed() ? group.getDefinition().getFailMessage() : null);
        if (groupMessage != null) {
            group.setMessage(groupMessage);
        }
        return testFailed;
    }

    private static double getPercentage(int numerator, int denominator) {
        return denominator > 0 ? (double)Math.round((double)numerator * 1000.0 / (double)denominator) / 10.0 : 0.0;
    }
}

