/*
 * Copyright (c) 2002-2024 Gargoyle Software Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.htmlunit.javascript;

import org.htmlunit.corejs.javascript.Context;
import org.htmlunit.corejs.javascript.debug.DebugFrame;
import org.htmlunit.corejs.javascript.debug.DebuggableScript;

/**
 * <p>
 * HtmlUnit's implementation of the {@link org.htmlunit.corejs.javascript.debug.Debugger} interface,
 * which registers {@link DebugFrameImpl} instances with Rhino for each new execution frame created.
 * See <a href="http://www.mozilla.org/rhino/rhino15R4-debugger.html">the Rhino documentation</a> or
 * <a href="http://lxr.mozilla.org/mozilla/source/js/rhino/src/org/mozilla/javascript/debug/Debugger.java">the
 * interface source code</a> for more info on the {@link org.htmlunit.corejs.javascript.debug.Debugger}
 * interface and its uses.
 * </p>
 *
 * <p>
 * Please note that this class is intended mainly to aid in the debugging and development of
 * HtmlUnit itself, rather than the debugging and development of web applications.
 * </p>
 *
 * <p>
 * In order to enable the debugging output, call
 * {@link HtmlUnitContextFactory#setDebugger(org.htmlunit.corejs.javascript.debug.Debugger)}, passing in
 * an instance of this class, and make sure your loggers are configured to output <code>TRACE</code> level log messages.
 * </p>
 *
 * @author Daniel Gredler
 * @see DebugFrameImpl
 * @see HtmlUnitContextFactory#setDebugger(org.htmlunit.corejs.javascript.debug.Debugger)
 */
public class DebuggerImpl extends DebuggerAdapter {

    /**
     * {@inheritDoc}
     */
    @Override
    public DebugFrame getFrame(final Context cx, final DebuggableScript functionOrScript) {
        return new DebugFrameImpl(functionOrScript);
    }

}
