package com.eegeo.mapapi.heatmaps;


import android.support.annotation.NonNull;
import android.support.annotation.UiThread;
import android.support.annotation.WorkerThread;

import com.eegeo.mapapi.geometry.LatLng;
import com.eegeo.mapapi.geometry.ElevationMode;
import com.eegeo.mapapi.geometry.WeightedLatLngAlt;
import com.eegeo.mapapi.polygons.PolygonOptions;
import com.eegeo.mapapi.util.NativeApiObject;

import java.security.InvalidParameterException;
import java.util.List;
import java.util.concurrent.Callable;

/**
 * A class for drawing heatmap overlays on a map, displaying the density of geographical points over
 * an area.
 * <br><br>
 * The heatmap is generated by drawing a blurred circle at each of the data points supplied. The
 * radius of each circle can be controlled, allowing the heatmap to visualize spatial trends in the
 * point data at different granularities. A larger radius will tend to cause circles to overlap with
 * nearby neighbors allowing overall trends to be seen, whereas a smaller radius will allow
 * localized detail to be visible.
 *
 * <br><br>
 * <b>Density Stops</b><br>
 * An intensity image is created by drawing a blurred circle at each input data point.
 * The heatmapDensityStops, heatmapRadii and heatmapGains properties determine the radius and
 * intensity of each circle drawn, and so in turn control the visual density of the heatmap.
 * A larger radius generates a smoother heatmap, good for showing spatial trends rather than
 * fine-grain detail. The circle has a blurred edge, with maximum intensity at its center, and falls
 * gradually to zero intensity at approximately 3x radius. The heatmapGains supply an intensity
 * multiplier for each density stop. This can be useful to adjust the appearance of the heatmap when
 * multiple density stops are used.
 * <br>
 * By specifying multiple density stops, multiple intensity images are created, one for each stop.
 * <br>
 * The densityBlend option controls which of these density images is displayed, by linearly
 * interpolating between the two density stops with the closest stop parameter.
 * <br><br>
 * <b>Color Gradient</b><br>
 * A color gradient allows heatmap intensity to be visualized with different colors, helping to
 * emphasize the information that you want to convey about your point data.
 * <br>
 * The color gradient defines a continuous color ramp to be applied to the intensity image pixels,
 * from minimum intensity to maximum intensity.
 * <br>
 * The input domain is 0.0 (corresponding to a heatmap intensity of weightMin) to 1.0 (corresponding
 * to a heatmap intensity of weightMax).
 * <br><br>
 * <b>Occluded Map Feature Styling</b>
 * <br>
 * When displaying a heatmap on an outdoor map, 3d map features such as tall buildings can obscure
 * areas of the ground, particularly at oblique (non-overhead) camera viewpoints. To aid visual
 * understanding of the heatmap when viewed in these circumstances, an optional styling is applied
 * to areas of the heatmap that are obscured by other map features, akin to a kind of x-ray effect.
 * <br>
 * This styling can be defined with the occludedMapFeatures, occludedAlpha, occludedSaturation and
 * occludedBrightness options.
 * <br>
 * Setting occludedMapFeatures to an empty array disables the styling.
 * <br>
 * Setting occludedAlpha to 0.0 will cause the specified map features to fully occlude areas of the
 * heatmap.
 * <br><br>
 * <b>Performance</b>
 * <br>
 * Internally, the class generates graphical resources for the intensity images used to draw the
 * heatmap, which can be computationally intensive. The algorithmic performance time, for density
 * stops with N entries and M input data points, grows asymptotically no faster than
 * (N * resolutionPixels^2 + M).
 * <br><br>
 * These resources are regenerated whenever the point data is changed, via setData(). In addition,
 * changing the value of some other properties causes regeneration, as listed below.
 * <br>
 * <br>&bull; setDensityStops()
 * <br>&bull; setIntensityBias()
 * <br>&bull; setWeightMin()
 * <br>&bull; setWeightMax()
 * <br>&bull; setResolutionPixels()
 * <br>&bull; setIntensityBias()
 * <br>&bull; setPolygonOptions()
 * <br>&bull; setUseApproximation()
 * <br><br>
 * Options not in the above list are suitable for changing at interactive rates, for example via UI
 * slider controls.
 * <br>
 * If these performance characteristics are constraining your application then please get in touch,
 * we’d love to hear from you! Contact us via [support]({{ site.baseurl }}/docs/api/#support), or by
 * opening a new [issue](https://github.com/wrld3d/android-api/issues/new).
 * <br><br>
 * Public methods in this class must be called on the Android UI thread.
 * <br><br>
 * See Also: [HeatmapOptions]({{ site.baseurl }}/docs/api/HeatmapOptions).
 * @see HeatmapOptions
 */
public class Heatmap extends NativeApiObject {

    private static final AllowHandleAccess m_allowHandleAccess = new AllowHandleAccess();
    private final HeatmapApi m_heatmapApi;
    private List<WeightedLatLngAlt> m_weightedPoints;
    private float[] m_heatmapDensityStops;
    private double[] m_heatmapRadii;
    private double[] m_heatmapGains;
    private float m_densityBlend;
    private boolean m_interpolateDensityByZoom;
    private double m_zoomMin;
    private double m_zoomMax;
    private double m_weightMin;
    private double m_weightMax;
    private float[] m_gradientStops;
    private int[] m_gradientColors;
    private float m_opacity;
    private int m_resolutionPixels;
    private float m_intensityBias;
    private float m_intensityScale;
    private String m_indoorMapId;
    private int m_indoorFloorId;
    private double m_elevation;
    private ElevationMode m_elevationMode;
    private HeatmapOcclusionMapFeature[] m_occludedMapFeatures;
    private float m_occludedStyleAlpha;
    private float m_occludedStyleSaturation;
    private float m_occludedStyleBrightness;
    private List<LatLng> m_polygonPoints;
    private List<List<LatLng>> m_polygonHoles;
    private float m_textureBorderPercent;
    private boolean m_useApproximation;

    /**
     * This constructor is for internal SDK use only -- use EegeoMap.addHeatmap to create a heatmap
     *
     * @eegeo.internal
     */
    @UiThread
    public Heatmap(@NonNull final HeatmapApi heatmapApi,
                  @NonNull final HeatmapOptions heatmapOptions) {
        super(heatmapApi.getNativeRunner(), heatmapApi.getUiRunner(),
                new Callable<Integer>() {
                    @Override
                    public Integer call() throws Exception {
                        return heatmapApi.create(heatmapOptions, m_allowHandleAccess);
                    }
                });

        PolygonOptions polygonOptions = heatmapOptions.getPolygonOptions();

        m_heatmapApi = heatmapApi;
        m_weightedPoints = heatmapOptions.getWeightedPoints();
        m_heatmapDensityStops = heatmapOptions.getHeatmapDensityStops();
        m_heatmapRadii = heatmapOptions.getHeatmapRadii();
        m_heatmapGains = heatmapOptions.getHeatmapGains();
        m_densityBlend = heatmapOptions.getDensityBlend();
        m_interpolateDensityByZoom = heatmapOptions.getInterpolateDensityByZoom();
        m_zoomMin = heatmapOptions.getZoomMin();
        m_zoomMax = heatmapOptions.getZoomMax();
        m_weightMin = heatmapOptions.getWeightMin();
        m_weightMax = heatmapOptions.getWeightMax();
        m_gradientColors = heatmapOptions.getGradientColors();
        m_gradientStops = heatmapOptions.getGradientStops();
        m_opacity = heatmapOptions.getOpacity();
        m_resolutionPixels = heatmapOptions.getResolutionPixels();
        m_intensityBias = heatmapOptions.getIntensityBias();
        m_intensityScale = heatmapOptions.getIntensityScale();
        m_indoorMapId = polygonOptions.getIndoorMapId();
        m_indoorFloorId = polygonOptions.getIndoorFloorId();
        m_elevation = polygonOptions.getElevation();
        m_elevationMode = polygonOptions.getElevationMode();
        m_occludedStyleAlpha = heatmapOptions.getOccludedStyleAlpha();
        m_occludedStyleSaturation = heatmapOptions.getOccludedStyleSaturation();
        m_occludedStyleBrightness = heatmapOptions.getOccludedStyleBrightness();
        m_occludedMapFeatures = heatmapOptions.getOccludedMapFeatures();
        m_polygonPoints = polygonOptions.getPoints();
        m_polygonHoles = polygonOptions.getHoles();
        m_textureBorderPercent = heatmapOptions.getTextureBorderPercent();
        m_useApproximation = heatmapOptions.getUseApproximation();


        submit(new Runnable() {
            @WorkerThread
            @Override
            public void run() {
                heatmapApi.register(Heatmap.this, m_allowHandleAccess);
            }
        });
    }

    /**
     *
     * @return The current data points to be drawn by this heatmap.
     */
    @UiThread
    public List<WeightedLatLngAlt> getWeightedPoints() { return m_weightedPoints; }

    /**
     *
     * @return The stop parameter array for the density stops.
     */
    public float[] getHeatmapDensityStops() { return m_heatmapDensityStops; }

    /**
     *
     * @return The circle radius array for the density stops.
     */
    public double[] getHeatmapRadii() { return m_heatmapRadii; }

    /**
     *
     * @return The gains array for the density stops.
     */
    public double[] getHeatmapGains() { return m_heatmapGains; }

    /**
     *
     * @return The heatmap density stop blend parameter.
     */
    public float getDensityBlend() { return m_densityBlend; }

    /**
     *
     * @return The interpolateDensityByZoom option.
     */
    public boolean getInterpolateDensityByZoom() { return m_interpolateDensityByZoom; }

    /**
     *
     * @return The zoomMin property.
     */
    public double getZoomMin() { return m_zoomMin; }

    /**
     *
     * @return The zoomMax property.
     */
    public double getZoomMax() { return m_zoomMax; }

    /**
     *
     * @return The weightMin property.
     */
    public double getWeightMin() { return m_weightMin; }

    /**
     *
     * @return The weightMax property.
     */
    public double getWeightMax() { return m_weightMax; }

    /**
     *
     * @return The stop parameter array for the color gradient.
     */
    public float[] getGradientStops() { return  m_gradientStops; }

    /**
     *
     * @return The color array for the color gradient.
     */
    public int[] getGradientColors() { return  m_gradientColors; }

    /**
     *
     * @return The heatmap opacity.
     */
    public float getOpacity() { return m_opacity; }

    /**
     *
     * @return The heatmap intensity image resolution in pixels.
     */
    public int getResolutionPixels() { return m_resolutionPixels; }


    /**
     *
     * @return The intensity bias parameter.
     */
    public float getIntensityBias() { return m_intensityBias; }

    /**
     *
     * @return The intensity scale parameter.
     */
    public float getIntensityScale() { return m_intensityScale; }

    /**
     * Gets the identifier of an indoor map on which this heatmap should be displayed, if any.
     *
     * @return For a heatmap on an indoor map, the string identifier of the indoor map; otherwise an
     * empty string.
     */
    @UiThread
    public String getIndoorMapId() {
        return m_indoorMapId;
    }

    /**
     * Gets the identifier of an indoor map floor on which this heatmap should be displayed, if any.
     *
     * @return The indoor map floor id.
     */
    @UiThread
    public int getIndoorFloorId() {
        return m_indoorFloorId;
    }

    /**
     * Returns the current elevation of the heatmap. The property is interpreted differently,
     * depending on the ElevationMode property.
     *
     * @return A height, in meters.
     */
    @UiThread
    public double getElevation() {
        return m_elevation;
    }

    /**
     * Returns the mode specifying how the Elevation property is interpreted.
     *
     * @return An enumerated value indicating whether Elevation is specified as a height above
     * terrain, or an absolute altitude above sea level.
     */
    @UiThread
    public ElevationMode getElevationMode() {
        return m_elevationMode;
    }

    /**
     *
     * @return The map feature types that, if occluding areas of the heatmap, will cause those areas
     * of the heatmap to be drawn with an alternate style.
     */
    public HeatmapOcclusionMapFeature[] getOccludedMapFeatures() { return m_occludedMapFeatures; }

    /**
     *
     * @return The opacity of occluded areas of the heatmap.
     */
    public float getOccludedStyleAlpha() { return m_occludedStyleAlpha; }

    /**
     *
     * @return The color saturation of occluded areas of the heatmap.
     */
    public float getOccludedStyleSaturation() { return m_occludedStyleSaturation; }

    /**
     *
     * @return The color brightness of occluded areas of the heatmap.
     */
    public float getOccludedStyleBrightness()  { return m_occludedStyleBrightness; }

    /**
     * Gets the outline points of the heatmap polygon.
     *
     * @return The vertices of the exterior ring (outline) of this heatmap.
     */
    @UiThread
    public List<LatLng> getPolygonPoints() {
        return m_polygonPoints;
    }

    /**
     * Gets the points that define holes for the heatmap polygon.
     *
     * @return A list of lists - each inner list contains the vertices of an interior ring (hole) of this heatmap.
     */
    @UiThread
    public List<List<LatLng>> getPolygonHoles() {
        return m_polygonHoles;
    }

    /**
     *
     * @return The size of a gutter at the edges of the heatmap images, around the bounds of the
     * data points./
     */
    public float getTextureBorderPercent() { return m_textureBorderPercent; }


    /**
     *
     * @return The useApproximation option.
     */
    public boolean getUseApproximation() { return m_useApproximation; }

    ////////

    /**
     * Sets the array of data points to draw for this heatmap. The WeightedLatLngAlt.intensity field
     * is used as a weighting value. A weight of N is equivalent to placing N points all at the same
     * coordinate, each with a weight of 1.
     * The weightMin and weightMax parameters specify a domain mapping from heatmap intensity value
     * [weightMin..weightMax] to color gradient stop function input [0..1].
     * @param weightedPoints The input data points.
     * @param weightMin The intensity corresponding to the lowest color gradient value.
     * @param weightMax The intensity corresponding to the heighest color gradient value.
     */
    public void setData(List<WeightedLatLngAlt> weightedPoints, double weightMin, double weightMax) {
        m_weightedPoints = weightedPoints;
        m_weightMin = weightMin;
        m_weightMax = weightMax;
        updateNativeData();
    }

    /**
     * Sets one or more density stop for the heatmap. Each density stop determines how the point
     * data should be drawn as a heatmap of specified density.
     * @throws InvalidParameterException if the parameter array sizes are not of equal length, or
     * are empty.
     * @param heatmapDensityStops An array of stop function input value in the range [0..1].
     * @param heatmapRadiiMeters An array with an element for each density stop, providing the
     *                           radius of the circledrawn for each point, in meters.
     * @param heatmapGains An array with an element for each density stop, providing an intensity
     *                     multiplier. This can be useful to adjust the appearance of the heatmap
     *                     when multiple density stops are used.
     */
    public void setDensityStops(float[] heatmapDensityStops, double[] heatmapRadiiMeters, double[] heatmapGains) {
        if (heatmapDensityStops.length == 0) {
            throw new InvalidParameterException("heatmapDensityStops must not be empty");
        }

        if (heatmapRadiiMeters.length != heatmapDensityStops.length) {
            throw new InvalidParameterException("heatmapRadiiMeters and stops must be equal length");
        }

        if (heatmapGains.length != heatmapDensityStops.length) {
            throw new InvalidParameterException("heatmapGains and stops must be equal length");
        }

        m_heatmapDensityStops = heatmapDensityStops;
        m_heatmapRadii = heatmapRadiiMeters;
        m_heatmapGains = heatmapGains;
        updateNativeHeatmapDensities();
    }

    /**
     * Sets the interpolation parameter between density stops.
     * @param densityBlend The density stop interpolation parameter, clamped to the range [0..1].
     */
    public void setDensityBlend(float densityBlend) {
        m_densityBlend = Math.min(Math.max(densityBlend, 0.0f), 1.0f);
        updateNativeDensityBlend();
    }

    /**
     * Sets the option to blend between density stops based on current map camera zoom level.
     * If true, densityBlend is ignored.
     *
     * @param interpolateDensityByZoom The option value.
     */
    public void setInterpolateDensityByZoom(boolean interpolateDensityByZoom) {
        m_interpolateDensityByZoom = interpolateDensityByZoom;
        updateNativeInterpolateDensityByZoom();
    }

    /**
     * Sets the extents over which heatmap density will change with map zoom, linearly interpolating
     * between them.
     * The values are ignored if interpolateDensityByZoom is false.
     * @param zoomMin The camera zoom level at which the highest density stop is displayed (with stop of 1.0)
     * @param zoomMax The camera zoom level at which the lowest density stop is displayed (with stop of 0.0)
     */
    public void setZoomExtents(double zoomMin, double zoomMax) {
        m_zoomMin = Math.max(zoomMin, 0.0);
        m_zoomMax = Math.max(zoomMax, 0.0);
        updateNativeInterpolateDensityByZoom();
    }

    /**
     * Sets the color ramp applied to each pixel of the heatmap intensity image, from minimum
     * intensity weightMin at stop == 0.0 to maximum intensity weightMax at stop == 1.0
     *
     * @param gradientStops Stop function input values in the range [0..1].
     * @param gradientColors The color value for each stop, as 32-bit ARGB color.
     * @throws InvalidParameterException if the input parameter arrays are not of equal length.
     */
    public void setGradient(float[] gradientStops, int[] gradientColors) throws InvalidParameterException {
        if (gradientColors.length != gradientStops.length) {
            throw new InvalidParameterException("gradientColors and gradientStops must have same length");
        }
        m_gradientColors = gradientColors;
        m_gradientStops = gradientStops;
        updateNativeGradient();
    }

    /**
     * Sets the overall opacity of the heatmap.
     *
     * @param opacity The opacity, clamped to the range [0..1].
     */
    public void setOpacity(float opacity) {
        m_opacity = Math.min(Math.max(opacity, 0.0f), 1.0f);
        updateNativeOpacity();
    }


    /**
     * Sets the intensity image dimensions.
     * See [HeatmapOptions.resolutionPixels()]({{ site.baseurl }}/docs/api/HeatmapOptions).
     * @param resolutionPixels The intensity image size in pixels, clamped to the range
     *                         [HeatmapOptions.RESOLUTION_PIXELS_MIN..HeatmapOptions.RESOLUTION_PIXELS_MAX]
     */
    public void setResolutionPixels(int resolutionPixels) {
        m_resolutionPixels = Math.min(
                Math.max(resolutionPixels, HeatmapOptions.RESOLUTION_PIXELS_MIN),
                HeatmapOptions.RESOLUTION_PIXELS_MAX
        );
        updateNativeResolution();
    }

    /**
     * Sets the intensity bias parameter.
     * See [HeatmapOptions.intensityBias()]({{ site.baseurl }}/docs/api/HeatmapOptions).
     * @param intensityBias The value, clamped to the range [0..1].
     */
    public void setIntensityBias(float intensityBias) {
        m_intensityBias = Math.min(Math.max(intensityBias, 0.0f), 1.0f);
        updateNativeIntensityBias();
    }

    /**
     * Sets the intensity scale parameter.
     * See [HeatmapOptions.intensityScale()]({{ site.baseurl }}/docs/api/HeatmapOptions).
     * @param intensityScale The intensity multiplier value.
     */
    public void setIntensityScale(float intensityScale) {
        m_intensityScale = Math.max(intensityScale, 0.0f);
        updateNativeIntensityScale();
    }

    /**
     * Sets the identifier of an indoor map on which this heatmap should be displayed, if any.
     * @param indoorMapId The indoor map id, or emptry string for an outdoor map.
     */
    @UiThread
    public void setIndoorMapId(String indoorMapId) {
        m_indoorMapId = indoorMapId;
        updateNativeIndoorMap();
    }

    /**
     * Sets the identifier of an indoor map floor on which this heatmap should be displayed, if any.
     *
     * @param indoorFloorId The indoor map floor id.
     */
    @UiThread
    public void setIndoorFloorId(int indoorFloorId) {
        m_indoorFloorId = indoorFloorId;
        updateNativeIndoorMap();
    }

    /**
     * Sets the elevation of this heatmap.
     *
     * @param elevation A height in meters. Interpretation depends on the current
     *                  HeatmapOptions.MarkerElevationMode.
     */
    @UiThread
    public void setElevation(double elevation) {
        m_elevation = elevation;
        updateNativeElevation();
    }

    /**
     * Sets the elevation mode for this heatmap.
     *
     * @param elevationMode The mode specifying how to interpret the Elevation property.
     */
    @UiThread
    public void setElevationMode(ElevationMode elevationMode) {
        m_elevationMode = elevationMode;
        updateNativeElevation();
    }

    /**
     * Sets the map feature types that, if occluding areas of the heatmap, will cause those areas of
     * the heatmap to be drawn with an alternate style.
     * @param occludedMapFeatures The array of occluded map features.
     */
    public void setOccludedMapFeatures(HeatmapOcclusionMapFeature[] occludedMapFeatures) {
        m_occludedMapFeatures = occludedMapFeatures;
        updateNativeOccludedStyle();
    }

    /**
     * Set style parameters used for drawing areas of the heatmap occluded by map features.
     * @param alpha The opacity, clamped to the range [0..1].
     * @param saturation The color saturation, clamped to the range [0..1].
     * @param brightness The color brightness, clamped to the range [0..1].
     */
    public void setOccludedStyle(float alpha, float saturation, float brightness) {
        m_occludedStyleAlpha = Math.min(Math.max(alpha, 0.0f), 1.0f);
        m_occludedStyleSaturation = Math.min(Math.max(saturation, 0.0f), 1.0f);
        m_occludedStyleBrightness = Math.min(Math.max(brightness, 0.0f), 1.0f);
        updateNativeOccludedStyle();
    }

    /**
     * Sets the occluded style alpha property.
     * @param alpha The opacity, clamped to the range [0..1].
     */
    public void setOccludedStyleAlpha(float alpha) {
        m_occludedStyleAlpha = Math.min(Math.max(alpha, 0.0f), 1.0f);
        updateNativeOccludedStyle();
    }

    /**
     * Sets the occluded style saturation property.
     * @param saturation The color saturation, clamped to the range [0..1].
     */
    public void setOccludedStyleSaturation(float saturation) {
        m_occludedStyleSaturation = Math.min(Math.max(saturation, 0.0f), 1.0f);
        updateNativeOccludedStyle();
    }

    /**
     * Sets the occluded style brightness property.
     * @param brightness The color brightness, clamped to the range [0..1].
     */
    public void setOccludedStyleBrightness(float brightness) {
        m_occludedStyleBrightness = Math.min(Math.max(brightness, 0.0f), 1.0f);
        updateNativeOccludedStyle();
    }

    /**
     * Sets the useApproximation option. See HeatmapOptions.useApproximation()
     * @param useApproximation The option value.
     */
    public void setUseApproximation(boolean useApproximation) {
        m_useApproximation = useApproximation;
        updateNativeUseApproximation();
    }

    /**
     * Removes this heatmap from the map and destroys the heatmap. Use EegeoMap.removeHeatmap
     *
     * @eegeo.internal
     */
    @UiThread
    public void destroy() {
        submit(new Runnable() {
            @WorkerThread
            @Override
            public void run() {
                m_heatmapApi.destroy(Heatmap.this, Heatmap.m_allowHandleAccess);
            }
        });

    }

    @UiThread
    private void updateNativeIndoorMap() {
        final String indoorMapId = m_indoorMapId;
        final int indoorFloorId = m_indoorFloorId;

        submit(new Runnable() {
            @WorkerThread
            public void run() {
                m_heatmapApi.setIndoorMap(
                        getNativeHandle(),
                        Heatmap.m_allowHandleAccess,
                        indoorMapId,
                        indoorFloorId);
            }
        });
    }

    @UiThread
    private void updateNativeElevation() {
        final double elevation = m_elevation;
        final ElevationMode elevationMode = m_elevationMode;

        submit(new Runnable() {
            @WorkerThread
            public void run() {
                m_heatmapApi.setElevation(
                        getNativeHandle(),
                        Heatmap.m_allowHandleAccess,
                        elevation,
                        elevationMode);
            }
        });
    }

    @UiThread
    private void updateNativeDensityBlend() {
        final double densityBlend = m_densityBlend;

        submit(new Runnable() {
            @WorkerThread
            public void run() {
                m_heatmapApi.setDensityBlend(
                        getNativeHandle(),
                        Heatmap.m_allowHandleAccess,
                        densityBlend);
            }
        });
    }

    @UiThread
    private void updateNativeInterpolateDensityByZoom() {
        final boolean interpolateDensityByZoom = m_interpolateDensityByZoom;
        final double zoomMin = m_zoomMin;
        final double zoomMax = m_zoomMax;

        submit(new Runnable() {
            @WorkerThread
            public void run() {
                m_heatmapApi.setInterpolateDensityByZoom(
                        getNativeHandle(),
                        Heatmap.m_allowHandleAccess,
                        interpolateDensityByZoom,
                        zoomMin,
                        zoomMax);
            }
        });
    }



    @UiThread
    private void updateNativeIntensityBias() {
        final float intensityBias = m_intensityBias;

        submit(new Runnable() {
            @WorkerThread
            public void run() {
                m_heatmapApi.setIntensityBias(
                        getNativeHandle(),
                        Heatmap.m_allowHandleAccess,
                        intensityBias);
            }
        });
    }

    @UiThread
    private void updateNativeIntensityScale() {
        final float intensityScale = m_intensityScale;

        submit(new Runnable() {
            @WorkerThread
            public void run() {
                m_heatmapApi.setIntensityScale(
                        getNativeHandle(),
                        Heatmap.m_allowHandleAccess,
                        intensityScale);
            }
        });
    }

    @UiThread
    private void updateNativeOpacity() {
        final double opacity = m_opacity;

        submit(new Runnable() {
            @WorkerThread
            public void run() {
                m_heatmapApi.setOpacity(
                        getNativeHandle(),
                        Heatmap.m_allowHandleAccess,
                        opacity);
            }
        });
    }

    @UiThread
    private void updateNativeGradient() {
        final float[] gradientStops = m_gradientStops;
        final int[] gradientColors = m_gradientColors;

        submit(new Runnable() {
            @WorkerThread
            public void run() {
                m_heatmapApi.setGradient(
                        getNativeHandle(),
                        Heatmap.m_allowHandleAccess,
                        gradientStops,
                        gradientColors);
            }
        });
    }



    @UiThread
    private void updateNativeOccludedStyle() {
        final HeatmapOcclusionMapFeature[] occludedFeatures = m_occludedMapFeatures;
        final float alpha = m_occludedStyleAlpha;
        final float saturation = m_occludedStyleSaturation;
        final float brightness = m_occludedStyleBrightness;

        submit(new Runnable() {
            @WorkerThread
            public void run() {
                m_heatmapApi.setOccludedStyle(
                        getNativeHandle(),
                        Heatmap.m_allowHandleAccess,
                        occludedFeatures,
                        alpha,
                        saturation,
                        brightness);
            }
        });
    }

    @UiThread
    private void updateNativeResolution() {
        final int resolutionPixels = m_resolutionPixels;

        submit(new Runnable() {
            @WorkerThread
            public void run() {
                m_heatmapApi.setResolution(
                        getNativeHandle(),
                        Heatmap.m_allowHandleAccess,
                        resolutionPixels
                );
            }
        });
    }

    @UiThread
    private void updateNativeHeatmapDensities() {
        final float[] heatmapDensityStops = m_heatmapDensityStops;
        final double[] heatmapRadii = m_heatmapRadii;
        final double[] heatmapGains = m_heatmapGains;

        submit(new Runnable() {
            @WorkerThread
            public void run() {
                m_heatmapApi.setHeatmapDensities(
                        getNativeHandle(),
                        Heatmap.m_allowHandleAccess,
                        heatmapDensityStops,
                        heatmapRadii,
                        heatmapGains
                );
            }
        });
    }

    @UiThread
    private void updateNativeUseApproximation() {
        final boolean useApproximation = m_useApproximation;

        submit(new Runnable() {
            @WorkerThread
            public void run() {
                m_heatmapApi.setUseApproximation(
                        getNativeHandle(),
                        Heatmap.m_allowHandleAccess,
                        m_useApproximation
                );
            }
        });
    }



    @UiThread
    private void updateNativeData() {
        final List<WeightedLatLngAlt> data = m_weightedPoints;
        final double weightMin = m_weightMin;
        final double weightMax = m_weightMax;

        submit(new Runnable() {
            @WorkerThread
            public void run() {
                m_heatmapApi.setData(
                        getNativeHandle(),
                        Heatmap.m_allowHandleAccess,
                        data,
                        weightMin,
                        weightMax
                );
            }
        });
    }

    @WorkerThread
    int getNativeHandle(AllowHandleAccess allowHandleAccess) {
        if (allowHandleAccess == null)
            throw new NullPointerException("Null access token. Method is intended for use by HeatmapApi");

        if (!hasNativeHandle())
            throw new IllegalStateException("Native handle not available");

        return getNativeHandle();
    }

    static final class AllowHandleAccess {
        @WorkerThread
        private AllowHandleAccess() {
        }
    }
}
