/*
 * Decompiled with CFR 0.152.
 */
package com.webauthn4j.validator.attestation.statement.tpm;

import com.webauthn4j.data.MessageDigestAlgorithm;
import com.webauthn4j.data.attestation.authenticator.AAGUID;
import com.webauthn4j.data.attestation.authenticator.AuthenticatorData;
import com.webauthn4j.data.attestation.statement.AttestationType;
import com.webauthn4j.data.attestation.statement.COSEAlgorithmIdentifier;
import com.webauthn4j.data.attestation.statement.ECCUnique;
import com.webauthn4j.data.attestation.statement.RSAUnique;
import com.webauthn4j.data.attestation.statement.TPMAttestationStatement;
import com.webauthn4j.data.attestation.statement.TPMGenerated;
import com.webauthn4j.data.attestation.statement.TPMIAlgHash;
import com.webauthn4j.data.attestation.statement.TPMIAlgPublic;
import com.webauthn4j.data.attestation.statement.TPMISTAttest;
import com.webauthn4j.data.attestation.statement.TPMSAttest;
import com.webauthn4j.data.attestation.statement.TPMSCertifyInfo;
import com.webauthn4j.data.attestation.statement.TPMSECCParms;
import com.webauthn4j.data.attestation.statement.TPMSRSAParms;
import com.webauthn4j.data.attestation.statement.TPMTPublic;
import com.webauthn4j.data.attestation.statement.TPMUPublicId;
import com.webauthn4j.data.extension.authenticator.RegistrationExtensionAuthenticatorOutput;
import com.webauthn4j.util.AssertUtil;
import com.webauthn4j.util.MessageDigestUtil;
import com.webauthn4j.util.SignatureUtil;
import com.webauthn4j.util.UnsignedNumberUtil;
import com.webauthn4j.validator.CoreRegistrationObject;
import com.webauthn4j.validator.attestation.statement.AbstractStatementValidator;
import com.webauthn4j.validator.attestation.statement.tpm.DefaultTPMDevicePropertyDecoder;
import com.webauthn4j.validator.attestation.statement.tpm.NullTPMDevicePropertyValidator;
import com.webauthn4j.validator.attestation.statement.tpm.TPMDeviceProperty;
import com.webauthn4j.validator.attestation.statement.tpm.TPMDevicePropertyDecoder;
import com.webauthn4j.validator.attestation.statement.tpm.TPMDevicePropertyValidator;
import com.webauthn4j.validator.exception.BadAttestationStatementException;
import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.security.InvalidKeyException;
import java.security.PublicKey;
import java.security.Signature;
import java.security.SignatureException;
import java.security.cert.CertificateParsingException;
import java.security.cert.X509Certificate;
import java.security.interfaces.ECPublicKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.EllipticCurve;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import org.jetbrains.annotations.NotNull;

public class TPMAttestationStatementValidator
extends AbstractStatementValidator<TPMAttestationStatement> {
    private static final String ID_FIDO_GEN_CE_AAGUID = "1.3.6.1.4.1.45724.1.1.4";
    private TPMDevicePropertyValidator tpmDevicePropertyValidator = new NullTPMDevicePropertyValidator();
    private TPMDevicePropertyDecoder tpmDevicePropertyDecoder = new DefaultTPMDevicePropertyDecoder();

    @Override
    @NotNull
    public AttestationType validate(@NotNull CoreRegistrationObject registrationObject) {
        AssertUtil.notNull((Object)registrationObject, (String)"registrationObject must not be null");
        if (!this.supports(registrationObject)) {
            throw new IllegalArgumentException("Specified format is not supported by " + this.getClass().getName());
        }
        TPMAttestationStatement attestationStatement = (TPMAttestationStatement)registrationObject.getAttestationObject().getAttestationStatement();
        this.validateAttestationStatementNotNull(attestationStatement);
        if (!attestationStatement.getVer().equals("2.0")) {
            throw new BadAttestationStatementException("TPM version is not supported.");
        }
        TPMSAttest certInfo = attestationStatement.getCertInfo();
        TPMTPublic pubArea = attestationStatement.getPubArea();
        AuthenticatorData<RegistrationExtensionAuthenticatorOutput> authenticatorData = registrationObject.getAttestationObject().getAuthenticatorData();
        this.validatePublicKeyEquality(pubArea, authenticatorData);
        byte[] attToBeSigned = this.getAttToBeSigned(registrationObject);
        if (certInfo.getMagic() != TPMGenerated.TPM_GENERATED_VALUE) {
            throw new BadAttestationStatementException("magic must be TPM_GENERATED_VALUE");
        }
        if (certInfo.getType() != TPMISTAttest.TPM_ST_ATTEST_CERTIFY) {
            throw new BadAttestationStatementException("type must be TPM_ST_ATTEST_CERTIFY");
        }
        COSEAlgorithmIdentifier alg = attestationStatement.getAlg();
        byte[] hash = this.calcMessageDigest(attToBeSigned, alg.toSignatureAlgorithm().getMessageDigestAlgorithm());
        if (!Arrays.equals(certInfo.getExtraData(), hash)) {
            throw new BadAttestationStatementException("extraData must be equals to the hash of attToBeSigned");
        }
        TPMSCertifyInfo certifyInfo = (TPMSCertifyInfo)certInfo.getAttested();
        TPMIAlgHash hashAlg = certifyInfo.getName().getHashAlg();
        String algJcaName = this.getAlgJcaName(hashAlg);
        byte[] pubAreaDigest = MessageDigestUtil.createMessageDigest((String)algJcaName).digest(pubArea.getBytes());
        if (!Arrays.equals(pubAreaDigest, certifyInfo.getName().getDigest())) {
            throw new BadAttestationStatementException("hash of `attested` doesn't match with name field of certifyInfo");
        }
        if (attestationStatement.getX5c() != null) {
            this.validateX5c(attestationStatement, certInfo, authenticatorData);
            return AttestationType.ATT_CA;
        }
        throw new BadAttestationStatementException("`x5c` or `ecdaaKeyId` must be present.");
    }

    void validateAttestationStatementNotNull(TPMAttestationStatement attestationStatement) {
        if (attestationStatement == null) {
            throw new BadAttestationStatementException("attestation statement is not found.");
        }
        this.validateTPMSAttestNotNull(attestationStatement.getCertInfo());
        this.validateTPMTPublicNotNull(attestationStatement.getPubArea());
    }

    void validateTPMSAttestNotNull(TPMSAttest tpmsAttest) {
        if (tpmsAttest == null) {
            throw new BadAttestationStatementException("certInfo must not be null");
        }
    }

    void validateTPMTPublicNotNull(TPMTPublic tpmtPublic) {
        if (tpmtPublic == null) {
            throw new BadAttestationStatementException("pubArea must not be null");
        }
    }

    private byte[] calcMessageDigest(byte[] data, MessageDigestAlgorithm alg) {
        return alg.createMessageDigestObject().digest(data);
    }

    private void validateX5c(TPMAttestationStatement attestationStatement, TPMSAttest certInfo, AuthenticatorData<RegistrationExtensionAuthenticatorOutput> authenticatorData) {
        X509Certificate aikCert = attestationStatement.getX5c().getEndEntityAttestationCertificate().getCertificate();
        String jcaName = this.getJcaName(attestationStatement.getAlg());
        Signature certInfoSignature = SignatureUtil.createSignature((String)jcaName);
        try {
            certInfoSignature.initVerify(aikCert.getPublicKey());
            certInfoSignature.update(certInfo.getBytes());
            if (!certInfoSignature.verify(attestationStatement.getSig())) {
                throw new BadAttestationStatementException("hash of certInfo doesn't match with sig.");
            }
        }
        catch (InvalidKeyException | SignatureException e) {
            throw new BadAttestationStatementException("Failed to validate the signature.", e);
        }
        this.validateAikCert(aikCert);
        byte[] aaguidBytes = aikCert.getExtensionValue(ID_FIDO_GEN_CE_AAGUID);
        if (aaguidBytes != null && !Objects.equals(new AAGUID(aaguidBytes), authenticatorData.getAttestedCredentialData().getAaguid())) {
            throw new BadAttestationStatementException("AAGUID in aikCert doesn't match with that in authenticatorData");
        }
    }

    String getAlgJcaName(TPMIAlgHash alg) {
        String algJcaName;
        switch (alg) {
            case TPM_ALG_SHA1: {
                algJcaName = "SHA-1";
                break;
            }
            case TPM_ALG_SHA256: {
                algJcaName = "SHA-256";
                break;
            }
            case TPM_ALG_SHA384: {
                algJcaName = "SHA-384";
                break;
            }
            case TPM_ALG_SHA512: {
                algJcaName = "SHA-512";
                break;
            }
            default: {
                throw new BadAttestationStatementException("nameAlg '" + alg.name() + "' is not supported.");
            }
        }
        return algJcaName;
    }

    public TPMDevicePropertyValidator getTpmDevicePropertyValidator() {
        return this.tpmDevicePropertyValidator;
    }

    public void setTpmDevicePropertyValidator(TPMDevicePropertyValidator tpmDevicePropertyValidator) {
        this.tpmDevicePropertyValidator = tpmDevicePropertyValidator;
    }

    public TPMDevicePropertyDecoder getTpmDevicePropertyDecoder() {
        return this.tpmDevicePropertyDecoder;
    }

    public void setTpmDevicePropertyDecoder(TPMDevicePropertyDecoder tpmDevicePropertyDecoder) {
        this.tpmDevicePropertyDecoder = tpmDevicePropertyDecoder;
    }

    private void validatePublicKeyEquality(TPMTPublic pubArea, AuthenticatorData<RegistrationExtensionAuthenticatorOutput> authenticatorData) {
        PublicKey publicKeyInAuthData = authenticatorData.getAttestedCredentialData().getCOSEKey().getPublicKey();
        TPMUPublicId publicKeyInPubArea = pubArea.getUnique();
        if (pubArea.getType() == TPMIAlgPublic.TPM_ALG_RSA && publicKeyInPubArea instanceof RSAUnique) {
            RSAPublicKey rsaPublicKey = (RSAPublicKey)publicKeyInAuthData;
            TPMSRSAParms parms = (TPMSRSAParms)pubArea.getParameters();
            RSAUnique rsaUnique = (RSAUnique)publicKeyInPubArea;
            long exponent = UnsignedNumberUtil.getUnsignedInt((byte[])parms.getExponent());
            if (exponent == 0L) {
                exponent = 65537L;
            }
            if (rsaPublicKey.getModulus().equals(new BigInteger(1, rsaUnique.getN())) && rsaPublicKey.getPublicExponent().equals(BigInteger.valueOf(exponent))) {
                return;
            }
        } else if (pubArea.getType() == TPMIAlgPublic.TPM_ALG_ECC && publicKeyInPubArea instanceof ECCUnique) {
            ECPublicKey ecPublicKey = (ECPublicKey)publicKeyInAuthData;
            TPMSECCParms parms = (TPMSECCParms)pubArea.getParameters();
            EllipticCurve curveInParms = parms.getCurveId().getEllipticCurve();
            ECCUnique eccUnique = (ECCUnique)publicKeyInPubArea;
            if (ecPublicKey.getParams().getCurve().equals(curveInParms) && ecPublicKey.getW().getAffineX().equals(new BigInteger(1, eccUnique.getX())) && ecPublicKey.getW().getAffineY().equals(new BigInteger(1, eccUnique.getY()))) {
                return;
            }
        }
        throw new BadAttestationStatementException("publicKey in authData and publicKey in unique pubArea doesn't match");
    }

    void validateAikCert(X509Certificate certificate) {
        try {
            if (!Objects.equals(certificate.getVersion(), 3)) {
                throw new BadAttestationStatementException("x5c must be version 3.");
            }
            if (!certificate.getSubjectX500Principal().getName().isEmpty()) {
                throw new BadAttestationStatementException("x5c subject field MUST be set to empty");
            }
            this.validateSubjectAlternativeName(certificate);
            if (certificate.getExtendedKeyUsage() == null || !certificate.getExtendedKeyUsage().contains("2.23.133.8.3")) {
                throw new BadAttestationStatementException("Attestation certificate doesn't contain tcg-kp-AIKCertificate (2.23.133.8.3) OID");
            }
            if (certificate.getBasicConstraints() != -1) {
                throw new BadAttestationStatementException("The Basic Constraints extension of attestation certificate must have the CA component set to false");
            }
        }
        catch (CertificateParsingException e) {
            throw new BadAttestationStatementException("Failed to parse attestation certificate", e);
        }
    }

    private void validateSubjectAlternativeName(X509Certificate certificate) throws CertificateParsingException {
        try {
            for (List<?> entry : certificate.getSubjectAlternativeNames()) {
                if (!entry.get(0).equals(4)) continue;
                TPMDeviceProperty tpmDeviceProperty = this.tpmDevicePropertyDecoder.decode((String)entry.get(1));
                this.tpmDevicePropertyValidator.validate(tpmDeviceProperty);
                return;
            }
        }
        catch (RuntimeException e) {
            throw new BadAttestationStatementException("The Subject Alternative Name extension of attestation certificate does not contain a TPM device property", e);
        }
        throw new BadAttestationStatementException("The Subject Alternative Name extension of attestation certificate does not contain a TPM device property");
    }

    private byte[] getAttToBeSigned(CoreRegistrationObject registrationObject) {
        byte[] authenticatorData = registrationObject.getAuthenticatorDataBytes();
        byte[] clientDataHash = registrationObject.getClientDataHash();
        return ByteBuffer.allocate(authenticatorData.length + clientDataHash.length).put(authenticatorData).put(clientDataHash).array();
    }
}

