package com.wassilak.cache_service.entity;

import java.util.Date;

/**
 * The EntityImpl class serves as the reference implementation of the Entity interface, which represents cached objects.
 * <p>
 * Copyright (C) 2014 John Wassilak (john@wassilak.com)
 * <p>
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * <p>
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * <p>
 * You should have received a copy of the GNU General Public License along with this program. If not, see
 * http://www.gnu.org/licenses/.
 * 
 * @param <T>
 *            The type of object that this Entity represents in a Cache.
 */
final class EntityImpl<T> implements Entity<T> {

    /**
     * The serialVersionUID constant represents the current serialization version of this class.
     */
    private static final long serialVersionUID = -5465289859131162503L;

    /**
     * The object property contains the object represented by this Entity.
     */
    private final T object;

    /**
     * The timeStamp property contains the date/time that the Entity was created.
     */
    private final Date timeStamp;

    /**
     * The timeToLive property contains the time that the Entity can live before it should be removed from the cache, in
     * milliseconds.
     */
    private final Long timeToLive;

    /**
     * The EntityImpl constructor creates an EntityImpl object using the given parameters.
     * 
     * @param object
     *            The object represented by this Entity.
     * @param timeToLive
     *            The time that the Entity can live before it should be removed from the cache, in milliseconds.
     * @throws EntityException
     *             If the Entity is being created with invalid parameters.
     */
    EntityImpl(T object, Long timeToLive) throws EntityException {
        if (object == null) {
            throw new EntityException("object must be provided");
        } else if (timeToLive == null) {
            throw new EntityException("timeToLive must be provided");
        } else if (timeToLive < Entity.MIN_TTL) {
            throw new EntityException("timeToLive must be at least " + Entity.MIN_TTL);
        } else if (timeToLive > Entity.MAX_TTL) {
            throw new EntityException("timeToLive must be at most " + Entity.MAX_TTL);
        } else {
            this.object = object;
            this.timeToLive = timeToLive;
            timeStamp = new Date();
        }
    }

    /**
     * The getObject method returns the underlying object that this cache Entity represents.
     * 
     * @return The cached object.
     */
    @Override
    public T getObject() {
        return object;
    }

    /**
     * The getTimeStamp method returns the date/time that the entity was created. This is used in determining whether or
     * not the Entity should be considered expired and removed from the cache.
     *
     * @return The date/time that the Entity was created.
     */
    @Override
    public Date getTimeStamp() {
        return timeStamp;
    }

    /**
     * The getTimeToLive method returns the time that the Entity can live before it should be removed from the cache, in
     * milliseconds.
     *
     * @return The length of time before the Entity should expire and be removed from the cache.
     */
    @Override
    public Long getTimeToLive() {
        return timeToLive;
    }
}
