package com.wassilak.cache_service.cache;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import com.wassilak.cache_service.entity.Entity;

/**
 * The CacheImpl class serves as the reference implementation of the Cache interface, which represents a cache of
 * entities..
 * <p>
 * Copyright (C) 2014 John Wassilak (john@wassilak.com)
 * <p>
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * <p>
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * <p>
 * You should have received a copy of the GNU General Public License along with this program. If not, see
 * http://www.gnu.org/licenses/.
 *
 * @param <K>
 *            The type of Keys in this Cache.
 * @param <V>
 *            The type of mapped Entities in this Cache.
 */
final class CacheImpl<K, V> implements Cache<K, V> {

    /**
     * The serialVersionUID constant represents the current serialization version of this class.
     */
    private static final long serialVersionUID = 8363605847331221757L;

    /**
     * The map property contains all of the cached Entities.
     */
    private final Map<K, List<Entity<V>>> map;

    /**
     * The CacheImpl constructor creates a cache for the implied types.
     */
    CacheImpl() {
        map = new HashMap<K, List<Entity<V>>>();
    }

    /**
     * The getEntities method returns all entites mapped to the given key. If no entites are found, null is returned.
     * 
     * @param key
     *            The key that the Entities are expected to be mapped to.
     * @return A list of Entities mapped to the given key, or null if none are found.
     */
    @Override
    public synchronized List<Entity<V>> getEntities(K key) {
        cleanCache();
        return map.get(key);
    }

    /**
     * The addEntity method adds the given Entity to the cache, mapped to the given key. Entities are mapped to the
     * given key as a list, thus if an Entity is already mapped to the given key, the given entity is simply added to
     * the list.
     * 
     * @param key
     *            The key that represents the given Entity. This is used to look up the Entity later. This key must have
     *            a unique hash code relative to objects of the same type.
     * @param entity
     *            The Entity to be added to the cache, mapped to the given key.
     */
    @Override
    public synchronized void addEntity(K key, Entity<V> entity) {
        cleanCache();
        List<Entity<V>> entityList = map.get(key);
        if (entityList == null) {
            entityList = new ArrayList<Entity<V>>();
        }
        entityList.add(entity);
        map.put(key, entityList);
    }

    /**
     * The removeEntity method removes the given Entity from the list of Entities mapped to the given key.
     * 
     * @param key
     *            The key to remove the given Entity from.
     * @param entity
     *            The Entity to remove from the list of Entities mapped to the given key.
     */
    @Override
    public synchronized void removeEntity(K key, Entity<V> entity) {
        cleanCache();
        remove(key, entity);
    }

    /**
     * The cleanCache method removes all entities that have existed longer than their defined timeToLive properties.
     */
    void cleanCache() {
        Map<K, List<Entity<V>>> removalMap = new HashMap<K, List<Entity<V>>>();
        Iterator<Entry<K, List<Entity<V>>>> iterator = map.entrySet().iterator();
        while (iterator.hasNext()) {
            Entry<K, List<Entity<V>>> entry = iterator.next();
            for (Entity<V> e : entry.getValue()) {
                if ((System.currentTimeMillis() - e.getTimeStamp().getTime()) > e.getTimeToLive()) {
                    List<Entity<V>> currentList = removalMap.get(entry.getKey());
                    if (currentList == null) {
                        currentList = new ArrayList<Entity<V>>();
                    }
                    currentList.add(e);
                    removalMap.put(entry.getKey(), currentList);
                }
            }
        }
        iterator = removalMap.entrySet().iterator();
        while (iterator.hasNext()) {
            Entry<K, List<Entity<V>>> entry = iterator.next();
            for (Entity<V> e : entry.getValue()) {
                remove(entry.getKey(), e);
            }
        }
    }

    /**
     * The removeEntity method removes the given Entity from the list of Entities mapped to the given key.
     * 
     * @param key
     *            The key to remove the given Entity from.
     * @param entity
     *            The Entity to remove from the list of Entities mapped to the given key.
     */
    void remove(K key, Entity<V> entity) {
        List<Entity<V>> entityList = map.get(key);
        while (entityList.contains(entity)) {
            entityList.remove(entity);
        }
        if (entityList.isEmpty()) {
            map.remove(key);
        } else {
            map.put(key, entityList);
        }
    }
}
