package com.wassilak.cache_service.cache;

import java.io.Serializable;
import java.util.List;

import com.wassilak.cache_service.entity.Entity;

/**
 * The Cache interface defines the behavior of the cache-service Cache object, which represents a cache of entities.
 * Multiple entites can be mapped to one key. Entities are removed from the Cache one-by-one when they have existed
 * longer than their defined timeToLive.
 * <p>
 * Copyright (C) 2014 John Wassilak (john@wassilak.com)
 * <p>
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * <p>
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * <p>
 * You should have received a copy of the GNU General Public License along with this program. If not, see
 * http://www.gnu.org/licenses/.
 *
 * @param <K>
 *            The type of Keys in this Cache.
 * @param <V>
 *            The type of mapped Entities in this Cache.
 */
public interface Cache<K, V> extends Serializable {

    /**
     * The getEntities method returns all entites mapped to the given key. If no entites are found, null is returned.
     * 
     * @param key
     *            The key that the Entities are expected to be mapped to.
     * @return A list of Entities mapped to the given key, or null if none are found.
     */
    public List<Entity<V>> getEntities(K key);

    /**
     * The addEntity method adds the given Entity to the cache, mapped to the given key. Entities are mapped to the
     * given key as a list, thus if an Entity is already mapped to the given key, the given entity is simply added to
     * the list.
     * 
     * @param key
     *            The key that represents the given Entity. This is used to look up the Entity later. This key must have
     *            a unique hash code relative to objects of the same type.
     * @param entity
     *            The Entity to be added to the cache, mapped to the given key.
     */
    public void addEntity(K key, Entity<V> entity);

    /**
     * The removeEntity method removes the given Entity from the list of Entities mapped to the given key.
     * 
     * @param key
     *            The key to remove the given Entity from.
     * @param entity
     *            The Entity to remove from the list of Entities mapped to the given key.
     */
    public void removeEntity(K key, Entity<V> entity);
}
