package webview.video.com.h5videowebview;

import android.animation.Animator;
import android.animation.AnimatorListenerAdapter;
import android.animation.ObjectAnimator;
import android.app.Activity;
import android.content.Context;
import android.content.pm.ActivityInfo;
import android.graphics.Color;
import android.net.http.SslError;
import android.os.Build;
import android.os.Handler;
import android.support.annotation.DrawableRes;
import android.support.annotation.IdRes;
import android.support.annotation.IntDef;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.util.DisplayMetrics;
import android.util.TypedValue;
import android.view.KeyEvent;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.view.WindowManager;
import android.webkit.SslErrorHandler;
import android.webkit.WebChromeClient;
import android.webkit.WebSettings;
import android.webkit.WebView;
import android.webkit.WebViewClient;
import android.widget.FrameLayout;
import android.widget.ImageView;
import android.widget.RelativeLayout;
import android.widget.TextView;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;

import static android.widget.RelativeLayout.CENTER_HORIZONTAL;
import static android.widget.RelativeLayout.CENTER_IN_PARENT;
import static android.widget.RelativeLayout.CENTER_VERTICAL;
import static android.widget.RelativeLayout.RIGHT_OF;

public class H5VideoWebView extends WebView {
    public static final int LEFT = 0;
    public static final int CENTER = 1;

    private static final int DEFAULT_MARGIN = 20;

    private static final String COM = "com";

    private final FrameLayout.LayoutParams FULL_SCREEN_PARAMS = new FrameLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT);

    private Activity mActivity;
    private View mCustomView;
    private WebChromeClient.CustomViewCallback mCustomViewCallBack;

    private FrameLayout mFullContainerView;

    private View mTitleView;

    private boolean mIsShowTitleView = true;
    private @Location
    int mLocation = LEFT;
    private int mHeight;
    private int mTitleViewBackground;
    private int mTitleColor;
    private int mTitleSize;
    private String mTitle;
    private int mResourceId = -1;

    private boolean mTitleShow = false;
    private ObjectAnimator mShowAnimator;
    private ObjectAnimator mHideAnimator;
    private float mShowAlpha = 0f;
    private int mBackMargin = DEFAULT_MARGIN;
    private int mTitleMargin = DEFAULT_MARGIN;

    public H5VideoWebView(Context context) {
        super(context);
        if (!(context instanceof Activity)) {
            throw new IllegalArgumentException("context must be Activity");
        }

        mActivity = (Activity) context;

        initWebView();
    }

    public H5VideoWebView(Context context, AttributeSet attrs) {
        super(context, attrs);
        if (!(context instanceof Activity)) {
            throw new IllegalArgumentException("context must be Activity");
        }

        mActivity = (Activity) context;

        initWebView();

    }

    public H5VideoWebView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        if (!(context instanceof Activity)) {
            throw new IllegalArgumentException("context must be Activity");
        }

        mActivity = (Activity) context;

        initWebView();
    }

    /**
     * 初始化WebView
     */
    private void initWebView() {
        initDefaultParam();

        WebSettings webSettings = getSettings();
        //设置支持缩放
        webSettings.setSupportZoom(true);
        //设置支持根据用户手势播放媒体,API17以上支持
        webSettings.setMediaPlaybackRequiresUserGesture(true);
        //设置支持内置缩放机制
        webSettings.setBuiltInZoomControls(true);
        //设置显示缩放控件,与setBuiltInZoomControls配合使用
        webSettings.setDisplayZoomControls(true);
        //设置支持文件访问
        webSettings.setAllowFileAccess(true);
        //设置支持内容URL访问
        webSettings.setAllowContentAccess(true);
        //设置内容根据屏幕宽度适配
        webSettings.setLoadWithOverviewMode(true);
        //设置字体缩放百分比，设置为100时字体大小不会随系统字体改变而改变
        webSettings.setTextZoom(100);
        //设置支持meta标签，用于调整布局宽度
        webSettings.setUseWideViewPort(true);
        //设置支持多窗口
        webSettings.setSupportMultipleWindows(true);
        //设置允许加载图片
        webSettings.setLoadsImagesAutomatically(true);
        //设置允许加载网络图片
        webSettings.setBlockNetworkImage(false);
        //设置允许加载网络资源
        webSettings.setBlockNetworkLoads(false);
        //设置允许JavaScript
        webSettings.setJavaScriptEnabled(true);
        //设置支持插件
        webSettings.setPluginState(WebSettings.PluginState.ON);
        //设置支持DOM缓存
        webSettings.setDomStorageEnabled(true);
        //设置支持定位，需要定位权限
        webSettings.setGeolocationEnabled(true);
        //设置缓存模式
        webSettings.setCacheMode(WebSettings.LOAD_DEFAULT);
        //设置内容混合模式，API21以上支持
        webSettings.setBuiltInZoomControls(true);
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            webSettings.setMixedContentMode(WebSettings.MIXED_CONTENT_COMPATIBILITY_MODE);
        }

        WebViewClient client = new WebViewClient() {
            @Override
            public void onReceivedSslError(WebView view, SslErrorHandler handler, SslError error) {
                handler.proceed();
                super.onReceivedSslError(view, handler, error);
            }
        };

        setWebViewClient(client);

        WebChromeClient chromeClient = new WebChromeClient() {

            @Nullable
            @Override
            public View getVideoLoadingProgressView() {
                return super.getVideoLoadingProgressView();
            }

            @Override
            public void onShowCustomView(View view, CustomViewCallback callback) {
                showCustomView(view, callback);
                //展示全屏设置为横屏
                mActivity.setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_LANDSCAPE);
            }

            @Override
            public void onHideCustomView() {
                hideCustomView();
                //隐藏全屏设置为竖屏
                mActivity.setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_PORTRAIT);
            }
        };

        setWebChromeClient(chromeClient);
    }

    /**
     * 初始化默认参数
     */
    private void initDefaultParam() {
        mHeight = dip2px(44);
        mTitleViewBackground = mActivity.getResources().getColor(R.color.colorGray);
        mTitleColor = mActivity.getResources().getColor(R.color.colorWhite);
        mTitleSize = 14;
    }

    /**
     * 展示自定义全屏窗口
     *
     * @param view     全屏View
     * @param callback {@link WebChromeClient.CustomViewCallback}
     */
    private void showCustomView(View view, WebChromeClient.CustomViewCallback callback) {
        if (mCustomView != null) {
            if (callback != null) {
                callback.onCustomViewHidden();
            }

            return;
        }

        FrameLayout decorView = (FrameLayout) mActivity.getWindow().getDecorView();
        decorView.setBackgroundColor(Color.BLACK);
        mFullContainerView = new FrameLayout(mActivity) {
            @Override
            public boolean dispatchTouchEvent(MotionEvent ev) {
                if (ev.getAction() == MotionEvent.ACTION_DOWN) {
                    if (mTitleView != null) {
                        if (!mTitleShow) {
                            mTitleShow = true;
                            if (mHideAnimator != null) {
                                mHideAnimator.cancel();
                            }
                            mShowAnimator = ObjectAnimator.ofFloat(mTitleView, ALPHA, mShowAlpha, 1);
                            mShowAnimator.setDuration(1000);
                            mShowAnimator.start();

                            hideTitleView();
                        } else {
                            mTitleShow = false;
                            if (mShowAnimator != null) {
                                mShowAnimator.cancel();
                            }

                            mHideAnimator = ObjectAnimator.ofFloat(mTitleView, ALPHA, 1, 0);
                            mHideAnimator.addListener(new AnimatorListenerAdapter() {

                                @Override
                                public void onAnimationEnd(Animator animation) {
                                    mShowAlpha = (float) ((ObjectAnimator) animation).getAnimatedValue();
                                    super.onAnimationEnd(animation);
                                }

                            });
                            mHideAnimator.setDuration(1000);
                            mHideAnimator.start();

                            if (mHideRun != null) {
                                mHandler.removeCallbacks(mHideRun);
                            }
                        }

                    }
                }
                return super.dispatchTouchEvent(ev);
            }
        };
        mFullContainerView.setBackgroundColor(Color.BLACK);
        mFullContainerView.addView(view, FULL_SCREEN_PARAMS);
        if (mIsShowTitleView) {
            if (mTitleView == null) {
                mTitleView = createTitleView();
            }
            mFullContainerView.addView(mTitleView);
            mTitleShow = true;
            hideTitleView();
        }
        decorView.addView(mFullContainerView, FULL_SCREEN_PARAMS);
        mCustomView = view;
        setStatusBarVisibility(false);
        mCustomViewCallBack = callback;
    }

    /**
     * 隐藏自定义全屏窗口
     */
    private void hideCustomView() {
        if (mCustomView == null) {
            return;
        }

        setStatusBarVisibility(true);
        FrameLayout decorView = (FrameLayout) mActivity.getWindow().getDecorView();
        decorView.removeView(mFullContainerView);
        mFullContainerView = null;
        mCustomView = null;
        mCustomViewCallBack.onCustomViewHidden();
        mTitleView = null;
        mTitleShow = false;
        mHandler.removeCallbacks(mHideRun);
        mHandler = null;
        setVisibility(VISIBLE);
    }


    /**
     * 设置状态栏是否可见
     *
     * @param visible 状态栏是否可见
     */
    private void setStatusBarVisibility(boolean visible) {
        int flag = visible ? 0 : WindowManager.LayoutParams.FLAG_FULLSCREEN;
        mActivity.getWindow().setFlags(flag, WindowManager.LayoutParams.FLAG_FULLSCREEN);
    }

    /**
     * 创建默认TitleView
     *
     * @return 创建好的TitleView
     */
    private View createTitleView() {
        mTitleView = new RelativeLayout(mActivity);
        mTitleView.setBackgroundColor(mTitleViewBackground);
        FrameLayout.LayoutParams params = new FrameLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, ViewGroup.LayoutParams.WRAP_CONTENT);
        params.height = mHeight;
        params.width = ViewGroup.LayoutParams.MATCH_PARENT;
        mTitleView.setLayoutParams(params);

        RelativeLayout.LayoutParams ivParams = new RelativeLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, ViewGroup.LayoutParams.WRAP_CONTENT);
        ImageView ivBack = new ImageView(mActivity);
        if (mResourceId != -1) {
            ivBack.setImageResource(mResourceId);
        } else {
            ivBack.setImageResource(R.mipmap.title_view_back);
        }
        ivBack.setId(View.generateViewId());
        int padding = 10;
        ivBack.setPadding(padding, padding, padding, padding);
        ivBack.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View v) {
                hideCustomView();
            }
        });
        ivParams.setMargins(mBackMargin, 0, 0, 0);
        ivParams.addRule(CENTER_VERTICAL);
        ivParams.addRule(LEFT);
        ivBack.setLayoutParams(ivParams);
        ((RelativeLayout) mTitleView).addView(ivBack, ivParams);

        RelativeLayout.LayoutParams tvParams = new RelativeLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, ViewGroup.LayoutParams.WRAP_CONTENT);
        TextView tvTitle = new TextView(mActivity);
        tvTitle.setTextColor(mTitleColor);
        tvTitle.setTextSize(TypedValue.COMPLEX_UNIT_SP, mTitleSize);
        if (mLocation == LEFT) {
            tvParams.addRule(RIGHT_OF, ivBack.getId());
            tvParams.addRule(CENTER_VERTICAL);
            tvParams.setMargins(mTitleMargin, 0, 0, 0);
        } else {
            tvParams.addRule(CENTER_IN_PARENT);
        }
        tvTitle.setLayoutParams(tvParams);
        String url = getUrl();
        if (TextUtils.isEmpty(mTitle) && !TextUtils.isEmpty(url)) {
            int comIndex = url.indexOf(COM);
            mTitle = url.substring(0, comIndex - 1);
        }
        tvTitle.setText(mTitle);
        ((RelativeLayout) mTitleView).addView(tvTitle, tvParams);

        return mTitleView;
    }

    /**
     * 设置TitleView
     *
     * @param titleView 需要展示的TitleView
     */
    public void setTitleView(View titleView) {
        mTitleView = titleView;
    }

    /**
     * 设置Title,不设置默认截取链接
     *
     * @param title 设置的Title
     */
    public void setTitle(String title) {
        mTitle = title;
    }

    /**
     * 设置Back图片
     * @param resourceId 资源Id
     */
    public void setBackResource(@DrawableRes int resourceId) {
        mResourceId = resourceId;
    }

    /**
     * 设置返回按钮左间距
     * @param backMargin 间距具体值
     */
    public void setBackLeftMargin(int backMargin) {
        mBackMargin = backMargin;
    }

    /**
     * 设置Title居左时距离返回按钮的左间距
     * @param titleMargin 间距具体值
     */
    public void setTitleLeftMargin(int titleMargin) {
        mTitleMargin = titleMargin;
    }

    /**
     * 设置TitleView背景颜色
     *
     * @param color 颜色资源
     */
    public void setTitleViewBackground(int color) {
        mTitleViewBackground = color;
    }

    /**
     * 设置Title颜色
     *
     * @param color 颜色资源
     */
    public void setTitleColor(int color) {
        mTitleColor = color;
    }

    /**
     * 设置Title字体大小
     *
     * @param size 字体大小
     */
    public void setmTitleSize(int size) {
        mTitleSize = size;
    }

    /**
     * 设置全屏时是否展示TitleView，默认为true
     *
     * @param isShowTitleView 是否展示TitleView
     */
    public void setIsShowTitleView(boolean isShowTitleView) {
        mIsShowTitleView = isShowTitleView;
    }

    @IntDef({LEFT, CENTER})
    @Retention(RetentionPolicy.SOURCE)
    @interface Location {
    }

    /**
     * 设置Title展示位置{@link Location}
     *
     * @param location Title展示位置
     */
    public void setTitleLocation(@Location int location) {
        mLocation = location;
    }

    /**
     * {@link H5VideoWebView#setHeight(int, boolean)},boolean默认为false
     *
     * @param height TitleView高度
     */
    public void setHeight(int height) {
        setHeight(height, false);
    }

    /**
     * 设置TitleView高度
     *
     * @param height TitleView高度
     * @param isDp   但是是否是dp
     */
    public void setHeight(int height, boolean isDp) {
        if (isDp) {
            mHeight = dip2px(height);
        } else {
            mHeight = height;
        }
    }

    /**
     * dip转px
     *
     * @param dipValue 需要转换的dp
     * @return px 转换后的px
     */
    private int dip2px(float dipValue) {
        if (null == mActivity) {
            return 0;
        }
        DisplayMetrics dm = mActivity.getApplicationContext().getResources().getDisplayMetrics();
        float density = dm.density;
        return (int) (dipValue * density + 0.5f);
    }

    @Override
    public boolean onKeyUp(int keyCode, KeyEvent event) {
        switch (keyCode) {
            case KeyEvent.KEYCODE_BACK:
                /** 回退键 事件处理 优先级:视频播放全屏-网页回退-关闭页面 */
                if (mCustomView != null) {
                    hideCustomView();
                } else if (canGoBack()) {
                    goBack();
                } else {
                    mActivity.finish();
                }
                return true;
            default:
                return super.onKeyUp(keyCode, event);
        }
    }

    private Handler mHandler;
    private Runnable mHideRun;

    /**
     * 自动隐藏TitleView
     */
    private void hideTitleView() {
        if (mTitleView != null) {
            if (mHandler == null) {
                mHandler = new Handler();
                mHandler.postDelayed(mHideRun = new Runnable() {
                    @Override
                    public void run() {
                        mTitleShow = false;
                        ObjectAnimator showAnimator = ObjectAnimator.ofFloat(mTitleView, View.ALPHA, 1, 0);
                        showAnimator.setDuration(1000);
                        showAnimator.start();
                    }
                }, 2000);

            } else {
                if (mHideRun != null) {
                    mHandler.removeCallbacks(mHideRun);
                }
                mHandler.postDelayed(mHideRun, 2000);
            }
        }
    }
}
