package com.vungle.ads.internal.util

import android.app.Activity
import android.app.Application
import android.content.Context
import android.content.Intent
import android.os.Bundle
import android.os.Handler
import android.os.Looper
import androidx.annotation.VisibleForTesting
import com.vungle.ads.AnalyticsClient
import com.vungle.ads.VungleError
import com.vungle.ads.internal.ui.PresenterAdOpenCallback
import java.lang.ref.WeakReference
import java.util.concurrent.ConcurrentHashMap
import java.util.concurrent.CopyOnWriteArraySet
import java.util.concurrent.atomic.AtomicBoolean

class ActivityManager private constructor() : Application.ActivityLifecycleCallbacks {
    open class LifeCycleCallback {
        open fun onStart() {
            //no-op
        }

        open fun onStop() {
            //no-op
        }

        open fun onResume() {
            //no-op
        }

        open fun onPause() {
            //no-op
        }
    }

    interface LeftApplicationCallback {
        fun onLeftApplication()
    }

    enum class State {
        STARTED, RESUMED, PAUSED, STOPPED, UNKNOWN,
    }

    private var state: State = State.UNKNOWN
    private var isInitialized = AtomicBoolean(false)
    private val startedActivitiesCount
        get() = startedActivities.size

    private val resumedActivitiesCount
        get() = resumedActivities.size

    private val startedActivities = CopyOnWriteArraySet<String>()
    private val resumedActivities = CopyOnWriteArraySet<String>()
    /** if init + loadAd/playAd happens on same activity, there will be no transitions that
     * ActivityManager will be aware of, so if there were no transitions we have to guess that
     * app is in foreground. For the most part it will be true, but it is still guessing */
    private var lastStoppedActivityName : String? = null

    private val callbacks = CopyOnWriteArraySet<LifeCycleCallback>()
    private val adLeftCallbacks = ConcurrentHashMap<LeftApplicationCallback?, LifeCycleCallback>()

    private val isAppForeground
        get() = resumedActivities.isNotEmpty()
    //handle configuration changes
    private var handler: Handler? = null
    private val noResumedActivities
        get() = resumedActivities.isEmpty()

    private val noStartedActivities
        get() = startedActivities.isEmpty()

    private val configChangeRunnable = Runnable {
        if (noResumedActivities && state != State.PAUSED) {
            state = State.PAUSED
            for (callback in callbacks) {
                callback.onPause()
            }
        }
        if (noStartedActivities && state == State.PAUSED) {
            state = State.STOPPED
            for (callback in callbacks) {
                callback.onStop()
            }
        }
    }

    fun init(context: Context) {
        if (this.isInitialized.getAndSet(true)) return
        try {
            handler = Handler(Looper.getMainLooper())
            handler?.post {
                val app = context.applicationContext as Application
                app.registerActivityLifecycleCallbacks(this)
            }
        } catch (e: Exception) {
            Logger.e(TAG, "Error initializing ActivityManager", e)
            isInitialized.set(false)
        }
    }

    @VisibleForTesting
    fun deInit(context: Context) {
        val app = context.applicationContext as Application
        app.unregisterActivityLifecycleCallbacks(this)
        startedActivities.clear()
        resumedActivities.clear()
        this.isInitialized.set(false)
        callbacks.clear()
        adLeftCallbacks.clear()
    }

    private fun inForeground(): Boolean {
        return !isInitialized.get() /* before init we don't know state, so just assume foreground */
                || lastStoppedActivityName == null /* assuming init is done in foreground, so app is
                 in foreground since init but before first onStop */
                || isAppForeground
    }

    fun addListener(callback: LifeCycleCallback) {
        callbacks.add(callback)
    }

    private fun removeListener(callback: LifeCycleCallback) {
        callbacks.remove(callback)
    }

    override fun onActivityCreated(activity: Activity, savedInstanceState: Bundle?) {
        //no-op
    }

    override fun onActivityStarted(activity: Activity) {
        startedActivities.add(activity.toString())
        if (startedActivitiesCount == 1 && state !in listOf(State.STARTED, State.RESUMED)) {
            state = State.STARTED
            for (callback in callbacks) {
                callback.onStart()
            }
        }
    }

    override fun onActivityStopped(activity: Activity) {
        lastStoppedActivityName = activity.toString()
        startedActivities.remove(activity.toString())
        if (noStartedActivities) {
            handler?.apply {
                removeCallbacks(configChangeRunnable)
                postDelayed(configChangeRunnable, CONFIG_CHANGE_DELAY)
            }
        }
    }

    override fun onActivityResumed(activity: Activity) {
        val wasPaused = noResumedActivities
        resumedActivities.add(activity.toString())
        if (resumedActivitiesCount == 1) {
            if (wasPaused && state !in listOf(State.RESUMED)) {
                state = State.RESUMED
                for (callback in callbacks) {
                    callback.onResume()
                }
            } else {
                handler?.removeCallbacks(configChangeRunnable)
            }
        }
    }

    override fun onActivityPaused(activity: Activity) {
        resumedActivities.remove(activity.toString())
        if (noResumedActivities) {
            handler?.removeCallbacks(configChangeRunnable)
            handler?.postDelayed(configChangeRunnable, CONFIG_CHANGE_DELAY)
        }
    }

    override fun onActivitySaveInstanceState(activity: Activity, outState: Bundle) {
        //no-op
    }

    override fun onActivityDestroyed(activity: Activity) {
        //no-op
    }

    fun addOnNextAppLeftCallback(leftCallback: LeftApplicationCallback?) {
        if (leftCallback == null) {
            return
        }
        if (!this.isInitialized.get()) {
            leftCallback.onLeftApplication()
            return
        }
        val weakCallback = WeakReference(leftCallback)
        val cancelRunnable: Runnable = object : Runnable {
            override fun run() {
                handler?.removeCallbacks(this)
                removeOnNextAppLeftCallback(weakCallback.get())
            }
        }
        val callback: LifeCycleCallback = object : LifeCycleCallback() {
            var wasPaused = false
            override fun onStop() {
                super.onStop()
                val leftCallback = weakCallback.get()
                if (wasPaused && leftCallback != null && adLeftCallbacks.containsKey(leftCallback)) {
                    leftCallback.onLeftApplication()
                }
                removeOnNextAppLeftCallback(leftCallback)
                handler?.removeCallbacks(cancelRunnable)
            }

            override fun onResume() {
                super.onResume()
                handler?.postDelayed(
                    cancelRunnable,
                    CONFIG_CHANGE_DELAY * 2
                ) //wait for delayed onPause
            }

            override fun onPause() {
                super.onPause()
                wasPaused = true
                handler?.removeCallbacks(cancelRunnable)
            }
        }

        //handle starting from background and no activity resolved
        adLeftCallbacks[leftCallback] = callback
        if (inForeground()) {
            handler?.postDelayed(cancelRunnable, TIMEOUT)
            addListener(callback)
        } else {
            instance.addListener(object : LifeCycleCallback() {
                override fun onStart() {
                    instance.removeListener(this)
                    val callback = adLeftCallbacks[weakCallback.get()]
                    if (callback != null) {
                        handler?.postDelayed(cancelRunnable, TIMEOUT)
                        addListener(callback)
                    }
                }
            })
        }
    }

    private fun removeOnNextAppLeftCallback(leftCallback: LeftApplicationCallback?) {
        if (leftCallback == null) return
        val callback = adLeftCallbacks.remove(leftCallback)
        callback?.let { removeListener(it) }
    }

    companion object {
        val TAG = ActivityManager::class.java.simpleName
        @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
        internal val instance = ActivityManager()

        @VisibleForTesting
        val TIMEOUT: Long = 3000 //no activity to show timeout

        @VisibleForTesting
        val CONFIG_CHANGE_DELAY: Long = 700 //ms, max render delay, must be less than TIMEOUT
        fun startWhenForeground(
            context: Context,
            deepLinkOverrideIntent: Intent?,
            defaultIntent: Intent?,
            leftCallback: LeftApplicationCallback?,
            adOpenCallback: PresenterAdOpenCallback?
        ) {
            val weakContext = WeakReference(context)
            if (instance.inForeground()) {
                if (startActivityHandleException(
                        context,
                        deepLinkOverrideIntent,
                        defaultIntent,
                        adOpenCallback
                    )
                ) {
                    instance.addOnNextAppLeftCallback(leftCallback)
                }
            } else {
                instance.addListener(object : LifeCycleCallback() {
                    override fun onStart() {
                        super.onStart()
                        instance.removeListener(this)
                        val context = weakContext.get()
                        if (context != null && startActivityHandleException(
                                context,
                                deepLinkOverrideIntent,
                                defaultIntent,
                                adOpenCallback
                            )
                        ) {
                            instance.addOnNextAppLeftCallback(leftCallback)
                        }
                    }
                })
            }
        }

        fun startWhenForeground(
            context: Context, deeplinkOverrideIntent: Intent?,
            defaultIntent: Intent?, leftCallback: LeftApplicationCallback?
        ) {
            startWhenForeground(context, deeplinkOverrideIntent, defaultIntent, leftCallback, null)
        }

        private fun startActivityHandleException(
            context: Context,
            deepLinkOverrideIntent: Intent?,
            defaultIntent: Intent?,
            adOpenCallback: PresenterAdOpenCallback?
        ): Boolean {
            if (deepLinkOverrideIntent == null && defaultIntent == null) {
                return false
            }
            try {
                if (deepLinkOverrideIntent != null) {
                    context.startActivity(deepLinkOverrideIntent)

                    adOpenCallback?.onDeeplinkClick(true)
                } else {
                    context.startActivity(defaultIntent)
                }
            } catch (exception: Exception) {
                Logger.e(
                    TAG,
                    "Cannot launch/find activity to handle the Implicit intent: $exception"
                )
                try {
                    if (deepLinkOverrideIntent != null) {
                        AnalyticsClient.logError(
                            VungleError.DEEPLINK_OPEN_FAILED,
                            "Fail to open ${deepLinkOverrideIntent.dataString}",
                            ""
                        )
                        adOpenCallback?.onDeeplinkClick(false)
                    }

                    if (deepLinkOverrideIntent == null || defaultIntent == null) {
                        return false
                    }
                    context.startActivity(defaultIntent)
                } catch (exception: Exception) {
                    return false
                }
                return true
            }
            return true
        }

        fun init(context: Context) {
            instance.init(context)
        }

        fun isForeground(): Boolean {
            return instance.inForeground()
        }

        fun addLifecycleListener(listener: LifeCycleCallback) {
            instance.addListener(listener)
        }
    }
}
