package com.vungle.ads.internal.ui

import android.app.Activity
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.content.res.Configuration
import android.graphics.Color
import android.media.AudioManager
import android.os.Bundle
import android.view.MotionEvent
import android.view.Window
import android.view.WindowManager
import android.widget.FrameLayout
import androidx.annotation.VisibleForTesting
import androidx.core.view.WindowCompat
import androidx.core.view.WindowInsetsCompat
import androidx.core.view.WindowInsetsControllerCompat
import com.vungle.ads.*
import com.vungle.ads.ServiceLocator.Companion.inject
import com.vungle.ads.internal.ConfigManager
import com.vungle.ads.internal.executor.Executors
import com.vungle.ads.internal.model.AdPayload
import com.vungle.ads.internal.model.BidPayload
import com.vungle.ads.internal.model.UnclosedAd
import com.vungle.ads.internal.omsdk.OMTracker
import com.vungle.ads.internal.platform.Platform
import com.vungle.ads.internal.presenter.AdEventListener
import com.vungle.ads.internal.presenter.MRAIDPresenter
import com.vungle.ads.internal.presenter.PresenterDelegate
import com.vungle.ads.internal.signals.SignalManager
import com.vungle.ads.internal.ui.view.MRAIDAdWidget
import com.vungle.ads.internal.util.Logger
import com.vungle.ads.internal.util.RingerModeReceiver
import java.util.concurrent.ExecutorService

abstract class AdActivity : Activity() {

    @VisibleForTesting internal var placementRefId: String = ""

    @VisibleForTesting internal var mraidPresenter: MRAIDPresenter? = null

    @VisibleForTesting internal var mraidAdWidget: MRAIDAdWidget? = null

    private var unclosedAd: UnclosedAd? = null

    private val ringerModeReceiver: RingerModeReceiver = RingerModeReceiver()

    private var isReceiverRegistered = false

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        requestWindowFeature(Window.FEATURE_NO_TITLE)

        window.setFlags(
            WindowManager.LayoutParams.FLAG_HARDWARE_ACCELERATED,
            WindowManager.LayoutParams.FLAG_HARDWARE_ACCELERATED
        )

        placementRefId = getPlacement(intent).let { it.toString() }

        val adv = advertisement
        val placement = ConfigManager.getPlacement(placementRefId)
        if (placement == null || adv == null) {
            eventListener?.onError(
                AdNotLoadedCantPlay(),
                placementRefId
            )
            finish()
            return
        }

        window.decorView.setBackgroundColor(Color.BLACK)

        val mraidAdWidget: MRAIDAdWidget?
        try {
            mraidAdWidget = MRAIDAdWidget(this)
        } catch (ex: InstantiationException) {
            eventListener?.onError(
                AdCantPlayWithoutWebView().apply {
                    setPlacementId(placementRefId)
                    setEventId(advertisement?.eventId())
                    setCreativeId(advertisement?.getCreativeId())
                }.logError(),
                placementRefId
            )
            finish()
            return
        }

        val signalManager: SignalManager by inject(this)
        unclosedAd = getEventId(intent)?.let { UnclosedAd(it) }
        unclosedAd?.let { signalManager.recordUnclosedAd(it) }

        mraidAdWidget.run {
            setCloseDelegate(object : MRAIDAdWidget.CloseDelegate {
                override fun close() {
                    unclosedAd?.let { signalManager.removeUnclosedAd(it) }
                    finish()
                }
            })

            setOnViewTouchListener(object : MRAIDAdWidget.OnViewTouchListener {
                override fun onTouch(event: MotionEvent?): Boolean {
                    mraidPresenter?.onViewTouched(event)
                    return false
                }
            })

            setOrientationDelegate(object : MRAIDAdWidget.OrientationDelegate {
                override fun setOrientation(orientation: Int) {
                    requestedOrientation = orientation
                }
            })
        }

        val executors: Executors by inject(this)
        val platform : Platform by inject(this)
        val offloadExecutor: ExecutorService = executors.offloadExecutor
        val webClient = VungleWebClient(adv, placement, offloadExecutor, signalManager, platform)
        val omTrackerFactory: OMTracker.Factory by inject(this)
        val omTracker = omTrackerFactory.make(ConfigManager.omEnabled() && adv.omEnabled())
        val jobExecutor: ExecutorService = executors.jobExecutor

        // Add webview observer for OMSDK
        webClient.setWebViewObserver(omTracker)

        ringerModeReceiver.webClient = webClient

        val mraidPresenter = MRAIDPresenter(
            mraidAdWidget, adv, placement, webClient, jobExecutor, omTracker, bidPayload, platform,
        )
        mraidPresenter.setEventListener(eventListener)
        mraidPresenter.setPresenterDelegate(presenterDelegate)

        mraidPresenter.prepare()

        setContentView(mraidAdWidget, mraidAdWidget.layoutParams)

        adv.adConfig?.getWatermark()?.let {
            val imageView = WatermarkView(this, it)
            window.decorView.findViewById<FrameLayout>(android.R.id.content).addView(imageView)
            imageView.bringToFront()
        }

        this.mraidAdWidget = mraidAdWidget
        this.mraidPresenter = mraidPresenter
    }

    override fun onNewIntent(intent: Intent) {
        super.onNewIntent(intent)
        val oldPlacement = getPlacement(getIntent())
        val newPlacement = getPlacement(intent)

        val oldEventId = getEventId(getIntent())
        val newEventId = getEventId(intent)

        // Check playing fullscreen ads with different placement or different advertisement that already display.
        if (oldPlacement != null && newPlacement != null && oldPlacement != newPlacement ||
            oldEventId != null && newEventId != null && oldEventId != newEventId
        ) {
            Logger.d(
                TAG,
                "Tried to play another placement $newPlacement while playing $oldPlacement"
            )
            onConcurrentPlaybackError(newPlacement)
        }
    }

    private fun onConcurrentPlaybackError(placement: String?) {
        val exception = ConcurrentPlaybackUnsupported()
        eventListener?.onError(exception, placement)
        with(exception) {
            placementId = placementRefId
            creativeId = advertisement?.getCreativeId()
            eventId = advertisement?.eventId()
            logErrorNoReturnValue()
        }
        Logger.e(TAG, "onConcurrentPlaybackError: ${exception.localizedMessage}")
    }

    override fun onResume() {
        super.onResume()
        hideSystemUi()

        try {
            if (!isReceiverRegistered) {
                val filter = IntentFilter(AudioManager.RINGER_MODE_CHANGED_ACTION)
                registerReceiver(ringerModeReceiver, filter)
                isReceiverRegistered = true
                Logger.d(TAG, "registerReceiver(): ${ringerModeReceiver.hashCode()}")
            }
        } catch (e: Exception) {
            Logger.e(TAG, "registerReceiver error: ${e.localizedMessage}")
        }

        mraidPresenter?.start()
    }

    override fun onPause() {
        super.onPause()

        try {
            if (isReceiverRegistered) {
                unregisterReceiver(ringerModeReceiver)
                isReceiverRegistered = false
                Logger.d(TAG, "unregisterReceiver(): ${ringerModeReceiver.hashCode()}")
            }
        } catch (e: Exception) {
            Logger.e(TAG, "unregisterReceiver error: ${e.localizedMessage}")
        }

        mraidPresenter?.stop()
    }

    override fun onConfigurationChanged(newConfig: Configuration) {
        try {
            super.onConfigurationChanged(newConfig)

            // Checks the orientation of the screen
            if (newConfig.orientation == Configuration.ORIENTATION_LANDSCAPE) {
                Logger.d(TAG, "landscape")
            } else if (newConfig.orientation == Configuration.ORIENTATION_PORTRAIT) {
                Logger.d(TAG, "portrait")
            }
            mraidPresenter?.onViewConfigurationChanged()
        } catch (e: Exception) {
            Logger.e(TAG, "onConfigurationChanged: ${e.localizedMessage}")
        }
    }

    override fun onBackPressed() {
        mraidPresenter?.handleExit()
    }

    override fun setRequestedOrientation(requestedOrientation: Int) {
        if (canRotate()) {
            super.setRequestedOrientation(requestedOrientation)
        }
    }

    @VisibleForTesting(otherwise = VisibleForTesting.PROTECTED)
    internal open fun canRotate(): Boolean {
        return false
    }

    override fun onDestroy() {
        mraidPresenter?.detach(
            (if (isChangingConfigurations) MRAIDAdWidget.AdStopReason.IS_CHANGING_CONFIGURATION else 0)
                    or MRAIDAdWidget.AdStopReason.IS_AD_FINISHING
        )
        super.onDestroy()
    }

    private fun hideSystemUi() {
        val windowInsetsController = WindowCompat.getInsetsController(window, window.decorView)
        // Configure the behavior of the hidden system bars
        windowInsetsController.systemBarsBehavior =
            WindowInsetsControllerCompat.BEHAVIOR_SHOW_TRANSIENT_BARS_BY_SWIPE
        // Hide both the status bar and the navigation bar
        windowInsetsController.hide(WindowInsetsCompat.Type.systemBars())
    }

    companion object {
        @VisibleForTesting internal const val REQUEST_KEY_EXTRA = "request"
        @VisibleForTesting internal const val REQUEST_KEY_EVENT_ID_EXTRA = "request_eventId"
        private const val TAG = "AdActivity"

        @get:VisibleForTesting
        internal var eventListener: AdEventListener? = null
        internal var presenterDelegate: PresenterDelegate? = null

        internal var advertisement: AdPayload? = null
        internal var bidPayload: BidPayload? = null

        fun createIntent(context: Context?, placement: String, eventId: String?): Intent {
            val intent = Intent(context, VungleActivity::class.java)
            if (context !is Activity) {
                intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
            }
            val bundle = Bundle()
            bundle.putString(REQUEST_KEY_EXTRA, placement)
            bundle.putString(REQUEST_KEY_EVENT_ID_EXTRA, eventId)
            intent.putExtras(bundle)
            return intent
        }

        private fun getPlacement(intent: Intent): String? {
            return try {
                intent.extras?.getString(REQUEST_KEY_EXTRA)
            } catch (_: Exception) {
                null
            }
        }

        private fun getEventId(intent: Intent): String? {
            return try {
                intent.extras?.getString(REQUEST_KEY_EVENT_ID_EXTRA)
            } catch (_: Exception) {
                null
            }
        }

    }
}
