package com.vungle.ads.internal.model

import android.util.Base64
import kotlinx.serialization.Serializable
import kotlinx.serialization.Transient
import kotlinx.serialization.decodeFromString
import kotlinx.serialization.json.Json
import java.io.ByteArrayInputStream
import java.util.zip.GZIPInputStream

@Serializable
data class BidPayload(
    val version: Int? = null,
    val adunit: String? = null,
    val impression: List<String>? = null
) {

    @Transient
    private val json = Json {
        ignoreUnknownKeys = true
        encodeDefaults = true
        explicitNulls = false
    }

    private val ad = adunit?.let {
        val adInst: AdPayload?
        val decoded = getDecodedAdsResponse()
        adInst = decoded?.let { json.decodeFromString<AdPayload>(decoded) }
        adInst
    }

    fun getPlacementId() = ad?.placementId()

    fun getEventId() = ad?.eventId()

    fun getAdPayload() = ad

    @Throws(Throwable::class)
    fun getDecodedAdsResponse(): String? {
        // it's ok if base64.decode throws IllegalArgumentException exception if not valid base64
        // string. The AdInternal will capture the exception in the decodeFromString<BidPayload>
        // then it will handle the exception there. No need to silence it here.
        val compressed: ByteArray? = Base64.decode(adunit, Base64.DEFAULT)
        return compressed?.let { gzipDecode(it) }
    }

    @Throws(Throwable::class)
    private fun gzipDecode(compressed: ByteArray): String {
        val bufferSize = 32
        val result = StringBuilder()
        ByteArrayInputStream(compressed).use { byteInputStream ->
            GZIPInputStream(byteInputStream, bufferSize).use { gzipInputStream ->
                val data = ByteArray(bufferSize)
                var bytesRead: Int
                while (gzipInputStream.read(data).also { bytesRead = it } != -1) {
                    result.append(String(data, 0, bytesRead))
                }
            }
        }
        return result.toString()
    }
}
