package com.vungle.warren.utility;

import android.text.TextUtils;
import android.util.Log;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import androidx.annotation.VisibleForTesting;

import static com.vungle.warren.utility.FileUtility.closeQuietly;

/**
 * Utility
 */
public class UnzipUtility {
    /**
     * Size of the buffer to read/write data
     */
    private static final int BUFFER_SIZE = 4096;
    private static final String TAG = UnzipUtility.class.getCanonicalName();

    /**
     * Extracts a zip file specified by the zipFilePath to a directory specified by
     * destDirectory (will be created if does not exists)
     *
     * @param zipFilePath
     * @param destDirectory
     * @throws IOException
     */
    public static List<File> unzip(String zipFilePath, String destDirectory) throws IOException {
        return unzip(zipFilePath, destDirectory, null);
    }


    public static List<File> unzip(String zipFilePath, String destDirectory, Filter filter) throws IOException {
        if (TextUtils.isEmpty(zipFilePath))
            throw new IOException("Path is empty");

        File src = new File(zipFilePath);

        if (!src.exists()) {
            throw new IOException("File does not exist");
        }

        File destDir = new File(destDirectory);
        if (!destDir.exists()) {
            destDir.mkdir();
        }

        ZipFile zipFile = null;

        List<File> extractedFiles = new ArrayList<>();
        try {
            zipFile = new ZipFile(src);
            Enumeration<? extends ZipEntry> entries = zipFile.entries();

            // iterates over entries in the zip file
            while (entries.hasMoreElements()) {

                ZipEntry entry = entries.nextElement();
                String filePath = destDirectory + File.separator + entry.getName();

                if (filter == null || filter.matches(filePath)) {
                    validateFilename(filePath, destDirectory);

                    if (entry.isDirectory()) {
                        // if the entry is a directory, make the directory if it does not exist.
                        File dir = new File(filePath);
                        if (!dir.exists()) {
                            dir.mkdir();
                        }
                    } else {
                        // if the entry is a file, extracts it
                        extractFile(zipFile.getInputStream(entry), filePath);
                        extractedFiles.add(new File(filePath));
                    }
                }

            }
        } finally {
            try {
                if (zipFile != null) {
                    zipFile.close();
                }
            } catch (IOException e) {
                //ignored
            }
        }

        return extractedFiles;
    }


    /**
     * Extracts a zip entry (file entry)
     *
     * @param zipIn
     * @param filePath
     * @throws IOException
     */
    @VisibleForTesting
    static void extractFile(InputStream zipIn, String filePath) throws IOException {
        File dest = new File(filePath);

        FileUtility.delete(dest);

        File parentDir = dest.getParentFile();
        if (parentDir != null && !parentDir.exists()) {
            parentDir.mkdirs();
        }

        BufferedOutputStream bos = null;
        FileOutputStream fos = null;
        try {
            bos = new BufferedOutputStream(fos = new FileOutputStream(filePath));
            byte[] bytesIn = new byte[BUFFER_SIZE];
            int read;
            while ((read = zipIn.read(bytesIn)) != -1) {
                bos.write(bytesIn, 0, read);
            }
        } finally {
            closeQuietly(zipIn);
            closeQuietly(bos);
            closeQuietly(fos);
        }
    }

    private static String validateFilename(String filename, String intendedDir)
            throws IOException {
        File f = new File(filename);
        String canonicalPath = f.getCanonicalPath();

        File iD = new File(intendedDir);
        String canonicalID = iD.getCanonicalPath();

        if (canonicalPath.startsWith(canonicalID)) {
            return canonicalPath;
        } else {
            final String msg = "File is outside extraction target directory.";
            Log.e(TAG, msg);
            throw new ZipSecurityException(msg);
        }
    }

    static class ZipSecurityException extends IOException {
        ZipSecurityException(String message) {
            super(message);
        }
    }

    public interface Filter {
        boolean matches(String extractPath);
    }
}
