package com.vungle.warren.utility;

import android.os.Handler;
import android.os.Looper;
import androidx.annotation.NonNull;

import java.util.List;
import java.util.concurrent.AbstractExecutorService;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

public class SDKExecutors implements Executors {

    private static final int SINGLE_CORE_POOL_SIZE = 1;

    private static final int IO_KEEP_ALIVE_TIME_SECONDS = 5;
    private static final int VUNGLE_KEEP_ALIVE_TIME_SECONDS = 10;

    private static final int JOBS_KEEP_ALIVE_TIME_SECONDS = 1;
    private static int NUMBER_OF_CORES = Runtime.getRuntime().availableProcessors();

    /**
     * Executor to execute IO file, db calls.
     */
    private static final ExecutorService IO_EXECUTOR;
    /**
     * Executor to execute business logic and wait for results from other executors.
     */
    private static final ExecutorService BACKGROUND_EXECUTOR;
    /**
     * Executor to execute tasks and jobs.
     */
    private static final ExecutorService JOB_EXECUTOR;
    /**
     * Main/UI thread executor implementation.
     */
    private static final ExecutorService UI_EXECUTOR;

    /**
     * Executor to execute public blocking Vungle API calls as fast as possible.
     */
    private static final ThreadPoolExecutor API_EXECUTOR;

    static {
        ExecutorService uiExecutor = new AbstractExecutorService() {

            private final Handler UIHandler = new Handler(Looper.getMainLooper());

            @Override
            public void shutdown() {
                //no-op
            }


            @Override
            public List<Runnable> shutdownNow() {
                return null;
            }

            @Override
            public boolean isShutdown() {
                return false;
            }

            @Override
            public boolean isTerminated() {
                return false;
            }

            @Override
            public boolean awaitTermination(long timeout, @NonNull TimeUnit unit) {
                return false;
            }

            @Override
            public void execute(@NonNull Runnable command) {
                UIHandler.post(command);
            }

        };

        ThreadPoolExecutor jobExecutor = new ThreadPoolExecutor(
                NUMBER_OF_CORES,
                NUMBER_OF_CORES,
                JOBS_KEEP_ALIVE_TIME_SECONDS,
                TimeUnit.SECONDS,
                new PriorityBlockingQueue<Runnable>(),
                new NamedThreadFactory("vng_jr")
        );
        jobExecutor.allowCoreThreadTimeOut(true);

        ThreadPoolExecutor ioExecutor = new ThreadPoolExecutor(
                SINGLE_CORE_POOL_SIZE,
                SINGLE_CORE_POOL_SIZE,
                IO_KEEP_ALIVE_TIME_SECONDS,
                TimeUnit.SECONDS,
                new LinkedBlockingQueue<Runnable>(),
                new NamedThreadFactory("vng_io")
        );
        ioExecutor.allowCoreThreadTimeOut(true);

        ThreadPoolExecutor backgroundExecutor = new ThreadPoolExecutor(
                SINGLE_CORE_POOL_SIZE,
                SINGLE_CORE_POOL_SIZE,
                VUNGLE_KEEP_ALIVE_TIME_SECONDS,
                TimeUnit.SECONDS,
                new LinkedBlockingQueue<Runnable>(),
                new NamedThreadFactory("vng_background")
        );
        backgroundExecutor.allowCoreThreadTimeOut(true);

        ThreadPoolExecutor apiExecutor = new ThreadPoolExecutor(
                SINGLE_CORE_POOL_SIZE,
                SINGLE_CORE_POOL_SIZE,
                VUNGLE_KEEP_ALIVE_TIME_SECONDS,
                TimeUnit.SECONDS,
                new LinkedBlockingQueue<Runnable>(),
                new NamedThreadFactory("vng_api")
        );
        apiExecutor.allowCoreThreadTimeOut(true);

        UI_EXECUTOR = uiExecutor;
        JOB_EXECUTOR = jobExecutor;
        IO_EXECUTOR = ioExecutor;
        BACKGROUND_EXECUTOR = backgroundExecutor;
        API_EXECUTOR = apiExecutor;
    }

    @Override
    public ExecutorService getUIExecutor() {
        return UI_EXECUTOR;
    }

    @Override
    public ExecutorService getBackgroundExecutor() {
        return BACKGROUND_EXECUTOR;
    }

    @Override
    public ExecutorService getIOExecutor() {
        return IO_EXECUTOR;
    }

    @Override
    public ExecutorService getJobExecutor() {
        return JOB_EXECUTOR;
    }

    @Override
    public ExecutorService getApiExecutor() {
        return API_EXECUTOR;
    }

    private static class NamedThreadFactory implements ThreadFactory {
        private final String name;
        private ThreadFactory threadFactory = java.util.concurrent.Executors.defaultThreadFactory();
        private AtomicInteger atomicInteger = new AtomicInteger(0);

        public NamedThreadFactory(String name) {
            this.name = name;
        }

        public Thread newThread(@NonNull Runnable r) {
            Thread t = threadFactory.newThread(r);
            t.setName(name + "-th-" + atomicInteger.incrementAndGet());
            return t;
        }
    }

}
