package com.vungle.warren.utility;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import android.util.Log;

import com.vungle.warren.BuildConfig;

import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;

public class FileUtility {

    private static final String TAG = FileUtility.class.getSimpleName();
    @VisibleForTesting
    protected static final List<Class<?>> allowedClasses = Arrays.<Class<?>>asList(
            LinkedHashSet.class, HashSet.class, HashMap.class, ArrayList.class, File.class
    );

    public static void printDirectoryTree(File folder) {
        if (!BuildConfig.DEBUG) return;

        if (folder == null) {
            Log.d(TAG, "File is null ");
            return;
        }

        if (!folder.exists()) {
            Log.d(TAG, "File does not exist " + folder.getPath());
            return;
        }

        if (!folder.isDirectory()) {
            Log.d(TAG, "File is not a directory " + folder.getPath());
            return;
        }

        int indent = 0;
        StringBuilder sb = new StringBuilder();
        printDirectoryTree(folder, indent, sb);
        Log.d("Vungle", sb.toString());
    }

    private static void printDirectoryTree(File folder, int indent, StringBuilder sb) {
        if (folder == null) {
            return;
        }

        if (!folder.isDirectory()) {
            throw new IllegalArgumentException("folder is not a Directory");
        }

        sb.append(getIndentString(indent)).append("+--").append(folder.getName()).append("/\n");
        File[] files = folder.listFiles();
        if (files == null) {
            return;
        }

        for (File file : folder.listFiles()) {
            if (file.isDirectory()) {
                printDirectoryTree(file, indent + 1, sb);
            } else {
                printFile(file, indent + 1, sb);
            }
        }
    }

    private static void printFile(File file, int indent, StringBuilder sb) {
        sb.append(getIndentString(indent)).append("+--").append(file.getName()).append('\n');
    }

    private static String getIndentString(int indent) {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < indent; i++) {
            sb.append("|  ");
        }
        return sb.toString();
    }

    /**
     * Helper method to recursively delete a directory. Used to clean up assets by their identifying
     * folder names.
     *
     * @param f The file or directory to delete.
     * @throws IOException if deleting a file fails for any reason.
     */
    public static void delete(File f) throws IOException {
        if (f == null || !f.exists()) return;

        if (f.isDirectory()) {
            File[] files = f.listFiles();
            if (files == null) {
                return;
            }

            for (File c : files) {
                delete(c);
            }
        }

        if (!f.delete()) {
            throw new FileNotFoundException("Failed to delete file: " + f);
        }
    }

    public static void closeQuietly(Closeable closeable) {
        try {
            if (closeable != null)
                closeable.close();
        } catch (IOException ignored) {

        }
    }

    @SuppressWarnings("unchecked")
    @NonNull
    public static HashMap<String, String> readMap(@NonNull String path) {
        File file = new File(path);

        Object ser = readSerializable(file);

        if (ser instanceof HashMap) {
            return (HashMap<String, String>) ser;
        }

        return new HashMap<>();
    }

    public static void writeMap(@NonNull String path, @NonNull HashMap<String, String> map) {
        File file = new File(path);

        if (map.isEmpty())
            return;

        writeSerializable(file, map);
    }

    @SuppressWarnings("unchecked")
    @NonNull
    @VisibleForTesting
    static ArrayList<String> readAllLines(@NonNull String path) {
        File file = new File(path);

        if (file.exists()) {
            Object ser = readSerializable(file);

            if (ser instanceof ArrayList) {
                return (ArrayList<String>) ser;
            }
        }

        return new ArrayList<>();
    }

    public static void writeAllLines(@NonNull String path, @NonNull ArrayList<String> lines) {
        File file = new File(path);

        if (file.exists())
            file.delete();

        if (lines.isEmpty())
            return;

        writeSerializable(file, lines);
    }

    public static void writeSerializable(@NonNull File file,
                                         @Nullable Serializable serializable) {
        if (file.exists())
            file.delete();

        if (serializable == null)
            return;

        FileOutputStream fout = null;
        ObjectOutputStream oout = null;
        try {
            fout = new FileOutputStream(file);
            oout = new ObjectOutputStream(fout);
            oout.writeObject(serializable);
            oout.reset();
        } catch (IOException e) {
            Log.e(TAG, "IOIOException", e);
        } finally {
            closeQuietly(oout);
            closeQuietly(fout);
        }
    }

    @SuppressWarnings("unchecked")
    @Nullable
    public static <T> T readSerializable(File file) {
        if (!file.exists())
            return null;

        FileInputStream fin = null;
        SafeObjectInputStream oin = null;
        try {
            fin = new FileInputStream(file);
            oin = new SafeObjectInputStream(fin, allowedClasses);
            return (T) oin.readObject();
        } catch (IOException e) {
            Log.e(TAG, "IOIOException", e);
        } catch (ClassNotFoundException e) {
            Log.e(TAG, "ClassNotFoundException", e);
        } catch (Exception e) {
            //TODO after implementing logging send this corrupted file to us for investigation
            Log.e(TAG, "cannot read serializable", e);
            try {
                FileUtility.delete(file);
            } catch (IOException ignored) {
            }
        } finally {
            closeQuietly(oin);
            closeQuietly(fin);
        }

        return null;
    }

    public static long size(@Nullable File file) {
        if (file == null || !file.exists())
            return 0;

        long length = 0;

        if (file.isDirectory()) {
            File[] children = file.listFiles();
            if (children != null && children.length > 0) {
                for (File child : children) {
                    length += size(child);
                }
            }

            return length;
        }

        return file.length();
    }

}
