package com.vungle.warren.utility;

import android.os.AsyncTask;

import java.io.File;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

public class AsyncFileUtils {

    /**
     * Since {@link File#exists} gives disk read violation for strict mode policy
     * this method does async file existence validation
     *
     * @param file     File
     * @param callback {@link FileExistCallback} callback to provide status of file.
     * @return {@link ExistenceOperation} object that can be used to cancel operation and remove
     * callback
     */
    public static ExistenceOperation isFileExistAsync(@Nullable final File file,
                                                      @NonNull final FileExistCallback callback) {
        FileExistenceTask task = new FileExistenceTask(file, callback);
        ExistenceOperation operation = new ExistenceOperation(task);
        task.execute();

        return operation;
    }

    /**
     * Async callback to check file existence.
     */
    public interface FileExistCallback {
        void status(boolean isExist);
    }

    private static class FileExistenceTask extends AsyncTask<Void, Void, Boolean> {

        private final File file;
        private FileExistCallback callback;

        public FileExistenceTask(@Nullable File file, @NonNull FileExistCallback callback) {
            this.file = file;
            this.callback = callback;
        }

        @Override
        protected Boolean doInBackground(Void... voids) {
            return file != null && file.exists();
        }

        @Override
        protected void onPostExecute(Boolean result) {
            super.onPostExecute(result);

            synchronized (this) {
                if (callback != null) {
                    callback.status(result);
                }
            }
        }

        private synchronized void clear() {
            callback = null;
        }
    }

    /**
     * Helper class to cancel running existence task
     */
    public static class ExistenceOperation {

        private final FileExistenceTask task;

        ExistenceOperation(@NonNull FileExistenceTask task) {
            this.task = task;
        }

        /**
         * Cancels currently running task and removes callback
         */
        public void cancel() {
            task.clear();
            task.cancel(true);
        }
    }
}
