package com.vungle.warren.ui.view;

import android.annotation.TargetApi;
import android.net.Uri;
import android.os.Build;
import android.text.TextUtils;
import android.util.Log;
import android.view.View;
import android.webkit.RenderProcessGoneDetail;
import android.webkit.WebResourceError;
import android.webkit.WebResourceRequest;
import android.webkit.WebView;
import android.webkit.WebViewClient;
import android.webkit.WebViewRenderProcess;
import android.webkit.WebViewRenderProcessClient;

import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;

import com.google.gson.JsonObject;
import com.vungle.warren.model.Advertisement;
import com.vungle.warren.model.Placement;

/**
 * A special web client we use to start MRAID ads and to process Vungle Postroll webpages. It injects
 * Android-specific implementations to functions that the webpage calls when the user clicks buttons.
 */
public class VungleWebClient extends WebViewClient implements WebViewAPI {

    public static final String TAG = VungleWebClient.class.getSimpleName();

    /**
     * The advertisement being played in the webview managed by this web client.
     */
    private Advertisement advertisement;

    /**
     * The placement in which the advertisement is being rendered.
     */
    private Placement placement;

    /**
     * The delegate which will handle incoming commands from
     */
    private MRAIDDelegate MRAIDDelegate;

    /**
     * Current data-gathering consent status.
     */
    private boolean collectConsent;

    /**
     * Copy of WebView , this is set only once it has been loaded
     */
    private WebView loadedWebView;

    private boolean ready;

    private String gdprTitle, gdprBody, gdprAccept, gdprDeny;

    private Boolean isViewable;
    /**
     * Error call to notify errors to presenter
     */
    private WebClientErrorHandler errorHandler;

    public VungleWebClient(Advertisement advertisement, Placement placement) {
        this.advertisement = advertisement;
        this.placement = placement;
    }

    /**
     * Provides the current data-gathering consent status to the webclient. This is used to determine
     * whether the MRAID bridge can collect certain information and to determine whether to prompt
     * the user for their consent in tracking their data. We also sent text for the GDPR Dialog in case
     * the dialog needs to be shown.
     *
     * @param collectedConsent <code>true</code> if the user has informed us about their preference,
     *                         <code>false</code> if we have not gathered consent in either direction.
     * @param title            - GDPR Title
     * @param message          - GDPR Message
     * @param accept           - GDPR Accept
     * @param deny             - GDPR Deny
     */
    @Override
    public void setConsentStatus(boolean collectedConsent, @Nullable String title,
                                 @Nullable String message, @Nullable String accept, @Nullable String deny) {
        this.collectConsent = collectedConsent;
        gdprTitle = title;
        gdprBody = message;
        gdprAccept = accept;
        gdprDeny = deny;
    }

    /**
     * Provide the web client with a delegate which will process the incoming commands from the
     * MRAID container.
     *
     * @param MRAIDDelegate
     */
    @Override
    public void setMRAIDDelegate(MRAIDDelegate MRAIDDelegate) {
        this.MRAIDDelegate = MRAIDDelegate;
    }

    @Override
    public boolean shouldOverrideUrlLoading(final WebView view, String url) {
        Log.d(TAG, "MRAID Command " + url);
        if (TextUtils.isEmpty(url)) {
            Log.e(TAG, "Invalid URL ");
            return false;
        }
        Uri uri = Uri.parse(url);
        if (uri.getScheme() != null) {
            String scheme = uri.getScheme();
            if (scheme.equals("mraid")) {
                /// MRAID Command intended for the SDK
                String command = uri.getHost();
                if (command != null) {
                    if ("propertiesChangeCompleted".equals(command) && !ready) {
                        /// Pass the URL for the assets to the webview. This can be handled by the web client.
                        final JsonObject mraidArgs = advertisement.createMRAIDArgs();
                        view.loadUrl("javascript:window.vungle.mraidBridge.notifyReadyEvent(" + mraidArgs + ")");
                        ready = true;
                    } else {
                        /// A command for the presenter of the advertisement, should be handled by the mraid
                        /// delegate, if any exists.
                        if (MRAIDDelegate != null) {
                            JsonObject args = new JsonObject();
                            for (String param : uri.getQueryParameterNames()) {
                                args.addProperty(param, uri.getQueryParameter(param));
                            }
                            if (MRAIDDelegate.processCommand(command, args)) {
                                view.loadUrl("javascript:window.vungle.mraidBridge.notifyCommandComplete()");
                            }
                        }
                    }
                    return true;
                } else {
                    return false;
                }
            } else if ("http".equalsIgnoreCase(scheme) || "https".equalsIgnoreCase(scheme)) {
                Log.d(TAG, "Open URL" + url);
                if (MRAIDDelegate != null) {
                    JsonObject args = new JsonObject();
                    args.addProperty("url", url);
                    MRAIDDelegate.processCommand("openNonMraid", args);
                }
                return true;
            }

        }
        return false;
    }

    @Override
    public void onPageFinished(WebView webView, String url) {
        super.onPageFinished(webView, url);

        switch (advertisement.getAdType()) {
            case Advertisement.TYPE_VUNGLE_LOCAL:
                /// Add the bridge function, actionClicked(String)
                String inject = "javascript:function actionClicked(action){" +
                        "Android.performAction(action);" +
                        "};";
                webView.loadUrl(inject);
                break;
            case Advertisement.TYPE_VUNGLE_MRAID:
                loadedWebView = webView;
                loadedWebView.setVisibility(View.VISIBLE);
                notifyPropertiesChange(true);
                break;
            default:
                throw new IllegalArgumentException("Unknown Client Type!");
        }

        /* The native and MRAID both uses the webview. let us catch the issue with both of them */
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
            webView.setWebViewRenderProcessClient(new VungleWebViewRenderProcessClient(errorHandler));
        }
    }

    /**
     * Notify the webview about the current device properties. This includes the screen size and
     * metadata about the current view. This should be called when the device rotates or an MRAID
     * webview is being initialized.
     */
    @Override
    public void notifyPropertiesChange(boolean skipCmdQueue) {
        if (loadedWebView != null) {
            /// Set the device properties. Does this get called again once the JS has executed?
            JsonObject screenJson = new JsonObject();

            JsonObject size = new JsonObject();
            size.addProperty("width", loadedWebView.getWidth());
            size.addProperty("height", loadedWebView.getHeight());

            JsonObject position = new JsonObject();
            position.addProperty("x", 0);
            position.addProperty("y", 0);
            position.addProperty("width", loadedWebView.getWidth());
            position.addProperty("height", loadedWebView.getHeight());

            JsonObject supports = new JsonObject();
            supports.addProperty("sms", false);
            supports.addProperty("tel", false);
            supports.addProperty("calendar", false);
            supports.addProperty("storePicture", false);
            supports.addProperty("inlineVideo", false);

            screenJson.add("maxSize", size);
            screenJson.add("screenSize", size);
            screenJson.add("defaultPosition", position);
            screenJson.add("currentPosition", position);
            screenJson.add("supports", supports);
            screenJson.addProperty("placementType", advertisement.getTemplateType());
            if (isViewable != null) {
                screenJson.addProperty("isViewable", isViewable);
            }
            screenJson.addProperty("os", "android");
            screenJson.addProperty("osVersion", Integer.toString(Build.VERSION.SDK_INT));
            screenJson.addProperty("incentivized", placement.isIncentivized());
            screenJson.addProperty("enableBackImmediately", advertisement.getShowCloseDelay(placement.isIncentivized()) == 0);
            screenJson.addProperty("version", "1.0");

            if (collectConsent) {
                /// We need to collect consent status
                screenJson.addProperty("consentRequired", true);
                screenJson.addProperty("consentTitleText", gdprTitle);
                screenJson.addProperty("consentBodyText", gdprBody);
                screenJson.addProperty("consentAcceptButtonText", gdprAccept);
                screenJson.addProperty("consentDenyButtonText", gdprDeny);
            } else {
                screenJson.addProperty("consentRequired", false);
            }

            Log.d(TAG, "loadJs" + "javascript:window.vungle.mraidBridge.notifyPropertiesChange(" +
                    screenJson + "," + skipCmdQueue + ")");
            loadedWebView.loadUrl("javascript:window.vungle.mraidBridge.notifyPropertiesChange(" +
                    screenJson + "," + skipCmdQueue + ")");
        }
    }

    /**
     * Pass on updated MRAID properties to MRAID bridge
     *
     * @param isViewable
     */
    @Override
    public void setAdVisibility(boolean isViewable) {
        this.isViewable = isViewable;
        notifyPropertiesChange(false);
    }

    @Override
    public void setErrorHandler(WebClientErrorHandler errorHandler) {
        this.errorHandler = errorHandler;
    }

    //This method is deprecated but called by devices under API 23
    //may also be invoked by devices api 23+ so we need to check the we don't
    @Override
    public void onReceivedError(WebView view, int errorCode, String description, String failingUrl) {
        super.onReceivedError(view, errorCode, description, failingUrl);
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.M) {
            Log.e(TAG, "Error desc " + description);
            Log.e(TAG, "Error for URL " + failingUrl);
            String errorDesc = failingUrl + " " + description;

            //Notify error
            if (errorHandler != null) {
                errorHandler.onReceivedError(errorDesc);
            }
        }
    }

    @Override
    public void onReceivedError(WebView view, WebResourceRequest request, WebResourceError error) {
        super.onReceivedError(view, request, error);
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            Log.e(TAG, "Error desc " + error.getDescription().toString());
            Log.e(TAG, "Error for URL " + request.getUrl().toString());
            String errorDesc = request.getUrl().toString() + " " + error.getDescription().toString();

            //Notify error
            if (errorHandler != null) {
                errorHandler.onReceivedError(errorDesc);
            }
        }
    }

    @Override
    @TargetApi(Build.VERSION_CODES.O)
    public boolean onRenderProcessGone(WebView view, RenderProcessGoneDetail detail) {
        Log.w(TAG, "onRenderProcessGone url: " + view.getUrl() + ", "
                + " did crash: " + detail.didCrash());

        //Invalidate the local webview reference as the same is used with the notifyPropertiesChange()
        loadedWebView = null;

        if (errorHandler != null)
            return errorHandler.onWebRenderingProcessGone(view, detail.didCrash());

        return super.onRenderProcessGone(view, detail);
    }

    @RequiresApi(Build.VERSION_CODES.Q)
    static class VungleWebViewRenderProcessClient extends WebViewRenderProcessClient {

        WebClientErrorHandler errorHandler;

        VungleWebViewRenderProcessClient(WebClientErrorHandler errorHandler) {
            this.errorHandler = errorHandler;
        }
        /*
        Detect the renderer hang/unresponsiveness for API 29+/Android Q/10+
        https://developer.android.com/reference/android/webkit/WebViewRenderProcessClient

        Action#1:
        We can terminate the renderer here as we can not recover the Ad from the current situation
        Restarting the half played Ad will also conflict with the third party trackers and other events
        sent to Vungle servers. But termination, without corresponding implementation of onRenderProcessGone()
        by all the Webview clients will crash the app. On termination of the renderer, the onRenderProcessGone()
        is called where we do the clean-up. Finally decided to take the action#2.

        Action#2:
        Detach and destroy the webview, do the clean-up and close the Ad.

        */
        @Override
        public void onRenderProcessUnresponsive(WebView webView, WebViewRenderProcess webViewRenderProcess) {
            Log.w(TAG, "onRenderProcessUnresponsive(Title = " + webView.getTitle() + ", URL = " +
                    webView.getOriginalUrl() + ", (webViewRenderProcess != null) = " + (webViewRenderProcess != null));

            if (errorHandler != null) {
                errorHandler.onRenderProcessUnresponsive(webView, webViewRenderProcess);
            }

        }

        @Override
        public void onRenderProcessResponsive(WebView webView, WebViewRenderProcess webViewRenderProcess) {
            /* DO NOTHING */

        }
    }

}
