package com.vungle.warren.ui.view;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.IntentFilter;
import android.graphics.Color;
import android.os.Build;
import android.os.Bundle;
import android.util.Log;
import android.util.Pair;
import android.view.View;
import android.view.ViewManager;
import android.view.ViewParent;
import android.webkit.WebView;
import android.widget.RelativeLayout;

import com.vungle.warren.AdConfig;
import com.vungle.warren.AdvertisementPresentationFactory;
import com.vungle.warren.PresentationFactory;
import com.vungle.warren.VungleNativeAd;
import com.vungle.warren.error.VungleException;
import com.vungle.warren.ui.CloseDelegate;
import com.vungle.warren.ui.JavascriptBridge;
import com.vungle.warren.ui.contract.AdContract;
import com.vungle.warren.ui.contract.WebAdContract;
import com.vungle.warren.utility.ExternalRouter;

import java.util.concurrent.atomic.AtomicReference;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.localbroadcastmanager.content.LocalBroadcastManager;

/**
 * Samsung Flexfeed is a view returned by Vungle.getNativeAd(). It extends WebView
 * and fills the width and height of the parent container. This method will return
 * null when Vungle is not initialized.
 */
public class VungleNativeView extends WebView implements WebAdContract.WebAdView, VungleNativeAd {

    private static final String TAG = VungleNativeView.class.getName();
    /**
     * TODO: Missing javadoc
     */
    private WebAdContract.WebAdPresenter presenter;
    private BroadcastReceiver broadcastReceiver;
    private final AdContract.AdvertisementPresenter.EventListener listener;
    private final String placementID;
    private final AdConfig config;

    /**
     * How long we wait for an ad to start playing before we end it and call error
     * Mostly likely from an invalid resource or invalid url. We'll have to be more particular in the future
     * and narrow down exactly why ads are not showing. If this is the case we should send error events
     * upstream to server
     * <p>
     * In testing time to start video currently ranges anywhere from 500ms to 2500ms on Pixel 2
     */

    PresentationFactory presenterFactory;
    private AtomicReference<Boolean> isAdVisible = new AtomicReference<>();

    @Override
    public void onResume() {
        super.onResume();

        //will need to call start on video dom object
        Log.d(TAG, "Resuming Flex");

        setAdVisibility(true);
    }

    @Override
    public void onPause() {
        super.onPause();
        setAdVisibility(false);
    }

    /**
     * TODO: Missing javadoc
     *
     * @param context
     * @param listener
     */
    public VungleNativeView(@NonNull Context context,
                            @NonNull final String placementID,
                            @Nullable final AdConfig config,
                            @NonNull PresentationFactory presentationFactory,
                            @NonNull final AdContract.AdvertisementPresenter.EventListener listener) {
        super(context);
        this.listener = listener;
        this.placementID = placementID;
        this.config = config;
        this.presenterFactory = presentationFactory;

        //set Hardware Acceleration
        //may negatively affect perfomance for devices under API 19
        //should for those devices: setLayerType(View.LAYER_TYPE_SOFTWARE, null);
        setLayerType(View.LAYER_TYPE_HARDWARE, null);

        // Set background transparent.
        setBackgroundColor(Color.TRANSPARENT);
    }

    private void prepare(Bundle savedInstanceState) {

        WebSettingsUtils.applyDefault(this);

        addJavascriptInterface(new JavascriptBridge(presenter), "Android");

        RelativeLayout.LayoutParams params = new RelativeLayout.LayoutParams(-1, -1);
        setLayoutParams(params);

        //on certain devices and OS pairings (Samsung S4 etc..., Autoplay does not work)
        //https://bugs.chromium.org/p/chromium/issues/detail?id=159336
        //by design autoplay is disabled
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.JELLY_BEAN_MR1) {
            //loadUrl("javascript:document.getElementsByTagName('video')[0]; myvideo.start();");
            //force a video start call to be made on older devices
            setAdVisibility(true);
        } else {
            // This breaks audiofocus
            getSettings().setMediaPlaybackRequiresUserGesture(false);
        }
    }

    @Override
    public void finishDisplayingAd() {
        finishDisplayingAdInternal(true);
    }

    public void finishDisplayingAdInternal(boolean isFinishByExternalApi) {
        if (presenter != null) {
            int flag = AdContract.AdStopReason.IS_AD_FINISHING
                    | (isFinishByExternalApi ? AdContract.AdStopReason.IS_AD_FINISHED_BY_API : 0);

            presenter.stop(flag);
        } else if (presenterFactory != null) {
            presenterFactory.destroy();
            presenterFactory = null;
            listener.onError(new VungleException(VungleException.OPERATION_CANCELED), placementID);
        }

        destroyAdView();
    }

    @Override
    public View renderNativeView() {
        return this;
    }

    @Override
    public void setAdVisibility(boolean isVisible) {
        if (presenter != null) {
            presenter.setAdVisibility(isVisible);
        } else {
            isAdVisible.set(isVisible);
        }
    }

    @Override
    public void setPresenter(@NonNull WebAdContract.WebAdPresenter presenter) {
        //no-op
    }

    @Override
    public void setOrientation(int orientation) {
        //no-op
    }

    @Override
    public void showWebsite(@NonNull String url) {
        loadUrl(url);
    }

    @Override
    public String getWebsiteUrl() {
        return getUrl();
    }

    @Override
    public void close() {
        if (presenter != null) {
            if (presenter.handleExit(null)) {
                finishDisplayingAdInternal(false);
            }
        } else if (presenterFactory != null) {
            presenterFactory.destroy();
            presenterFactory = null;
            listener.onError(new VungleException(VungleException.OPERATION_CANCELED), placementID);
        }
    }

    @Override
    public void destroyAdView() {
        removeJavascriptInterface("Android");
        loadUrl("about:blank");
    }

    @Override
    public boolean hasWebView() {
        return true;
    }

    @Override
    public void showCloseButton() {
        /// NativeView does not have a close button
        throw new UnsupportedOperationException("VungleNativeView does not implement a close button");
    }

    @Override
    public void open(@NonNull String url) {
        Log.d(TAG, "Opening " + url);
        if (!ExternalRouter.launch(url, getContext())) {
            Log.e(TAG, "Cannot open url " + url);
        }
    }

    @Override
    public void showDialog(@Nullable String dialogTitle,
                           @Nullable String dialogBody,
                           @NonNull String dialogContinue,
                           @NonNull String dialogClose,
                           @Nullable DialogInterface.OnClickListener responseListener) {
        /// NativeView does not implement the dialog functionality
        throw new UnsupportedOperationException("VungleNativeView does not implement a dialog.");
    }

    @Override
    public void setImmersiveMode() {
        //no-op
    }

    @Override
    public void resumeWeb() {
        onResume();
    }

    @Override
    public void pauseWeb() {
        onPause();
    }

    @Override
    public void removeWebView() {
        ViewParent viewParent = getParent();

        if (viewParent instanceof ViewManager) {
            ((ViewManager) viewParent).removeView(this);
        }
    }

    @Override
    public void updateWindow(boolean isFlexView) {
        //NO-OP this is only for FlexView which happen inside VungleActivity
    }

    @Override
    public void setVisibility(boolean isVisible) {
        int setVisibilityState = isVisible ? View.VISIBLE : View.INVISIBLE;
        setVisibility(setVisibilityState);
    }

    @Override
    protected void onAttachedToWindow() {
        super.onAttachedToWindow();

        if (presenterFactory != null && presenter == null) {
            presenterFactory.getNativeViewPresentation(placementID, config,
                    new CloseDelegate() {
                        @Override
                        public void close() {
                            finishDisplayingAdInternal(false);
                        }
                    },
                    new AdvertisementPresentationFactory.ViewCallback() {
                        @Override
                        public void onResult(@Nullable Pair<WebAdContract.WebAdPresenter, VungleWebClient> result, @Nullable VungleException  error) {
                            presenterFactory = null;
                            if (result == null || error != null) {
                                if (listener != null) {
                                    listener.onError(error != null ? error : new VungleException(VungleException.AD_UNABLE_TO_PLAY), placementID);
                                }
                                return;
                            }

                            presenter = result.first;
                            VungleWebClient webViewClient = result.second;
                            setWebViewClient(webViewClient);

                            presenter.setEventListener(listener);
                            presenter.attach(VungleNativeView.this, null);
                            prepare(null);
                            if (isAdVisible.get() != null) {
                                setAdVisibility(isAdVisible.get());
                            }
                        }
                    });
        }

        broadcastReceiver = new BroadcastReceiver() {
            @Override
            public void onReceive(Context context, Intent intent) {
                String command = intent.getStringExtra(AdContract.AdvertisementBus.COMMAND);
                if (AdContract.AdvertisementBus.STOP_ALL.equalsIgnoreCase(command)) {
                    finishDisplayingAdInternal(false);
                }
            }
        };

        LocalBroadcastManager.getInstance(getContext()).registerReceiver(broadcastReceiver, new IntentFilter(AdContract.AdvertisementBus.ACTION));

        resumeWeb();
    }

    @Override
    public void onWindowFocusChanged(boolean hasWindowFocus) {
        super.onWindowFocusChanged(hasWindowFocus);
        setAdVisibility(hasWindowFocus);
    }

    @Override
    protected void onDetachedFromWindow() {
        LocalBroadcastManager.getInstance(getContext()).unregisterReceiver(broadcastReceiver);
        super.onDetachedFromWindow();
        if (presenterFactory != null) {
            presenterFactory.destroy();
        }

        pauseWeb();
    }
}