package com.vungle.warren.ui.view;

import android.content.Context;
import android.content.ContextWrapper;
import android.content.res.Configuration;
import android.content.res.Resources;
import android.graphics.Bitmap;
import android.graphics.Color;
import android.media.MediaPlayer;
import android.net.Uri;
import android.os.Build;
import androidx.annotation.IntDef;
import androidx.annotation.Nullable;
import android.util.DisplayMetrics;
import android.util.Log;
import android.util.TypedValue;
import android.view.GestureDetector;
import android.view.Gravity;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewTreeObserver;
import android.view.Window;
import android.view.WindowManager;
import android.webkit.WebView;
import android.webkit.WebViewClient;
import android.widget.ImageView;
import android.widget.ProgressBar;
import android.widget.RelativeLayout;
import android.widget.VideoView;

import com.vungle.warren.ui.JavascriptBridge;
import com.vungle.warren.utility.ViewUtility;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.util.HashMap;
import java.util.Map;

public class FullAdWidget extends RelativeLayout {
    /**
     * Flexview Video Ratio
     */
    protected static final double NINE_BY_SIXTEEN_ASPECT_RATIO = 0.5625;


    private static final String TAG = FullAdWidget.class.getSimpleName();

    private Map<View, Integer> viewToId = new HashMap<>();
    private final LayoutParams matchParentLayoutParams;
    private final Window window;

    ///!
    public final VideoView videoView;

    /**
     * Created this layout since videoview does not responding to onClick
     */
    private final RelativeLayout videoViewContainer;
    ///!

    @Nullable
    private WebView webView;
    private final ProgressBar progressBar;
    private final ImageView muteButton;
    private final ImageView closeButton;
    private final ImageView ctaOverlay;
    private final ImageView privacyOverlay;

    private OnItemClickListener onClickProxy;
    private MediaPlayer.OnPreparedListener onPreparedListener;
    private MediaPlayer.OnErrorListener onErrorListener;
    private MediaPlayer.OnCompletionListener onCompletionListener;
    private int startPosition;

    private GestureDetector gestureDetector;

    public FullAdWidget(Context context, Window window) throws InstantiationException {
        super(context);
        this.window = window;

        Resources resources = getResources();
        matchParentLayoutParams = new RelativeLayout.LayoutParams(
                RelativeLayout.LayoutParams.MATCH_PARENT,
                RelativeLayout.LayoutParams.MATCH_PARENT);

        setLayoutParams(matchParentLayoutParams);

        videoView = new VideoView(new AudioContextWrapper(context));
        RelativeLayout.LayoutParams videoViewLayoutParams = new RelativeLayout.LayoutParams(
                RelativeLayout.LayoutParams.MATCH_PARENT,
                RelativeLayout.LayoutParams.MATCH_PARENT);
        videoViewLayoutParams.addRule(RelativeLayout.CENTER_IN_PARENT);
        videoView.setLayoutParams(videoViewLayoutParams);

        videoViewContainer = new RelativeLayout(context);
        videoViewContainer.setTag("videoViewContainer");
        videoViewContainer.setLayoutParams(matchParentLayoutParams);
        videoViewContainer.addView(videoView, videoViewLayoutParams);
        addView(videoViewContainer, matchParentLayoutParams);

        gestureDetector = new GestureDetector(context, singleTapOnVideoListener);

        webView = ViewUtility.getWebView(context);
        webView.setLayoutParams(matchParentLayoutParams);
        webView.setTag("webView");
        addView(webView, matchParentLayoutParams);

        //todo test if android.attr will affect any jar integration
        progressBar = new ProgressBar(context, null, android.R.attr.progressBarStyleHorizontal);
        int progressBarHeight = (int) TypedValue.applyDimension(
                TypedValue.COMPLEX_UNIT_DIP,
                4,
                resources.getDisplayMetrics());
        RelativeLayout.LayoutParams progressBarLayoutParams = new RelativeLayout.LayoutParams(
                RelativeLayout.LayoutParams.MATCH_PARENT,
                progressBarHeight);
        progressBarLayoutParams.addRule(RelativeLayout.ALIGN_PARENT_BOTTOM);
        progressBar.setLayoutParams(progressBarLayoutParams);
        progressBar.setMax(100);
        progressBar.setIndeterminate(false);
        progressBar.setVisibility(View.INVISIBLE);
        addView(progressBar);

        int imageViewDimension = (int) TypedValue.applyDimension(
                TypedValue.COMPLEX_UNIT_DIP,
                24,
                resources.getDisplayMetrics());
        RelativeLayout.LayoutParams imageViewLayoutParams = new RelativeLayout.LayoutParams(
                imageViewDimension,
                imageViewDimension);
        int imageViewMargin = (int) TypedValue.applyDimension(
                TypedValue.COMPLEX_UNIT_DIP,
                12,
                resources.getDisplayMetrics());
        imageViewLayoutParams.setMargins(imageViewMargin, imageViewMargin, imageViewMargin, imageViewMargin);
        muteButton = new ImageView(context);
        muteButton.setImageBitmap(ViewUtility.getBitmap(ViewUtility.Asset.unMute, context));
        muteButton.setLayoutParams(imageViewLayoutParams);
        muteButton.setVisibility(View.GONE);
        addView(muteButton);

        RelativeLayout.LayoutParams closeLayoutParams = new RelativeLayout.LayoutParams(
                imageViewDimension,
                imageViewDimension);
        closeLayoutParams.setMargins(imageViewMargin, imageViewMargin, imageViewMargin, imageViewMargin);
        closeButton = new ImageView(context);
        closeButton.setTag("closeButton");
        closeButton.setImageBitmap(ViewUtility.getBitmap(ViewUtility.Asset.close, context));
        closeLayoutParams.addRule(RelativeLayout.ALIGN_PARENT_RIGHT);
        closeButton.setLayoutParams(closeLayoutParams);
        closeButton.setVisibility(View.GONE);
        addView(closeButton);

        RelativeLayout.LayoutParams ctaOverlayLayoutParams = new RelativeLayout.LayoutParams(
                imageViewDimension,
                imageViewDimension);
        ctaOverlayLayoutParams.addRule(RelativeLayout.ALIGN_PARENT_BOTTOM);
        ctaOverlayLayoutParams.addRule(RelativeLayout.ALIGN_PARENT_RIGHT);
        ctaOverlayLayoutParams.setMargins(imageViewMargin, imageViewMargin, imageViewMargin, imageViewMargin);
        ctaOverlay = new ImageView(context);
        ctaOverlay.setTag("ctaOverlay");
        ctaOverlay.setLayoutParams(ctaOverlayLayoutParams);
        ctaOverlay.setImageBitmap(ViewUtility.getBitmap(ViewUtility.Asset.cta, getContext()));
        ctaOverlay.setVisibility(View.GONE);
        addView(ctaOverlay);

        RelativeLayout.LayoutParams privacyLayoutParams = new RelativeLayout.LayoutParams(
                imageViewDimension,
                imageViewDimension);
        privacyLayoutParams.addRule(RelativeLayout.ALIGN_PARENT_BOTTOM);
        privacyLayoutParams.addRule(RelativeLayout.ALIGN_PARENT_LEFT);
        privacyLayoutParams.setMargins(imageViewMargin, imageViewMargin, imageViewMargin, imageViewMargin);
        privacyOverlay = new ImageView(context);
        privacyOverlay.setLayoutParams(privacyLayoutParams);
        privacyOverlay.setVisibility(View.GONE);
        addView(privacyOverlay);

        bindListeners();
        prepare();
    }

    //pass video click to proxy
    private GestureDetector.SimpleOnGestureListener singleTapOnVideoListener = new GestureDetector.SimpleOnGestureListener() {

        @Override
        public boolean onSingleTapConfirmed(MotionEvent e) {
            proxyClickListener.onClick(videoViewContainer);
            return true;
        }

    };

    private void bindListeners() {
        bindListener(closeButton, ViewEvent.CLOSE_CLICK);
        bindListener(ctaOverlay, ViewEvent.CTA_CLICK);
        bindListener(muteButton, ViewEvent.MUTE_CLICK);
        bindListener(privacyOverlay, ViewEvent.PRIVACY_CLICK);

        viewToId.put(videoViewContainer, ViewEvent.VIDEO_CLICK);

        //Detect only single taps on the video container and not clicks and swipes on drawers
        videoViewContainer.setOnTouchListener(new OnTouchListener() {
            @Override
            public boolean onTouch(View v, MotionEvent event) {
                gestureDetector.onTouchEvent(event);
                return true;
            }
        });

        videoView.setOnPreparedListener(new MediaPlayer.OnPreparedListener() {
            @Override
            public void onPrepared(MediaPlayer mp) {
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                    //avoid seeking with MediaPlayer.SEEK_PREVIOUS_SYNC inside VideoView implementation
                    //check https://android.googlesource.com/platform/frameworks/base/+/master/core/java/android/widget/VideoView.java#508
                    mp.seekTo(startPosition, MediaPlayer.SEEK_CLOSEST);
                }
                if (onPreparedListener != null) {
                    onPreparedListener.onPrepared(mp);
                }

                muteButton.setVisibility(View.VISIBLE); //Showing muteButton after media player initialized
            }
        });

        videoView.setOnErrorListener(new MediaPlayer.OnErrorListener() {
            @Override
            public boolean onError(MediaPlayer mp, int what, int extra) {
                if (onErrorListener != null) {
                    return onErrorListener.onError(mp, what, extra);
                }
                return false;
            }
        });

        videoView.setOnCompletionListener(new MediaPlayer.OnCompletionListener() {
            @Override
            public void onCompletion(MediaPlayer mp) {
                if (onCompletionListener != null) {
                    onCompletionListener.onCompletion(mp);
                }

                muteButton.setEnabled(false);
            }
        });
    }

    private void bindListener(View v, @ViewEvent int event) {
        viewToId.put(v, event);
        v.setOnClickListener(proxyClickListener);
    }

    private void prepare() {
        if (webView != null) {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
                // This breaks audiofocus
                webView.getSettings().setMediaPlaybackRequiresUserGesture(false);
            }
            webView.setVisibility(View.GONE);
        }

        videoViewContainer.setVisibility(View.GONE);
    }

    //TODO: REWORK, this is here for temp needs
    public void linkWebView(WebViewClient webViewClient, JavascriptBridge javascriptBridge) {
        if (webView == null)
            return;

        WebSettingsUtils.applyDefault(webView);
        webView.setWebViewClient(webViewClient);
        webView.addJavascriptInterface(javascriptBridge, "Android");
    }

    public void showWebsite(String url) {
        if (webView == null)
            return;

        Log.d(TAG, "loadJs: " + url);
        webView.loadUrl(url);
        webView.setVisibility(View.VISIBLE);
        videoViewContainer.setVisibility(View.GONE);
        videoViewContainer.setOnClickListener(null);
        progressBar.setVisibility(View.GONE);
        closeButton.setVisibility(View.GONE);
        muteButton.setVisibility(View.GONE);
        ctaOverlay.setVisibility(View.GONE);
        privacyOverlay.setVisibility(View.GONE);
    }

    public void setCtaEnabled(boolean isCtaIconEnabled) {
        ctaOverlay.setVisibility(isCtaIconEnabled ? View.VISIBLE : View.GONE);
    }

    public boolean isVideoPlaying() {
        return videoView.isPlaying();
    }

    public void stopPlayback() {
        videoView.stopPlayback();
    }

    public void pausePlayback() {
        videoView.pause();
    }

    public boolean startPlayback(int position) {
        if (!videoView.isPlaying()) {
            //if dialog is not showing or null then resume else...
            videoView.requestFocus();
            startPosition = position;
            if (Build.VERSION.SDK_INT < Build.VERSION_CODES.O) {
                videoView.seekTo(startPosition);
            }
            videoView.start();
        }

        return videoView.isPlaying();
    }

    ViewTreeObserver.OnGlobalLayoutListener immersiveModeListener = new ViewTreeObserver.OnGlobalLayoutListener() {

        @Override
        public void onGlobalLayout() {
            removeImmersiveModeListener();

            // Set the IMMERSIVE flag.
            // Set the content to appear under the system bars so that the content
            // doesn't resize when the system bars hide and show.
            window.getDecorView().setSystemUiVisibility(
                    View.SYSTEM_UI_FLAG_LAYOUT_STABLE
                            | View.SYSTEM_UI_FLAG_LAYOUT_HIDE_NAVIGATION
                            | View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN
                            | View.SYSTEM_UI_FLAG_HIDE_NAVIGATION       // hide nav bar
                            | View.SYSTEM_UI_FLAG_FULLSCREEN            // hide status bar
                            | View.SYSTEM_UI_FLAG_IMMERSIVE_STICKY);
        }
    };

    public void setImmersiveMode() {
        getViewTreeObserver().addOnGlobalLayoutListener(immersiveModeListener);
    }

    private void removeImmersiveModeListener() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
            getViewTreeObserver().removeOnGlobalLayoutListener(immersiveModeListener);
        } else {
            getViewTreeObserver().removeGlobalOnLayoutListener(immersiveModeListener);
        }
    }

    public void updateWindow(boolean isFlexView) {
        if (isFlexView) {
            //need to get screen size every time phone rotates
            DisplayMetrics metrics = new DisplayMetrics();
            window.getWindowManager().getDefaultDisplay().getMetrics(metrics);
            int heightPixels = metrics.heightPixels;
            int widthPixels = metrics.widthPixels;

            int orientation = getResources().getConfiguration().orientation;

            RelativeLayout.LayoutParams params = null;
            if (orientation == Configuration.ORIENTATION_PORTRAIT) {
                //Width is fixed equal to size of display and height only needs to be maintaining 16:9 aspect ratio

                window.setGravity(Gravity.BOTTOM | Gravity.LEFT);
                window.setLayout(widthPixels, (int) Math.round(widthPixels * NINE_BY_SIXTEEN_ASPECT_RATIO));

                params = new RelativeLayout.LayoutParams(widthPixels, (int) (widthPixels * NINE_BY_SIXTEEN_ASPECT_RATIO));
            } else if (orientation == Configuration.ORIENTATION_LANDSCAPE) {
                window.setLayout((int) Math.round(heightPixels * NINE_BY_SIXTEEN_ASPECT_RATIO), heightPixels);
                window.setGravity(Gravity.BOTTOM | Gravity.RIGHT);
                params = new RelativeLayout.LayoutParams((int) Math.round(heightPixels * NINE_BY_SIXTEEN_ASPECT_RATIO), heightPixels);
                params.addRule(RelativeLayout.ALIGN_PARENT_RIGHT, RelativeLayout.TRUE);
            }

            if (webView != null) {
                webView.setLayoutParams(params);
            }
            // Pass touch events to the background activity
            window.addFlags(WindowManager.LayoutParams.FLAG_NOT_TOUCH_MODAL | WindowManager.LayoutParams.FLAG_LAYOUT_IN_SCREEN);

        } else {
            /// Enter fullscreen mode
            window.setFlags(WindowManager.LayoutParams.FLAG_FULLSCREEN,
                    WindowManager.LayoutParams.FLAG_FULLSCREEN);
            window.getDecorView().setBackgroundColor(Color.BLACK);
        }
    }

    public int getCurrentVideoPosition() {
        return videoView.getCurrentPosition();
    }

    public int getVideoDuration() {
        return videoView.getDuration();
    }

    private View.OnClickListener proxyClickListener = new OnClickListener() {
        @Override
        public void onClick(View v) {
            if (onClickProxy != null) {
                onClickProxy.onItemClicked(matchView(v));
            }
        }
    };

    public void setOnItemClickListener(OnItemClickListener proxyClickListener) {
        this.onClickProxy = proxyClickListener;
    }

    private int matchView(View v) {
        Integer id = viewToId.get(v);
        return id == null ? -1 : id;
    }

    public void setMuted(boolean muted) {
        final Bitmap muteBitmap = ViewUtility.getBitmap(ViewUtility.Asset.mute, getContext());
        final Bitmap unMuteBitmap = ViewUtility.getBitmap(ViewUtility.Asset.unMute, getContext());

        muteButton.setImageBitmap(muted ? muteBitmap : unMuteBitmap);
    }

    public void playVideo(Uri uri, int videoPosition) {
        videoViewContainer.setVisibility(View.VISIBLE);
        videoView.setVideoURI(uri);

        /// Load the mute icon and set the state based on the settings
        final Bitmap privacyBitmap = ViewUtility.getBitmap(ViewUtility.Asset.privacy, getContext());

        privacyOverlay.setImageBitmap(privacyBitmap);
        privacyOverlay.setVisibility(View.VISIBLE);

        progressBar.setVisibility(View.VISIBLE);
        progressBar.setMax(videoView.getDuration());

        startPlayback(videoPosition);
    }

    public void setOnPreparedListener(MediaPlayer.OnPreparedListener preapredListener) {
        onPreparedListener = preapredListener;
    }

    public void setOnErrorListener(MediaPlayer.OnErrorListener errorListener) {
        onErrorListener = errorListener;
    }

    public void setOnCompletionListener(MediaPlayer.OnCompletionListener completionListener) {
        onCompletionListener = completionListener;
    }

    public void release() {
        removeWebView();
        videoView.stopPlayback();
        videoView.setOnCompletionListener(null);
        videoView.setOnErrorListener(null);
        videoView.setOnPreparedListener(null);
        videoView.suspend();
    }

    public void pauseWeb() {
        if (webView != null) {
            webView.onPause();
        }
        //To avoid leak, trying to remove if still added
        removeImmersiveModeListener();
    }

    public void resumeWeb() {
        if (webView != null) {
            webView.onResume();
        }
    }

    public String getUrl() {
        return webView == null ? null : webView.getUrl();
    }

    public void showCloseButton(boolean visible) {
        closeButton.setVisibility(visible ? View.VISIBLE : View.GONE);
    }

    public void setProgress(int position, float duration) {
        progressBar.setMax((int) duration);
        progressBar.setProgress(position);
    }

    public void removeWebView() {
        if (webView == null)
            return;

        webView.stopLoading();
        webView.removeJavascriptInterface("Android");
        webView.setWebViewClient(null);
        webView.setWebChromeClient(null);
        webView.loadUrl("about:blank");

        //WebView.destroy() causes crash on 4.4.x if WebView is not removed from parent
        removeView(webView);
        webView.removeAllViews();
        webView.destroy();
        webView = null;
    }

    public boolean hasWebView() {
        return webView != null;
    }

    public interface OnItemClickListener {

        void onItemClicked(@ViewEvent int code);

    }

    //Prevents leak if stopPlayback()->abandonFocus() was not called
    public static class AudioContextWrapper extends ContextWrapper {

        public AudioContextWrapper(Context base) {
            super(base);
        }

        @Override
        public Object getSystemService(String name) {
            if (Context.AUDIO_SERVICE.equals(name)) {
                return getApplicationContext().getSystemService(name);
            }
            return super.getSystemService(name);
        }
    }

    @IntDef(value = {ViewEvent.CLOSE_CLICK, ViewEvent.CTA_CLICK, ViewEvent.MUTE_CLICK, ViewEvent.PRIVACY_CLICK, ViewEvent.VIDEO_CLICK})
    @Retention(RetentionPolicy.SOURCE)
    public @interface ViewEvent {
        int CLOSE_CLICK = 1;
        int CTA_CLICK = 2;
        int MUTE_CLICK = 3;
        int PRIVACY_CLICK = 4;
        int VIDEO_CLICK = 5;
    }
}
