package com.vungle.warren.ui.view;

import android.app.AlertDialog;
import android.app.Dialog;
import android.content.Context;
import android.content.DialogInterface;
import android.os.Handler;
import android.os.Looper;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import android.text.TextUtils;
import android.util.Log;
import android.view.ContextThemeWrapper;

import com.vungle.warren.ui.CloseDelegate;
import com.vungle.warren.ui.OrientationDelegate;
import com.vungle.warren.ui.contract.AdContract;
import com.vungle.warren.utility.ExternalRouter;

import java.util.concurrent.atomic.AtomicReference;

/**
 * Created by Alexander Semenov on 10-Aug-18.
 */

public abstract class BaseAdView<T extends AdContract.AdvertisementPresenter> implements AdContract.AdView<T> {

    private final OrientationDelegate orientationDelegate;
    private final CloseDelegate closeDelegate;
    protected Handler handler = new Handler(Looper.getMainLooper());
    protected final String TAG = getClass().getSimpleName();
    protected final FullAdWidget view;
    protected final Context context;
    protected Dialog currentDialog;

    public BaseAdView(@NonNull Context context,
                      @NonNull FullAdWidget view,
                      @NonNull OrientationDelegate orientationDelegate,
                      @NonNull CloseDelegate closeDelegate) {
        this.view = view;
        this.context = context;
        this.orientationDelegate = orientationDelegate;
        this.closeDelegate = closeDelegate;
    }


    @Override
    public void setOrientation(int orientation) {
        orientationDelegate.setOrientation(orientation);
    }

    @Override
    public String getWebsiteUrl() {
        return view.getUrl();
    }

    @Override
    public void close() {
        closeDelegate.close();
    }

    @Override
    public void destroyAdView() {
        view.release();
    }

    @Override
    public void showCloseButton() {
        view.showCloseButton(true);
    }

    @Override
    public void open(@NonNull String url) {
        Log.d(TAG, "Opening " + url);
        if (!ExternalRouter.launch(url, context)) {
            Log.e(TAG, "Cannot open url " + url);
        }
    }

    @Override
    public void showDialog(@Nullable String dialogTitle,
                           @Nullable String dialogBody,
                           @NonNull String dialogContinue,
                           @NonNull String dialogClose,
                           @Nullable final DialogInterface.OnClickListener responseListener) {

        AlertDialog.Builder dialogBuilder = new AlertDialog.Builder(new ContextThemeWrapper(context, context.getApplicationInfo().theme));

        DialogInterface.OnDismissListener dismissListener = new DialogInterface.OnDismissListener() {
            @Override
            public void onDismiss(DialogInterface dialog) {
                currentDialog = null;
            }
        };

        DialogInterface.OnClickListener onClickListenerProxy = new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {
                currentDialog = null;
                if(responseListener != null){
                    responseListener.onClick(dialog, which);
                }
            }
        };

        final DialogClickListenerProxy listener = new DialogClickListenerProxy(onClickListenerProxy, dismissListener);

        if (!TextUtils.isEmpty(dialogTitle)) {
            dialogBuilder.setTitle(dialogTitle);
        }

        if (!TextUtils.isEmpty(dialogBody)) {
            dialogBuilder.setMessage(dialogBody);
        }

        dialogBuilder.setPositiveButton(dialogContinue, listener);
        dialogBuilder.setNegativeButton(dialogClose, listener);

        dialogBuilder.setCancelable(false);
        currentDialog = dialogBuilder.create();

        listener.autoRelease(currentDialog);

        currentDialog.show();
    }

    @Override
    public void resumeWeb() {
        view.resumeWeb();
    }

    @Override
    public void pauseWeb() {
        view.pauseWeb();
    }

    @Override
    public void setImmersiveMode() {
        view.setImmersiveMode();
    }

    private static class DialogClickListenerProxy implements DialogInterface.OnClickListener, Dialog.OnDismissListener {

        private AtomicReference<DialogInterface.OnClickListener> ckickRef = new AtomicReference<>();
        private AtomicReference<DialogInterface.OnDismissListener> dismissRef = new AtomicReference<>();

        public DialogClickListenerProxy(DialogInterface.OnClickListener onClickListener, DialogInterface.OnDismissListener onDismissListener) {
            ckickRef.set(onClickListener);
            dismissRef.set(onDismissListener);
        }

        @Override
        public void onClick(DialogInterface dialog, int which) {
            dialog.dismiss();
            DialogInterface.OnClickListener base = ckickRef.get();
            if (base != null) {
                base.onClick(dialog, which);
            }
        }

        private void autoRelease(Dialog dialog) {
            dialog.setOnDismissListener(this);
        }

        @Override
        public void onDismiss(DialogInterface dialog) {
            DialogInterface.OnDismissListener base = dismissRef.get();
            if (base != null) {
                base.onDismiss(dialog);
            }
            dismissRef.set(null);
            ckickRef.set(null);
        }
    }

    @Override
    public void removeWebView() {
        view.removeWebView();
    }

    @Override
    public boolean hasWebView() {
        return view.hasWebView();
    }
}
