package com.vungle.warren.ui.presenter;

import android.content.ActivityNotFoundException;
import android.content.pm.ActivityInfo;
import android.text.TextUtils;
import android.util.Log;
import android.webkit.WebView;
import android.webkit.WebViewRenderProcess;

import com.google.gson.JsonObject;
import com.google.gson.JsonPrimitive;
import com.vungle.warren.AdConfig;
import com.vungle.warren.SessionData;
import com.vungle.warren.analytics.AdAnalytics;
import com.vungle.warren.analytics.AnalyticsEvent;
import com.vungle.warren.error.VungleException;
import com.vungle.warren.model.Advertisement;
import com.vungle.warren.model.Cookie;
import com.vungle.warren.model.Placement;
import com.vungle.warren.model.Report;
import com.vungle.warren.persistence.Repository;
import com.vungle.warren.ui.DurationRecorder;
import com.vungle.warren.ui.PresenterAppLeftCallback;
import com.vungle.warren.ui.contract.AdContract;
import com.vungle.warren.ui.contract.WebAdContract;
import com.vungle.warren.ui.state.OptionsState;
import com.vungle.warren.ui.view.VungleWebClient;
import com.vungle.warren.ui.view.WebViewAPI;
import com.vungle.warren.utility.ActivityManager;
import com.vungle.warren.utility.AsyncFileUtils;
import com.vungle.warren.utility.Scheduler;

import java.io.File;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import static com.vungle.warren.analytics.AnalyticsEvent.Ad.videoLength;
import static com.vungle.warren.error.VungleException.AD_UNABLE_TO_PLAY;

/**
 * This class handles the presentation logic for all web-based advertisements.
 */
public class MRAIDAdPresenter implements WebAdContract.WebAdPresenter, VungleWebClient.MRAIDDelegate, WebViewAPI.WebClientErrorHandler {

    private static final String TAG = MRAIDAdPresenter.class.getCanonicalName();

    private static final String EXTRA_INCENTIVIZED_SENT = "incentivized_sent";
    private static final String CLOSE = "close";
    private static final String CONSENT_ACTION = "consentAction";
    private static final String ACTION_WITH_VALUE = "actionWithValue";
    private static final String VIDEO_VIEWED = "videoViewed";
    private static final String TPAT = "tpat";
    private static final String ACTION = "action";
    private static final String OPEN = "open";
    private static final String OPEN_NON_MRAID = "openNonMraid";
    private static final String USE_CUSTOM_CLOSE = "useCustomClose";
    private static final String USE_CUSTOM_PRIVACY = "useCustomPrivacy";
    private static final String OPEN_PRIVACY = "openPrivacy";
    private static final String SUCCESSFUL_VIEW = "successfulView";
    private static final String EXTRA_REPORT = "saved_report";
    private static final String FLEXVIEW = "flexview";

    private final Scheduler scheduler;
    private final AdAnalytics analytics;
    private Map<String, Cookie> cookieMap = new HashMap<>();
    private AsyncFileUtils.ExistenceOperation fileExistenceOperation;

    /**
     * The event bus which is used to communicate advertisement lifecycle events back to the listener
     * managing the presentation of this advertisement.
     */
    private EventListener bus;

    /**
     * The advertisement being presented. This object contains all the metadata pertaining to this ad
     * that we need to know in order to display it.
     */
    private Advertisement advertisement;

    /**
     * The advertisement report that we use to notify the Vungle server about this advertisement being
     * played. This object allows everyone to make money.
     */
    private Report report;

    /**
     * Placement metadata for the advertisement being played. This is the container for the advertisement
     * and necessary for reporting lifecycle events back to the listener.
     */
    @NonNull
    private final Placement placement;

    /**
     * The WebClient used for web lifecycle event management. We use our own version in order to
     * override default behaviours.
     */
    private WebViewAPI webClient;

    /**
     * Reference to the SDK's persistor. This persistor manages the interactions between our SDK's
     * in-memory objects and the values on disk.
     */
    private Repository repository;

    /**
     * Reference to the local asset directory, provided by the designer.
     */
    private File assetDir;

    /**
     * The Advertisement View that this presenter has been attached to. All rendering happens within
     */
    private WebAdContract.WebAdView adView;

    /**
     * If true, the back button will close the advertisement. Otherwise,
     */
    private boolean backEnabled;

    /**
     * The duration, in milliseconds, of the advertisement video.
     */
    private long duration;

    /**
     *
     */
    private SessionData sessionData;

    private AtomicBoolean sendReportIncentivized = new AtomicBoolean(false);
    private AtomicBoolean isDestroying = new AtomicBoolean(false);
    private Repository.SaveCallback repoCallback = new Repository.SaveCallback() {
        boolean errorHappened = false;

        @Override
        public void onSaved() {

        }

        @Override
        public void onError(Exception e) {
            if (errorHappened)
                return;
            errorHappened = true;
            makeBusError(VungleException.DB_ERROR);
            closeView();
        }
    };

    private DurationRecorder durationRecorder;
    private ActivityManager activityManager;

    /**
     * Basic constructor for the MRAIDAdPresenter, this creates a presenter that will manage the advertisement
     * given.
     *
     * @param advertisement The Advertisement to render
     * @param placement     The placement in which this advertisement is being played.
     * @param repository    The database interface from which metadata is sourced and placed.
     */
    public MRAIDAdPresenter(@NonNull Advertisement advertisement,
                            @NonNull Placement placement,
                            @NonNull Repository repository,
                            @NonNull Scheduler scheduler,
                            @NonNull AdAnalytics adAnalytics,
                            @NonNull WebViewAPI webViewAPI,
                            @Nullable OptionsState state,
                            @NonNull File assetDir,
                            @NonNull SessionData sessionData,
                            @NonNull ActivityManager activityManager) {
        this.advertisement = advertisement;
        this.repository = repository;
        this.placement = placement;
        this.scheduler = scheduler;
        this.analytics = adAnalytics;
        this.webClient = webViewAPI;
        this.assetDir = assetDir;
        this.activityManager = activityManager;
        this.sessionData = sessionData;
        loadData(state);
    }

    @Override
    public void setEventListener(@Nullable EventListener listener) {
        this.bus = listener;
    }


    public void reportAction(@NonNull String action, @Nullable String value) {
        if (action.equals(AnalyticsEvent.Ad.videoLength)) {
            /// Hacky way to report video length, since the advertisement metadata from the server
            /// does not contain this information.
            duration = Long.parseLong(value);
            report.setVideoLength(duration);
            repository.save(report, repoCallback);
            return; /// This is not an action that should be included in the report.
        }
        report.recordAction(action, value, System.currentTimeMillis());
        repository.save(report, repoCallback);
    }

    @Override
    public void onViewConfigurationChanged() {
        adView.updateWindow(advertisement.getTemplateType().equals(FLEXVIEW));
        webClient.notifyPropertiesChange(true);
    }

    @Override
    public void attach(@NonNull WebAdContract.WebAdView adView, @Nullable OptionsState state) {
        isDestroying.set(false);
        this.adView = adView;
        adView.setPresenter(this);
        /// Process the advertisement settings
        @AdConfig.Settings int settings = advertisement.getAdConfig().getSettings();
        if (settings > 0) {
            backEnabled = (settings & AdConfig.IMMEDIATE_BACK) == AdConfig.IMMEDIATE_BACK;
        }

        /// Check if the advertisement has a specified orientation, then lock the window to that
        int requestedOrientation = -1; /// Sentinel value
        int adOrientation = advertisement.getAdConfig().getAdOrientation();
        if (adOrientation == AdConfig.MATCH_VIDEO) {
            switch (advertisement.getOrientation()) {
                case Advertisement.PORTRAIT:
                    /// Lock the orientation to portrait.
                    requestedOrientation = ActivityInfo.SCREEN_ORIENTATION_SENSOR_PORTRAIT;
                    break;
                case Advertisement.LANDSCAPE:
                    /// Lock the orientation to landscape.
                    requestedOrientation = ActivityInfo.SCREEN_ORIENTATION_SENSOR_LANDSCAPE;
                    break;
            }
        } else if (adOrientation == AdConfig.PORTRAIT) {
            requestedOrientation = ActivityInfo.SCREEN_ORIENTATION_SENSOR_PORTRAIT;
        } else if (adOrientation == AdConfig.LANDSCAPE) {
            requestedOrientation = ActivityInfo.SCREEN_ORIENTATION_SENSOR_LANDSCAPE;
        } else {
            requestedOrientation = ActivityInfo.SCREEN_ORIENTATION_SENSOR;
        }

        Log.d(TAG, "Requested Orientation " + requestedOrientation);
        adView.setOrientation(requestedOrientation);
        prepare(state);
    }

    @Override
    public void detach(boolean changingConfiguration) {
        int flag = (changingConfiguration ? AdContract.AdStopReason.IS_CHANGING_CONFIGURATION : 0)
                | AdContract.AdStopReason.IS_AD_FINISHING;

        if (fileExistenceOperation != null) {
            fileExistenceOperation.cancel();
        }

        stop(flag);
        adView.destroyAdView();
    }

    private void prepare(@Nullable OptionsState state) {
        webClient.setMRAIDDelegate(this);
        webClient.setErrorHandler(this);

        File template = new File(assetDir.getPath() + File.separator + Advertisement.KEY_TEMPLATE);
        loadMraid(template);

        if (FLEXVIEW.equals(advertisement.getTemplateType()) && advertisement.getAdConfig().getFlexViewCloseTime() > 0) {
            scheduler.schedule(new Runnable() {
                @Override
                public void run() {
                    long closeTime = System.currentTimeMillis();

                    //we need to send these events to the report, specific to flexviewclose by timer
                    report.recordAction("mraidCloseByTimer", "", closeTime);
                    report.recordAction("mraidClose", "", closeTime);
                    repository.save(report, repoCallback);

                    closeView();
                }
            }, ((long) advertisement.getAdConfig().getFlexViewCloseTime()) * 1000L);
        }

        Cookie incentivizedCookie = cookieMap.get(Cookie.INCENTIVIZED_TEXT_COOKIE);
        String userIdFromCookie = (incentivizedCookie == null) ? null : incentivizedCookie.getString("userID");

        if (report == null) {
            report = new Report(advertisement, placement, System.currentTimeMillis(), userIdFromCookie, sessionData);
            report.setTtDownload(advertisement.getTtDownload());
            repository.save(report, repoCallback);
        }

        if (durationRecorder == null) {
            durationRecorder = new DurationRecorder(report, repository, repoCallback);
        }

        Cookie gdprConsent = cookieMap.get(Cookie.CONSENT_COOKIE);
        if (gdprConsent != null) {
            //only show dialog if user isCountryDataProtected and status is unknown
            boolean collectConsent = gdprConsent.getBoolean("is_country_data_protected") && "unknown".equals(gdprConsent.getString("consent_status"));

            webClient.setConsentStatus(collectConsent,
                    gdprConsent.getString("consent_title"),
                    gdprConsent.getString("consent_message"),
                    gdprConsent.getString("button_accept"),
                    gdprConsent.getString("button_deny")
            );

            //by default save consent as not collected
            //TODO: Need to put this in a place that guarantees gdpr dialog is shown to user
            if (collectConsent) {
                gdprConsent.putValue("consent_status", "opted_out_by_timeout");
                gdprConsent.putValue("timestamp", System.currentTimeMillis() / 1000);
                gdprConsent.putValue("consent_source", "vungle_modal");
                repository.save(gdprConsent, repoCallback);
            }
        }

        /// Enable the back button once the delay has elapsed. If no delay, enable immediately
        int delay = advertisement.getShowCloseDelay(placement.isIncentivized());
        if (delay > 0) {
            /// Use a simple timer to enact the delay mechanism
            scheduler.schedule(new Runnable() {
                @Override
                public void run() {
                    backEnabled = true;
                }
            }, delay);
        } else {
            backEnabled = true;
        }

        adView.updateWindow(FLEXVIEW.equals(advertisement.getTemplateType()));

        if (bus != null) {
            bus.onNext("start", null, placement.getId());
        }
    }

    /**
     * Loads the template file into the Advertisement view. If the file is a zip directory, we first unzip
     * and inject the Android-specific javascript code into the main bridge file, then instruct the
     * AdView to load the main html content for the advertisement.
     *
     * @param template The template directory, which could be zipped.
     */
    private void loadMraid(@NonNull File template) {
        /// First, unzip the mraid assets
        File dest = new File(template.getParent());

        /// Now that the mraid.js file has been constructed, load the index.html for the advertisement.
        final File indexHtml = new File(dest.getPath() + File.separator + "index.html");

        fileExistenceOperation = AsyncFileUtils.isFileExistAsync(indexHtml,
                new AsyncFileUtils.FileExistCallback() {
                    @Override
                    public void status(boolean isExist) {
                        if (!isExist) {
                            // Template most likely is corrupted we need to pass error
                            makeBusError(VungleException.RENDER_ERROR);
                            //General cannot play error
                            makeBusError(AD_UNABLE_TO_PLAY);
                            adView.close();
                            return;
                        }
                        adView.showWebsite("file://" + indexHtml.getPath());
                    }
                });
    }

    @Override
    public void start() {
        if (!adView.hasWebView()) {
            reportErrorAndCloseAd(VungleException.WEB_CRASH);
            return;
        }

        adView.setImmersiveMode();
        adView.resumeWeb();
        setAdVisibility(true);
    }

    @Override
    public void stop(@AdContract.AdStopReason int stopReason) {
        boolean isChangingConfigurations = (stopReason & AdContract.AdStopReason.IS_CHANGING_CONFIGURATION) != 0;
        boolean isFinishing = (stopReason & AdContract.AdStopReason.IS_AD_FINISHING) != 0;
        boolean isFinishByAPI = (stopReason & AdContract.AdStopReason.IS_AD_FINISHED_BY_API) != 0;

        adView.pauseWeb();

        /// Pause the javascript ad
        setAdVisibility(false);
        if (!isChangingConfigurations && isFinishing && !isDestroying.getAndSet(true)) {
            /// Clean up the MRAID delegate, otherwise the webclient will keep sending commands.
            /// It uses reference counting to determine whether to send commands.
            if (webClient != null) {
                webClient.setMRAIDDelegate(null);
            }

            if (isFinishByAPI) {
                reportAction("mraidCloseByApi", null);
            }

            repository.save(report, repoCallback);

            if (bus != null) {
                //TODO: CTA click event has been replaced by onAdClick(String) so this parameter should be removed in future.
                bus.onNext("end", report.isCTAClicked() ? "isCTAClicked" : null, placement.getId());
            }
        }
    }

    @Override
    public void setAdVisibility(boolean isViewable) {
        webClient.setAdVisibility(isViewable);
        if (isViewable) {
            durationRecorder.start();
        } else {
            durationRecorder.stop();
        }
    }

    @Override
    public void generateSaveState(@Nullable OptionsState state) {
        if (state == null) {
            return;
        }

        repository.save(report, repoCallback);
        state.put(EXTRA_REPORT, report.getId());
        state.put(EXTRA_INCENTIVIZED_SENT, sendReportIncentivized.get());
    }

    @Override
    public void restoreFromSave(@Nullable OptionsState state) {
        if (state == null) {
            return;
        }


        boolean incentivizedShown = state.getBoolean(EXTRA_INCENTIVIZED_SENT, false);

        if (incentivizedShown) {
            sendReportIncentivized.set(incentivizedShown);
        }

        //old code
        if (report == null) {
            /// The advertisement was not started and cannot be restored.
            adView.close();
            return;
        }
    }


    @Override
    public boolean handleExit(@Nullable String flexViewCloseApiPlacementID) {
        if (flexViewCloseApiPlacementID != null) {
            if (advertisement == null) {
                Log.e(TAG, "Unable to close advertisement");
                return false;
            }
            if (!placement.getId().equals(flexViewCloseApiPlacementID)) {
                Log.e(TAG, "Cannot close FlexView Ad with invalid placement reference id");
                return false;
            }
            if (!FLEXVIEW.equals(advertisement.getTemplateType())) {
                Log.e(TAG, "Cannot close a Non FlexView ad");
                return false;
            }
            adView.showWebsite("javascript:window.vungle.mraidBridgeExt.requestMRAIDClose()");
            reportAction("mraidCloseByApi", null);
            return true;
        } else if (backEnabled) {
            /// Pass the action through the bridge. The template will then decide what to do.
            adView.showWebsite("javascript:window.vungle.mraidBridgeExt.requestMRAIDClose()");
        }

        return false;
    }


    @Override
    public boolean processCommand(@NonNull String command, @NonNull JsonObject arguments) {
        switch (command) {
            case CLOSE: {
                /// Close the advertisement
                reportAction("mraidClose", null);
                closeView();
                return true;
            }
            case CONSENT_ACTION: {
                Cookie gdprConsent = cookieMap.get(Cookie.CONSENT_COOKIE);
                if (gdprConsent == null) {
                    gdprConsent = new Cookie(Cookie.CONSENT_COOKIE);
                }

                String value = arguments.get("event").getAsString();
                gdprConsent.putValue("consent_status", value);
                gdprConsent.putValue("consent_source", "vungle_modal");
                gdprConsent.putValue("timestamp", System.currentTimeMillis() / 1000);


                repository.save(gdprConsent, repoCallback);
                return true;
            }
            case ACTION_WITH_VALUE: {
                /// Record the action to report later
                String action = arguments.get("event").getAsString();
                String value = arguments.get("value").getAsString();
                report.recordAction(action, value, System.currentTimeMillis());
                repository.save(report, repoCallback);

                /// If the value was an update on the position viewed, report this to the event bus.

                if (action.equals(VIDEO_VIEWED) && duration > 0) {
                    /// videoViewed value is given in milliseconds.
                    int percent = 0;
                    try {
                        float position = Float.parseFloat(value);
                        percent = (int) ((position / duration) * 100);
                    } catch (NumberFormatException nfe) {
                        Log.e(TAG, "value for videoViewed is null !");
                    }

                    if (percent > 0) {
                        if (bus != null) {
                            bus.onNext("percentViewed:" + percent, null, placement.getId());
                        }

                        Cookie configCookie = cookieMap.get(Cookie.CONFIG_COOKIE);
                        if (placement.isIncentivized() && percent > 75 && configCookie != null &&
                                configCookie.getBoolean("isReportIncentivizedEnabled") &&
                                !sendReportIncentivized.getAndSet(true)) {
                            //send ri only once past 75 percentile
                            JsonObject body = new JsonObject();
                            body.add("placement_reference_id", new JsonPrimitive(placement.getId()));
                            body.add("app_id", new JsonPrimitive(advertisement.getAppID()));
                            body.add("adStartTime", new JsonPrimitive(report.getAdStartTime()));
                            body.add("user", new JsonPrimitive(report.getUserID()));
                            analytics.ri(body);
                        }
                    }

                    durationRecorder.update();
                }

                if (action.equals("videoLength")) {
                    /// The MRAID container has to tell us about the length of the video. We store this
                    /// in an instance variable in order to report the percentage of the video that
                    /// was viewed later.
                    duration = Long.parseLong(value);
                    reportAction(videoLength, value);

                    //'isViewable' flag will be ignored until now - see AND-2087
                    webClient.notifyPropertiesChange(true);
                }
                adView.setVisibility(true);
                return true;
            }
            case TPAT: {
                /// fire off the tpat event. these are http calls to a specific URL provided by
                /// the ad server. No retry logic is necessary since these are only valid if they
                /// are live
                String event = arguments.get("event").getAsString();

                analytics.ping(advertisement.getTpatUrls(event));
                return true;
            }
            case ACTION: {
                // TODO
                return true;
            }
            case OPEN:
            case OPEN_NON_MRAID: {
                reportAction("download", null);
                if (OPEN.equalsIgnoreCase(command)) {
                    reportAction("mraidOpen", null);
                } else if (OPEN_NON_MRAID.equalsIgnoreCase(command)) {
                    reportAction("nonMraidOpen", null);
                }

                String url = arguments.get("url").getAsString();
                activityManager.addOnNextAppLeftCallback(new PresenterAppLeftCallback(bus, placement));
                if(url == null || url.isEmpty())  {
                    Log.e(TAG, "CTA destination URL is not configured properly");
                } else {
                    adView.open(url);
                }

                if (bus != null) {
                    bus.onNext("open", "adClick", placement.getId());
                }
                return true;
            }
            case USE_CUSTOM_CLOSE: {
                String value = arguments.get("sdkCloseButton").getAsString();
                switch (value) {
                    case "gone":
                    case "invisible":
                    case "visible":
                        break;
                    default:
                        throw new IllegalArgumentException("Unknown value " + value);
                }
                return true;
            }
            case USE_CUSTOM_PRIVACY: {
                String value = arguments.get("useCustomPrivacy").getAsString();
                switch (value) {
                    case "gone":
                    case "true":
                    case "false":
                        break;
                    default:
                        throw new IllegalArgumentException("Unknown value " + value);
                }
                return true;
            }
            case OPEN_PRIVACY: {
                /// Open the privacy link, which will explain all the data we collect. The ad should
                /// resume after this, so we do not call finish().
                String url = arguments.get("url").getAsString();
                activityManager.addOnNextAppLeftCallback(new PresenterAppLeftCallback(bus, placement));
                adView.open(url);
                return true;
            }
            case SUCCESSFUL_VIEW: {
                /// No op, we only acknowledge that the view was successful. We store this and report
                /// it back to the publisher in the start ad callback.
                if (bus != null) {
                    bus.onNext("successfulView", null, placement.getId());
                }

                Cookie configCookie = cookieMap.get(Cookie.CONFIG_COOKIE);
                if (placement.isIncentivized() && configCookie != null && configCookie.getBoolean("isReportIncentivizedEnabled") && !sendReportIncentivized.getAndSet(true)) {
                    //send ri only once past 75 percentile
                    JsonObject body = new JsonObject();
                    body.add("placement_reference_id", new JsonPrimitive(placement.getId()));
                    body.add("app_id", new JsonPrimitive(advertisement.getAppID()));
                    body.add("adStartTime", new JsonPrimitive(report.getAdStartTime()));
                    body.add("user", new JsonPrimitive(report.getUserID()));
                    analytics.ri(body);
                }
                return true;
            }
            default:
                /// Unknown command, but not a fatal error
                return false;
        }
    }

    @Override
    public void onMraidAction(@NonNull String action) {
        switch (action) {
            case CLOSE_ACTION: {
                /// Close the advertisement, finish the activity.
                closeView();
                break;
            }
            case DOWNLOAD_ACTION: {
                download();
                break;
            }
            case PRIVACY_ACTION: {
                break;
            }
            default:
                throw new IllegalArgumentException("Unknown action " + action);
        }
    }

    private void download() {
        reportAction("cta", "");
        /// Close the advertisement, take the user to the download link.
        try {
            /// Send the TPAT Hit
            analytics.ping(new String[]{advertisement.getCTAURL(true)});
            /// Open the Destination URL
            activityManager.addOnNextAppLeftCallback(new PresenterAppLeftCallback(bus, placement));
            adView.open(advertisement.getCTAURL(false));
        } catch (ActivityNotFoundException invalid) {
            // Todo : What to do in this case ? Need to check with Team.
            // closeView();
        }
    }

    private void closeView() {
        adView.close();
        scheduler.cancelAll();
    }

    private void loadData(OptionsState state) {
        cookieMap.put(Cookie.INCENTIVIZED_TEXT_COOKIE, repository.load(Cookie.INCENTIVIZED_TEXT_COOKIE, Cookie.class).get());
        cookieMap.put(Cookie.CONSENT_COOKIE, repository.load(Cookie.CONSENT_COOKIE, Cookie.class).get());
        cookieMap.put(Cookie.CONFIG_COOKIE, repository.load(Cookie.CONFIG_COOKIE, Cookie.class).get());

        if (state != null) {
            String reportId = state.getString(EXTRA_REPORT);
            Report restoredReport = TextUtils.isEmpty(reportId) ? null : repository.load(reportId, Report.class).get();

            if (restoredReport != null) {
                report = restoredReport;
            }
        }
    }

    @Override
    public void onReceivedError(String errorDesc) {
        if (report != null) {
            report.recordError(errorDesc);
            repository.save(report, repoCallback);
        }
    }

    @Override
    public boolean onWebRenderingProcessGone(WebView view, boolean didCrash) {
        handleWebViewException(VungleException.WEB_CRASH);
        return true;
    }

    @Override
    public void onRenderProcessUnresponsive(@NonNull WebView webView, @Nullable WebViewRenderProcess webViewRenderProcess) {
        handleWebViewException(VungleException.WEBVIEW_RENDER_UNRESPONSIVE);
    }

    private void handleWebViewException(@VungleException.ExceptionCode int reason) {
        if (adView != null) {
            adView.removeWebView();
        }
        reportErrorAndCloseAd(reason);
    }

    private void reportErrorAndCloseAd(@VungleException.ExceptionCode int reason) {
        //TODO reportAction(reason, getLocalizedMessage() );
        makeBusError(reason);
        closeView();
    }

    private void makeBusError(@VungleException.ExceptionCode int code) {
        if (bus != null) {
            bus.onError(new VungleException(code), placement.getId());
        }
    }
}
