package com.vungle.warren.ui.presenter;

import android.app.AlertDialog;
import android.content.ActivityNotFoundException;
import android.content.DialogInterface;
import android.content.pm.ActivityInfo;
import android.text.TextUtils;
import android.util.Log;
import android.webkit.WebView;
import android.webkit.WebViewRenderProcess;

import com.google.gson.JsonObject;
import com.google.gson.JsonPrimitive;
import com.vungle.warren.AdConfig;
import com.vungle.warren.SessionData;
import com.vungle.warren.analytics.AdAnalytics;
import com.vungle.warren.analytics.AnalyticsEvent;
import com.vungle.warren.analytics.AnalyticsVideoTracker;
import com.vungle.warren.error.VungleException;
import com.vungle.warren.model.Advertisement;
import com.vungle.warren.model.Cookie;
import com.vungle.warren.model.Placement;
import com.vungle.warren.model.Report;
import com.vungle.warren.persistence.Repository;
import com.vungle.warren.ui.DurationRecorder;
import com.vungle.warren.ui.PresenterAppLeftCallback;
import com.vungle.warren.ui.contract.AdContract;
import com.vungle.warren.ui.contract.LocalAdContract;
import com.vungle.warren.ui.state.OptionsState;
import com.vungle.warren.ui.view.WebViewAPI;
import com.vungle.warren.utility.ActivityManager;
import com.vungle.warren.utility.AsyncFileUtils;
import com.vungle.warren.utility.Scheduler;

import java.io.File;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import static com.vungle.warren.analytics.AnalyticsEvent.Ad.clickUrl;
import static com.vungle.warren.analytics.AnalyticsEvent.Ad.mute;
import static com.vungle.warren.analytics.AnalyticsEvent.Ad.postrollClick;
import static com.vungle.warren.analytics.AnalyticsEvent.Ad.postrollView;
import static com.vungle.warren.analytics.AnalyticsEvent.Ad.unmute;
import static com.vungle.warren.analytics.AnalyticsEvent.Ad.videoClick;
import static com.vungle.warren.analytics.AnalyticsEvent.Ad.videoClose;
import static com.vungle.warren.error.VungleException.AD_UNABLE_TO_PLAY;
import static com.vungle.warren.error.VungleException.DB_ERROR;
import static com.vungle.warren.model.Cookie.CONFIG_COOKIE;
import static com.vungle.warren.model.Cookie.CONSENT_COOKIE;
import static com.vungle.warren.model.Cookie.INCENTIVIZED_TEXT_COOKIE;

/**
 * Presenter that handles the business logic of showing a vungle_local ad. Given an {@link Advertisement}
 * and {@link Placement}, this presenter manages the {@link AdContract.AdView} which is showing the ad. The
 * presenter is also responsible for creating the ad report and managing any extra components, such
 * as the event listener, which are not the responsibility of the view layer.
 */
public class LocalAdPresenter implements LocalAdContract.LocalPresenter, WebViewAPI.WebClientErrorHandler {

    static final String EXTRA_INCENTIVIZED_SENT = "incentivized_sent";
    static final String TAG = "LocalAdPresenter";
    static final String EXTRA_REPORT = "saved_report";
    static final String EXTRA_IN_POST = "in_post_roll";
    static final String EXTRA_IS_MUTED = "is_muted_mode";
    static final String EXTRA_VIDEO_POSITION = "videoPosition";
    static final String HTTPS_VUNGLE_COM_PRIVACY = "https://vungle.com/privacy/";
    public static final int INCENTIVIZED_TRESHOLD = 75;
    private final Scheduler scheduler;
    private final AdAnalytics analytics;
    private final AnalyticsVideoTracker videoTracker;
    private final WebViewAPI webViewAPI;
    private final Map<String, Cookie> cookies = new HashMap<>();
    private AsyncFileUtils.ExistenceOperation fileExistenceOperation;

    /**
     * The placement being presenter to the user. This reference holds information about business
     * logic, such as if this advertisement is incentivized.
     */
    @NonNull
    private Placement placement;

    /**
     * The advertisement metadata. Includes information about where to locate the assets, as well
     * as internal communicaiton information, such as the third-party attribution URLs.
     */
    private Advertisement advertisement;

    /**
     * The advertisement report which we will send to the AdServer once the advertisement has finished.
     * It includes all of the information about how the advertisement was played, and for how long.
     */
    private Report report;

    /**
     * Reference to the SDK's persistor implementation, which manages serializing and deserializing
     * data from disk.
     */
    private Repository repository;

    /**
     * A convenient refence to the directory where the assets for this advertisement are stored.
     */
    private File assetDir;

    /**
     * State-tracking variable that informs us about the current mute state of the advertisement.
     * This is independent of the device muted state, and is therefore stored independently.
     */
    private boolean muted;

    /**
     * State-tracking variable that informs us if the back button is currently enabled. We sometimes
     * block the user from leaving an advertisement. Currently it also tied to the showClose delay.
     */
    private boolean userExitEnabled;

    /**
     * State-tracking variable that informs us if we are done playing the advertisement video and
     * are currently in the post-roll stage.
     */
    private boolean inPost;

    /**
     * The view which is displaying the advertisement and taking direction from this presenter.
     */
    private LocalAdContract.LocalView adView;

    /**
     * User-friendly strings to inform them about the consequences of ending an ad before receiving
     * their reward for watching the ad.
     */
    private String dialogTitle = "Are you sure?",
            dialogBody = "If you exit now, you will not get your reward",
            dialogContinue = "Continue",
            dialogClose = "Close";

    /**
     * The event bus where we emit lifecycle information, such as "start", "end", and "error".
     */
    private AdContract.AdvertisementPresenter.EventListener bus;

    /**
     * The duration of the advertisement video, in seconds.
     */
    private int duration;

    /**
     *
     */
    private SessionData sessionData;

    private AtomicBoolean sendReportIncentivized = new AtomicBoolean(false);
    private AtomicBoolean isDestroying = new AtomicBoolean(false);
    private int videoPosition;
    private int progress;
    private LinkedList<Advertisement.Checkpoint> checkpointList = new LinkedList<>();
    private Repository.SaveCallback repoCallback = new Repository.SaveCallback() {
        boolean errorHappened = false;

        @Override
        public void onSaved() {

        }

        @Override
        public void onError(Exception e) {
            if (errorHappened)
                return;
            errorHappened = true;
            makeBusError(DB_ERROR);
            closeAndReport();
        }
    };

    private DurationRecorder durationRecorder;
    private ActivityManager activityManager;

    /**
     * @param advertisement
     * @param placement
     */
    public LocalAdPresenter(@NonNull Advertisement advertisement,
                            @NonNull Placement placement,
                            @NonNull Repository repository,
                            @NonNull Scheduler scheduler,
                            @NonNull AdAnalytics adAnalytics,
                            @NonNull AnalyticsVideoTracker videoTracker,
                            @NonNull WebViewAPI webViewAPI,
                            @Nullable OptionsState state,
                            @NonNull File assetDir,
                            @NonNull SessionData sessionData,
                            @NonNull ActivityManager activityManager) {
        this.advertisement = advertisement;
        this.placement = placement;
        this.scheduler = scheduler;
        this.analytics = adAnalytics;
        this.videoTracker = videoTracker;
        this.webViewAPI = webViewAPI;
        this.repository = repository;
        this.assetDir = assetDir;
        this.activityManager = activityManager;
        this.sessionData = sessionData;

        if (advertisement.getCheckpoints() != null) {
            checkpointList.addAll(advertisement.getCheckpoints());
            Collections.sort(checkpointList);
        }

        loadData(state);
    }

    @Override
    public void setEventListener(@Nullable EventListener listener) {
        this.bus = listener;
    }

    private void reportError(@NonNull String error) {
        report.recordError(error);
        repository.save(report, repoCallback);

        makeBusError(VungleException.RENDER_ERROR);
        if (!inPost && advertisement.hasPostroll()) {
            //Since video error and postroll is there, start it, playPost() will decide next
            playPost();
        } else {
            //Since video error and no postroll to start, let pass error and close the screen.
            makeBusError(AD_UNABLE_TO_PLAY);
            adView.close();
        }
    }

    public void reportAction(@NonNull String action, @Nullable String value) {
        if (action.equals(AnalyticsEvent.Ad.videoLength)) {
            /// Hacky way to report video length, since the advertisement metadata from the server
            /// does not contain this information.
            duration = Integer.parseInt(value);
            report.setVideoLength(duration);
            repository.save(report, repoCallback);
            return; /// This is not an action , but should be reported under plays in the report.
        }

        //TODO for later we should have a separate method in AdPresenter handleFiringTPATs
        //that does this across the presenter for all actions or best have a switch case in here
        //so that all the firing TPAT logic is encapsulated in one place, and not scattered
        switch (action) {
            case mute:
            case unmute:
            case videoClose:
                analytics.ping(advertisement.getTpatUrls(action));
            default:
                break;
        }

        report.recordAction(action, value, System.currentTimeMillis());
        repository.save(report, repoCallback);
    }

    @Override
    public void onViewConfigurationChanged() {
        //do we need this
        webViewAPI.notifyPropertiesChange(true);
    }

    @Override
    public void attach(@NonNull LocalAdContract.LocalView adView, @Nullable OptionsState state) {
        isDestroying.set(false);
        /// Extract incentivized fields, if the placement is incentivized and they were included in
        /// the intent.
        this.adView = adView;
        adView.setPresenter(this);

        /// Process the advertisement settings
        @AdConfig.Settings int settings = advertisement.getAdConfig().getSettings();
        if (settings > 0) {
            /// Start the advertisement muted, but this must be done when the video is prepared
            /// for local ads and passed in as an mraid command to mraid ads.
            muted = (settings & AdConfig.MUTED) == AdConfig.MUTED;
            userExitEnabled = (settings & AdConfig.IMMEDIATE_BACK) == AdConfig.IMMEDIATE_BACK;
        }

        /// Check if the advertisement has a specified orientation, then lock the window to that
        int requestedOrientation = -1; /// Sentinel value
        int adOrientation = advertisement.getAdConfig().getAdOrientation();
        if (adOrientation == AdConfig.MATCH_VIDEO) {
            switch (advertisement.getOrientation()) {
                case Advertisement.PORTRAIT:
                    /// Lock the orientation to portrait.
                    requestedOrientation = ActivityInfo.SCREEN_ORIENTATION_SENSOR_PORTRAIT;
                    break;
                case Advertisement.LANDSCAPE:
                    /// Lock the orientation to landscape.
                    requestedOrientation = ActivityInfo.SCREEN_ORIENTATION_SENSOR_LANDSCAPE;
                    break;
            }
        } else if (adOrientation == AdConfig.PORTRAIT) {
            requestedOrientation = ActivityInfo.SCREEN_ORIENTATION_SENSOR_PORTRAIT;
        } else if (adOrientation == AdConfig.LANDSCAPE) {
            requestedOrientation = ActivityInfo.SCREEN_ORIENTATION_SENSOR_LANDSCAPE;
        } else {
            requestedOrientation = ActivityInfo.SCREEN_ORIENTATION_SENSOR;
        }

        Log.d(TAG, "Requested Orientation " + requestedOrientation);
        adView.setOrientation(requestedOrientation);
        prepare(state);
    }

    @Override
    public void detach(boolean changingConfiguration) {
        int flag = (changingConfiguration ? AdContract.AdStopReason.IS_CHANGING_CONFIGURATION : 0)
                | AdContract.AdStopReason.IS_AD_FINISHING;

        if (fileExistenceOperation != null) {
            fileExistenceOperation.cancel();
        }

        stop(flag);
        adView.destroyAdView();
    }

    /**
     * Load the postroll website into the activity's webview and inform the server that the postroll
     * has been viewed.
     */
    private void playPost() {
        File post = new File(assetDir.getPath());
        final File indexHtml = new File(post.getPath() + File.separator + "index.html");

        fileExistenceOperation = AsyncFileUtils.isFileExistAsync(indexHtml,
                new AsyncFileUtils.FileExistCallback() {
                    @Override
                    public void status(boolean isExist) {
                        if (isExist) {
                            /// Make sure we call moat stop tracking when we start playing post roll
                            if (videoTracker != null) {
                                videoTracker.stop();
                            }

                            adView.showWebsite("file://" + indexHtml.getPath());

                            /// Inform the ad server that the postroll has been viewed
                            analytics.ping(advertisement.getTpatUrls(postrollView));

                            /// Update activity state
                            inPost = true;
                        } else {
                            makeBusError(VungleException.RENDER_ERROR);
                            makeBusError(AD_UNABLE_TO_PLAY);
                            closeAndReport();
                        }
                    }
                });
    }

    private void prepare(@Nullable OptionsState state) {
        restoreFromSave(state);

        Cookie incentivizedCookie = cookies.get(INCENTIVIZED_TEXT_COOKIE);
        String userIdFromCookie = (incentivizedCookie == null) ? null : incentivizedCookie.getString("userID");

        if (report == null) {
            report = new Report(advertisement, placement, System.currentTimeMillis(), userIdFromCookie, sessionData);
            report.setTtDownload(advertisement.getTtDownload());
            repository.save(report, repoCallback);
        }

        if (durationRecorder == null) {
            durationRecorder = new DurationRecorder(report, repository, repoCallback);
        }

        webViewAPI.setErrorHandler(this);

        //Showing CTA button immediately if enabled, no delay
        adView.showCTAOverlay(advertisement.isCtaOverlayEnabled(), advertisement.getCtaClickArea());
        if (bus != null) {
            bus.onNext("start", null, placement.getId());
        }
    }

    private boolean needShowGDPR(@Nullable Cookie gdprConsent) {
        return gdprConsent != null && gdprConsent.getBoolean("is_country_data_protected")
                && "unknown".equals(gdprConsent.getString("consent_status"));
    }

    private void showGDPR(@NonNull Cookie gdprConsent) {
        final Cookie finalGdpr = gdprConsent;
        AlertDialog.OnClickListener listener = new AlertDialog.OnClickListener() {

            @Override
            public void onClick(DialogInterface dialogInterface, int i) {
                String consented = "opted_out_by_timeout";
                if (i == AlertDialog.BUTTON_NEGATIVE) {
                    consented = "opted_out";
                } else if (i == AlertDialog.BUTTON_POSITIVE) {
                    consented = "opted_in";
                }

                finalGdpr.putValue("consent_status", consented);
                finalGdpr.putValue("timestamp", System.currentTimeMillis() / 1000);
                finalGdpr.putValue("consent_source", "vungle_modal");
                repository.save(finalGdpr, null);

                start();
            }
        };

        //by default save the user decision as opted out until user makes a choice
        gdprConsent.putValue("consent_status", "opted_out_by_timeout");
        finalGdpr.putValue("timestamp", System.currentTimeMillis() / 1000);
        finalGdpr.putValue("consent_source", "vungle_modal");
        repository.save(finalGdpr, repoCallback);

        /// Strings will come from the server for language appropriate to user.
        showDialog(gdprConsent.getString("consent_title"),
                gdprConsent.getString("consent_message"),
                gdprConsent.getString("button_accept"),
                gdprConsent.getString("button_deny"),
                listener);
    }

    @Override
    public boolean handleExit(@Nullable String flexViewCloseApiPlacementID) {
        if (inPost) {
            closeAndReport();
            return true;
        }

        //Block Android back button if not enabled till this time
        if (!userExitEnabled) {
            return false;
        }

        /// If the placement is incentivized, we have to display an "Are You Sure?" currentDialog
        /// before they close the ad. All incentivized ads display this if the close button
        /// or the back button is clicked before the ad has finished and gone to postroll.
        if (placement.isIncentivized() && progress <= INCENTIVIZED_TRESHOLD) {
            showIncetivizedDialog();
            return false;
        }

        reportAction("video_close", null);
        /// If the advertisement has a post-roll bundle to start, display that even though the user
        /// has pressed back.
        if (advertisement.hasPostroll()) {
            playPost();
            return false;
        } else {
            closeAndReport();
            return true;
        }
    }

    private void showIncetivizedDialog() {
        String titleText = dialogTitle;
        String bodyText = dialogBody;
        String continueText = dialogContinue;
        String closeText = dialogClose;

        Cookie incentivizedCookie = cookies.get(INCENTIVIZED_TEXT_COOKIE);
        if (incentivizedCookie != null) {
            titleText = incentivizedCookie.getString("title") == null ? dialogTitle : incentivizedCookie.getString("title");
            bodyText = incentivizedCookie.getString("body") == null ? dialogBody : incentivizedCookie.getString("body");
            continueText = incentivizedCookie.getString("continue") == null ? dialogContinue : incentivizedCookie.getString("continue");
            closeText = incentivizedCookie.getString("close") == null ? dialogClose : incentivizedCookie.getString("close");
        }

        showDialog(titleText, bodyText, continueText, closeText, new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {
                if (which == DialogInterface.BUTTON_NEGATIVE) {
                    /// If the advertisement has a postroll bundle to start, display that even though the user
                    /// has closed the Ad. AND-1505
                    reportAction("video_close", null);
                    continueWithPostroll();
                }
            }
        });
    }

    private void showDialog(String titleText, String bodyText, String continueText, String closeText, DialogInterface.OnClickListener listener) {
        adView.pauseVideo();
        adView.showDialog(titleText, bodyText, continueText, closeText, listener);
    }

    private void continueWithPostroll() {
        if (advertisement.hasPostroll()) {
            playPost();
        } else {
            closeAndReport();
        }
    }

    private boolean isWebPageBlank() {
        String url = adView.getWebsiteUrl();
        return TextUtils.isEmpty(url) || "about:blank".equalsIgnoreCase(url);
    }

    @Override
    public void start() {
        durationRecorder.start();

        if (!adView.hasWebView()) {
            reportErrorAndCloseAd(VungleException.WEB_CRASH);
            return;
        }

        adView.setImmersiveMode();
        adView.resumeWeb();

        /// Check for GDPR Consent State
        Cookie gdprConsent = cookies.get(CONSENT_COOKIE);
        if (needShowGDPR(gdprConsent)) {
            showGDPR(gdprConsent);
            return;
        }

        if (inPost) {
            if (isWebPageBlank()) {
                playPost();
            }
        } else if (!adView.isVideoPlaying() && !adView.isDialogVisible()) {
            /// Load the video assets, extract the URI, hand it off to the ad view
            File video = new File(assetDir.getPath() + File.separator + Advertisement.KEY_VIDEO);
            adView.playVideo(video, muted, videoPosition);

            /// Enable the back button once the delay has elapsed. If no delay, enable immediately
            int delayInMillis = advertisement.getShowCloseDelay(placement.isIncentivized());
            if (delayInMillis > 0) {
                /// Use a simple timer to enact the delay mechanism
                scheduler.schedule(new Runnable() {
                    @Override
                    public void run() {
                        userExitEnabled = true;
                        if (!inPost) {    //May be delay expired after Postroll
                            adView.showCloseButton();
                        }
                    }
                }, delayInMillis);
            } else {
                userExitEnabled = true;
                adView.showCloseButton();
            }
        }
    }

    @Override
    public void stop(@AdContract.AdStopReason int stopReason) {
        durationRecorder.stop();

        boolean isChangingConfigurations = (stopReason & AdContract.AdStopReason.IS_CHANGING_CONFIGURATION) != 0;
        boolean isFinishing = (stopReason & AdContract.AdStopReason.IS_AD_FINISHING) != 0;

        adView.pauseWeb();

        if (adView.isVideoPlaying()) {
            videoPosition = adView.getVideoPosition();
            adView.pauseVideo();
        }

        if (!isChangingConfigurations && isFinishing) {
            if (!isDestroying.getAndSet(true)) {
                reportAction("close", null);
                scheduler.cancelAll();
                if (bus != null) {
                    //TODO: CTA click event has been replaced by onAdClick(String) so this parameter should be removed in future.
                    bus.onNext("end", report.isCTAClicked() ? "isCTAClicked" : null, placement.getId());
                }
            }
        } else if (inPost || isFinishing) {
            adView.showWebsite("about:blank");
        }
    }

    @Override
    public void onProgressUpdate(int position, float duration) {
        progress = (int) ((position / duration) * 100);
        videoPosition = position;
        durationRecorder.update();

        if (bus != null) {
            bus.onNext("percentViewed:" + progress, null, placement.getId());
        }

        reportAction("video_viewed", String.format(Locale.ENGLISH, "%d", position));

        videoTracker.onProgress(progress);

        if (progress == 100) { /// Advertisement video ended
            /// Send the completed checkpoint, if one exists.

            videoTracker.stop();

            if (checkpointList.peekLast() != null && checkpointList.peekLast().getPercentage() == 100) {
                analytics.ping(checkpointList.pollLast().getUrls());
            }

            // What is the expected action on video end?
            continueWithPostroll();
        }

        /// Update the video playback progress in ms
        report.recordProgress(videoPosition);
        repository.save(report, repoCallback);
        /// Here we check with < instead of <= because it is not deterministic if the 100
        /// percent playback event will be caught in this loop, and we send the event
        /// in the completion block.
        while (checkpointList.peek() != null && progress > checkpointList.peek().getPercentage()) {
            /// If we have passed the checkpoint threshold, ping the URL and load up the
            /// next checkpoint.
            analytics.ping(checkpointList.poll().getUrls());
        }


        Cookie configCookie = cookies.get(CONFIG_COOKIE);
        if (placement.isIncentivized() && progress > INCENTIVIZED_TRESHOLD
                && configCookie != null
                && configCookie.getBoolean("isReportIncentivizedEnabled")
                && !sendReportIncentivized.getAndSet(true)) {

            //send ri only once past 75 percentile
            JsonObject body = new JsonObject();
            body.add("placement_reference_id", new JsonPrimitive(placement.getId()));
            body.add("app_id", new JsonPrimitive(advertisement.getAppID()));
            body.add("adStartTime", new JsonPrimitive(report.getAdStartTime()));
            body.add("user", new JsonPrimitive(report.getUserID()));
            analytics.ri(body);
        }

    }

    @Override
    public void onVideoStart(int position, float duration) {
        reportAction("videoLength", String.format(Locale.ENGLISH, "%d", (int) duration));
        videoTracker.start((int) duration);
        videoTracker.setPlayerVolume(muted);
    }

    @Override
    public void onMute(boolean muted) {
        this.muted = muted;
        if (muted) {
            reportAction("mute", "true");
        } else {
            reportAction("unmute", "false");
        }
        videoTracker.setPlayerVolume(muted);
    }

    @Override
    public void onDownload() {
        download();
    }

    @Override
    public boolean onMediaError(@NonNull String description) {
        reportError(description);
        return false;
    }

    @Override
    public void onPrivacy() {
        activityManager.addOnNextAppLeftCallback(new PresenterAppLeftCallback(bus, placement));
        adView.open(HTTPS_VUNGLE_COM_PRIVACY);
    }

    @Override
    public void generateSaveState(@Nullable OptionsState state) {
        if (state == null) {
            return;
        }

        repository.save(report, repoCallback);
        state.put(EXTRA_REPORT, report == null ? null : report.getId());
        state.put(EXTRA_INCENTIVIZED_SENT, sendReportIncentivized.get());
        state.put(EXTRA_IN_POST, inPost);
        state.put(EXTRA_IS_MUTED, muted);
        state.put(EXTRA_VIDEO_POSITION, (adView != null && adView.isVideoPlaying()) ? adView.getVideoPosition() : videoPosition);
    }

    @Override
    public void restoreFromSave(@Nullable OptionsState state) {
        if (state == null) {
            return;
        }

        boolean isIncentivizedShown = state.getBoolean(EXTRA_INCENTIVIZED_SENT, false);
        if (isIncentivizedShown) {
            sendReportIncentivized.set(true);
        }

        inPost = state.getBoolean(EXTRA_IN_POST, inPost);
        muted = state.getBoolean(EXTRA_IS_MUTED, muted);
        videoPosition = state.getInt(EXTRA_VIDEO_POSITION, videoPosition);
    }

    @Override
    public void onMraidAction(@NonNull String action) {
        switch (action) {
            case CLOSE_ACTION: {
                closeAndReport();
                break;
            }

            case DOWNLOAD_ACTION: {
                download();
                closeAndReport();
                break;
            }

            case PRIVACY_ACTION: {
                break;
            }

            default:
                throw new IllegalArgumentException("Unknown action " + action);
        }
    }

    private void download() {
        reportAction("cta", "");

        try {
            analytics.ping(advertisement.getTpatUrls(postrollClick));
            analytics.ping(advertisement.getTpatUrls(clickUrl));
            analytics.ping(advertisement.getTpatUrls(videoClick));
            analytics.ping(new String[]{advertisement.getCTAURL(true)});

            reportAction("download", null);
            //we need to call adView.close which call finish, this has to happen before adView.open that launches another activity
            //so we load isFinishing to be true to send onAdEnd callback

            String url = advertisement.getCTAURL(false);
            activityManager.addOnNextAppLeftCallback(new PresenterAppLeftCallback(bus, placement));
            if(url == null || url.isEmpty())  {
                Log.e(TAG, "CTA destination URL is not configured properly");
            } else {
                adView.open(url);
            }

            if (bus != null) {
                bus.onNext("open", "adClick", placement.getId());
            }
        } catch (ActivityNotFoundException invalid) {
            Log.e(TAG, "Unable to find destination activity");
        }
    }

    private void closeAndReport() {
        /// Close the advertisement, finish the activity.

        if (adView.isVideoPlaying()) {
            videoTracker.stop();
        }

        if (busy.get()) {
            Log.w(TAG, "Busy with closing");
            return;
        }

        busy.set(true);
        reportAction("close", null);
        scheduler.cancelAll();

        adView.close();
    }

    private AtomicBoolean busy = new AtomicBoolean(false);


    private void loadData(OptionsState optionsState) {
        cookies.put(INCENTIVIZED_TEXT_COOKIE, repository.load(INCENTIVIZED_TEXT_COOKIE, Cookie.class).get());
        cookies.put(CONSENT_COOKIE, repository.load(CONSENT_COOKIE, Cookie.class).get());
        cookies.put(CONFIG_COOKIE, repository.load(CONFIG_COOKIE, Cookie.class).get());

        if (optionsState != null) {
            String reportId = optionsState.getString(EXTRA_REPORT);
            Report restoredReport = TextUtils.isEmpty(reportId) ? null : repository.load(reportId, Report.class).get();

            if (restoredReport != null) {
                report = restoredReport;
            }
        }
    }

    //Errors while loading postroll.
    @Override
    public void onReceivedError(String errorDesc) {
        if (report != null) {
            report.recordError(errorDesc);
            repository.save(report, repoCallback);
        }
    }

    @Override
    public boolean onWebRenderingProcessGone(WebView view, boolean didCrash) {
        handleWebViewException(VungleException.WEB_CRASH);
        return true;
    }

    @Override
    public void onRenderProcessUnresponsive(@NonNull WebView webView, @Nullable WebViewRenderProcess webViewRenderProcess) {
        handleWebViewException(VungleException.WEBVIEW_RENDER_UNRESPONSIVE);
    }

    private void handleWebViewException(@VungleException.ExceptionCode int reason) {
        if (adView != null) {
            adView.removeWebView();
        }
        reportErrorAndCloseAd(reason);
    }

    private void reportErrorAndCloseAd(@VungleException.ExceptionCode int reason) {
        // TODO reportAction(reason, getLocalizedMessage() );
        makeBusError(reason);
        closeAndReport();
    }

    private void makeBusError(@VungleException.ExceptionCode int code) {
        if (bus != null) {
            bus.onError(new VungleException(code), placement.getId());
        }
    }
}
