package com.vungle.warren.ui.contract;

import android.content.DialogInterface;
import androidx.annotation.IntDef;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.vungle.warren.error.VungleException;
import com.vungle.warren.ui.JavascriptBridge;
import com.vungle.warren.ui.state.OptionsState;

public interface AdContract {
    /**
     * Interface that allows us to define an AdView, whether it be a fragment, Activity, or custom View.
     * This allows our {@link AdvertisementPresenter} to render the ad in any
     * container that adheres to this contract.
     */
    interface AdView<T extends AdvertisementPresenter> {

        void setPresenter(@NonNull T presenter);

        /**
         * Set the screen orientation for this view. Advertisements can be presented in LANDSCAPE,
         * PORTRAIT, or AUTO_ROTATE orientations.
         *
         * @param orientation The desired orientation.
         */
        void setOrientation(int orientation);

        /**
         * Show the given website in the view.
         *
         * @param url The location of the website's index.html
         */
        void showWebsite(@NonNull String url);

        /**
         * returns the url of website loaded in the webview.
         */
        String getWebsiteUrl();

        /**
         * Close the view and free up any resources being occupied by the view.
         */
        void close();

        /**
         * Display the close button so that the end user can interact with it. Once the button is visible,
         * it also needs to be enabled.
         */
        void showCloseButton();

        /**
         * Opens the url by starting a new activity with it. This could be an external website, a link
         * to the start store, or a link to an external app page.
         *
         * @param url The URL to open
         */
        void open(@NonNull String url);

        /**
         * Shows a dialog to the user in order to inform them about a consequence of their actions.
         *
         * @param title            The title of the dialog
         * @param body             The body of the dialog
         * @param resume           The text of the "continue watching" button
         * @param close            The text of the "close" button.
         * @param responseListener
         */
        void showDialog(@Nullable String title,
                        @Nullable String body,
                        @NonNull String resume,
                        @NonNull String close,
                        @Nullable DialogInterface.OnClickListener responseListener);

        /**
         * Turn on android full screen mode
         */
        void setImmersiveMode();

        /**
         * Resume web components after pause
         */
        void resumeWeb();

        /**
         * Pause web components
         */
        void pauseWeb();

        /**
         * Removes contained {@link android.webkit.WebView} from this view.
         * Or if this view inherits {@link android.webkit.WebView} it
         * will attempt to remove self from parent.
         */
        void removeWebView();

        /**
         * Identify if this view has/is {@link android.webkit.WebView}
         *
         * @return {@code true} if view contains or is WebView, {@code false} otherwise
         */
        boolean hasWebView();

        /**
         * Releases current view and its resources
         */
        void destroyAdView();
    }

    /**
     * Advertisement Rendering contract. This Presenter interface defines the way that we render advertisements
     * into a view using the {@link com.vungle.warren.model.Advertisement} model.
     */
    interface AdvertisementPresenter<T extends AdView> extends JavascriptBridge.MraidHandler {

        /**
         * Attach a view to this presenter. This allows the presenter to manipulate the view state and
         * act on user interactions.
         * Prepares the presenter by initializing the required fields and doing any work required to present
         * the advertisement.
         * This can include validating disk state, unzipping files for playback, etc.
         *
         * @param adView The view which will use this presenter
         * @param state
         */
        void attach(@NonNull T adView, @Nullable OptionsState state);

        void detach(boolean changingConfiguration);

        /**
         * The user has clicked the back button, the presenter should handle the appropriate action and
         * inform the view of what to do next.
         *
         * @param flexViewCloseApiPlacementId - parameter is needed to determine if someone calls
         *                                    closeflexview, whether activity should close and disregard
         *                                    if flexview id isn't the id to close
         *
         * @return true if the advertisement is ending and the view will be dismissed
         */
        boolean handleExit(@Nullable String flexViewCloseApiPlacementId); /// refactor to handleExit()

        /**
         * Starts the advertisement.
         */
        void start();

        /**
         * Stops the video. After this, it is no longer possible to resume the video.
         *
         * @param int bitmask which combines three boolean
         *            {@link AdStopReason#IS_CHANGING_CONFIGURATION} : Stop because of configuration changed
         *            {@link AdStopReason#IS_AD_FINISHING} : stop because of Activity Finishing
         *            {@link AdStopReason#IS_AD_FINISHED_BY_API} : stop using external sdk apis
         */
        void stop(@AdStopReason int stopReason);

        /**
         * Save the presenter state so that it can be restored later.
         *
         * @return An identifier for the save state that can be used to restore from.
         */
        void generateSaveState(@Nullable OptionsState state);

        /**
         * Restores the save state of this presenter, allowing the advertisement to resume from where it
         * left off.
         */
        void restoreFromSave(@Nullable OptionsState state);

        /**
         * Provide an event listener to the presenter. This event listener emits events when certain
         * requirements are met. e.g. Emits updates on the percentage of the advertisement that has been
         * watched.
         *
         * @param listener The EventListener to send events to.
         */
        void setEventListener(@Nullable EventListener listener);

        /**
         * This will facilitate View telling presenter that one or more properties have changed
         * and Presenter should adapt accordingly
         * Currently this is being used for Orientation change, but can be expanded for other usecases
         */
        void onViewConfigurationChanged();
        
        /**
         * The EventListener contract. Any object that wishes to listen to advertisement lifecycle events
         * must implement this contract.
         */
        interface EventListener {
            /**
             * Emits the event value to all listeners.
             *
             * @param event The event that occurred.
             */
            void onNext(@NonNull String event, @Nullable String value, @Nullable String id);

            /**
             * Emits the error to all listeners.
             *
             * @param exception VungleException that was encountered.
             */
            void onError(@NonNull VungleException exception, @Nullable String id);
        }
    }

    interface AdvertisementBus {
        String ACTION = "AdvertisementBus";
        String PLACEMENT = "placement";
        String COMMAND = "command";
        String CLOSE_FLEX = "closeFlex";
        String STOP_ALL = "stopAll";
    }

    @IntDef(flag = true, value = {AdStopReason.IS_CHANGING_CONFIGURATION, AdStopReason.IS_AD_FINISHING, AdStopReason.IS_AD_FINISHED_BY_API})
    @interface AdStopReason {
        int IS_CHANGING_CONFIGURATION = 1;     //1 << 0
        int IS_AD_FINISHING = 1 << 1;
        int IS_AD_FINISHED_BY_API = 1 << 2;
    }
}
