package com.vungle.warren.tasks;

import android.os.Bundle;
import androidx.annotation.NonNull;

import com.vungle.warren.AdLoader;
import com.vungle.warren.VungleStaticApi;

import java.util.Collection;

/**
 * Download Job is responsible for downloading the data from a given URL to the given file. It will
 * schedule this work for when there is network connectivity available.
 */

public class DownloadJob implements Job {

    static final String TAG = DownloadJob.class.getCanonicalName();
    private static final String PLACEMENT_KEY = "placement";
    private final AdLoader adLoader;
    private final VungleStaticApi vungleApi;

    public DownloadJob(@NonNull AdLoader adLoader, @NonNull VungleStaticApi vungleApi) {
        this.adLoader = adLoader;
        this.vungleApi = vungleApi;
    }

    public static JobInfo makeJobInfo(String placementID) {
        Bundle extras = new Bundle();
        extras.putString(PLACEMENT_KEY, placementID);

        String tag = TAG + " " + placementID;

        return new JobInfo(tag)
                .setUpdateCurrent(true)
                .setExtras(extras)
                .setPriority(JobInfo.Priority.HIGHEST);
    }

    @Override
    @Result
    public int onRunJob(Bundle params, JobRunner jobRunner) {
        String placementID = params.getString(PLACEMENT_KEY, null);

        /*
         * It is possible that a job was scheduled to download assets for a placement that is no
         * longer valid. Make sure that we're not attempting to download assets for an invalid
         * placement here.
         */
        Collection<String> validPlacements = vungleApi.getValidPlacements();

        if (placementID == null || !validPlacements.contains(placementID)) {
            return Result.FAILURE;
        }

        adLoader.loadPendingInternal(placementID);

        return Result.SUCCESS;
    }
}
