package com.vungle.warren.persistence;

import android.Manifest;
import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.DatabaseUtils;
import android.database.sqlite.SQLiteDatabase;
import android.os.Build;
import android.os.Environment;
import android.text.TextUtils;
import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import androidx.core.content.PermissionChecker;

import com.vungle.warren.AdLoader;
import com.vungle.warren.model.AdAsset;
import com.vungle.warren.model.AdAssetDBAdapter;
import com.vungle.warren.model.Advertisement;
import com.vungle.warren.model.AdvertisementDBAdapter;
import com.vungle.warren.model.Cookie;
import com.vungle.warren.model.CookieDBAdapter;
import com.vungle.warren.model.Placement;
import com.vungle.warren.model.PlacementDBAdapter;
import com.vungle.warren.model.Report;
import com.vungle.warren.model.ReportDBAdapter;
import com.vungle.warren.model.VisionData;
import com.vungle.warren.model.VisionDataDBAdapter;
import com.vungle.warren.utility.FileUtility;
import com.vungle.warren.vision.VisionAggregationData;
import com.vungle.warren.vision.VisionAggregationInfo;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;

import static com.vungle.warren.model.Placement.TYPE_DEFAULT;
import static com.vungle.warren.model.PlacementDBAdapter.PlacementColumns.COLUMN_AD_SIZE;
import static com.vungle.warren.model.PlacementDBAdapter.PlacementColumns.COLUMN_REFRESH_DURATION;
import static com.vungle.warren.model.PlacementDBAdapter.PlacementColumns.COLUMN_SUPPORTED_TEMPLATE_TYPES;
import static com.vungle.warren.model.ReportDBAdapter.ReportColumns.COLUMN_REPORT_STATUS;
import static com.vungle.warren.model.ReportDBAdapter.ReportColumns.COLUMN_TT_DOWNLOAD;

public class Repository {

    private static final String TAG = Repository.class.getSimpleName();

    @VisibleForTesting
    protected DatabaseHelper dbHelper;
    private final ExecutorService backgroundExecutor;
    private final ExecutorService uiExecutor;
    private final Designer designer;
    private final Context appCtx;
    public static int VERSION = 5;

    private Map<Class, DBAdapter> adapters = new HashMap<>();

    public Repository(Context context, Designer designer, ExecutorService backgroundExecutor, ExecutorService uiExecutor) {
        this(context, designer, backgroundExecutor, uiExecutor, VERSION);
    }

    public Repository(Context context, Designer designer, ExecutorService backgroundExecutor, ExecutorService uiExecutor, int version) {
        this.appCtx = context.getApplicationContext();
        this.backgroundExecutor = backgroundExecutor;
        this.uiExecutor = uiExecutor;

        dbHelper = new DatabaseHelper(context, version, new VungleDatabaseCreator(appCtx));
        this.designer = designer;

        adapters.put(Placement.class, new PlacementDBAdapter());
        adapters.put(Cookie.class, new CookieDBAdapter());
        adapters.put(Report.class, new ReportDBAdapter());
        adapters.put(Advertisement.class, new AdvertisementDBAdapter());
        adapters.put(AdAsset.class, new AdAssetDBAdapter());
        adapters.put(VisionData.class, new VisionDataDBAdapter());
    }

    public void init() throws DatabaseHelper.DBException {
        runAndWait(new Callable<Void>() {
            @Override
            public Void call() throws Exception {
                dbHelper.init();
                ContentValues contentValues = new ContentValues();
                contentValues.put(AdvertisementDBAdapter.AdvertisementColumns.COLUMN_STATE, Advertisement.DONE);

                Query query = new Query(AdvertisementDBAdapter.AdvertisementColumns.TABLE_NAME);
                query.selection = AdvertisementDBAdapter.AdvertisementColumns.COLUMN_STATE + "=?";
                query.args = new String[]{String.valueOf(Advertisement.VIEWING)};

                dbHelper.update(query, contentValues);
                return null;
            }
        });
    }

    private <T> List<T> loadAllModels(Class<T> tClass) {
        DBAdapter<T> adapter = adapters.get(tClass);
        if (adapter == null) return Collections.EMPTY_LIST;

        Cursor cursor = dbHelper.query(new Query(adapter.tableName()));

        return extractModels(tClass, cursor);
    }

    @NonNull
    private <T> List<T> extractModels(Class<T> clazz, Cursor cursor) {
        if (cursor == null || cursor.isClosed()) {
            return Collections.emptyList();
        }

        List<T> items = new ArrayList<>();

        try {
            DBAdapter<T> adapter = adapters.get(clazz);

            while (cursor.moveToNext()) {
                ContentValues values = new ContentValues();
                DatabaseUtils.cursorRowToContentValues(cursor, values);
                items.add(adapter.fromContentValues(values));
            }
        } finally {
            cursor.close();
        }

        return items;
    }

    private <T> T loadModel(String id, Class<T> tClass) {
        DBAdapter<T> adapter = adapters.get(tClass);

        Query query = new Query(adapter.tableName());
        query.selection = IdColumns.COLUMN_IDENTIFIER + " = ? ";
        query.args = new String[]{id};

        Cursor cursor = dbHelper.query(query);

        if (cursor != null) {
            try {
                if (cursor.moveToNext()) {
                    ContentValues values = new ContentValues();
                    DatabaseUtils.cursorRowToContentValues(cursor, values);
                    return adapter.fromContentValues(values);
                }
            } finally {
                cursor.close();
            }
        }

        return null;
    }

    private <T> void saveModel(T model) throws DatabaseHelper.DBException {
        DBAdapter<T> adapter = adapters.get(model.getClass());
        ContentValues contentValues = adapter.toContentValues(model);
        dbHelper.insertWithConflict(adapter.tableName(), contentValues, SQLiteDatabase.CONFLICT_REPLACE);
    }

    public <T> FutureResult<T> load(@NonNull final String id, @NonNull final Class<T> clazz) {
        return new FutureResult<>(backgroundExecutor.submit(new Callable<T>() {
            @Override
            public T call() {
                return loadModel(id, clazz);
            }
        })
        );
    }

    public <T> void load(@NonNull final String id, @NonNull final Class<T> clazz, @NonNull final LoadCallback<T> loadCallback) {
        backgroundExecutor.execute(new Runnable() {
            @Override
            public void run() {
                final T result = loadModel(id, clazz);
                uiExecutor.execute(new Runnable() {
                    @Override
                    public void run() {
                        loadCallback.onLoaded(result);
                    }
                });
            }
        });
    }

    public <T> void save(final T item) throws DatabaseHelper.DBException {
        runAndWait(new Callable<Void>() {
            @Override
            public Void call() throws Exception {
                saveModel(item);
                return null;
            }
        });
    }

    public <T> void save(final T item, @Nullable final SaveCallback callback) {
        try {
            backgroundExecutor.submit(new Runnable() {
                @Override
                public void run() {
                    try {
                        saveModel(item);
                    } catch (final DatabaseHelper.DBException e) {
                        if (callback != null) {
                            uiExecutor.execute(new Runnable() {
                                @Override
                                public void run() {
                                    callback.onError(e);
                                }
                            });
                        }
                        return;
                    }
                    if (callback != null) {
                        uiExecutor.execute(new Runnable() {
                            @Override
                            public void run() {
                                callback.onSaved();
                            }
                        });
                    }
                }
            }).get();
        } catch (InterruptedException e) {
            Log.e(TAG, "InterruptedException ", e);
            Thread.currentThread().interrupt();
        } catch (ExecutionException e) {
            e.printStackTrace();
        }
    }

    /**
     * Looks for valid advertisement for {@link Placement} id.
     *
     * @param placementId
     * @return non expired {@link Advertisement} with {@link Advertisement#READY}
     * or {@link Advertisement#NEW} states
     */
    public FutureResult<Advertisement> findValidAdvertisementForPlacement(final String placementId) {
        return new FutureResult<>(backgroundExecutor.submit(new Callable<Advertisement>() {
            @Override
            public Advertisement call() {
                Log.i(TAG, " Searching for valid adv for pl " + placementId);

                Query query = new Query(AdvertisementDBAdapter.AdvertisementColumns.TABLE_NAME);
                query.selection = AdvertisementDBAdapter.AdvertisementColumns.COLUMN_PLACEMENT_ID + " = ? " +
                        "AND (" + AdvertisementDBAdapter.AdvertisementColumns.COLUMN_STATE + " = ? OR  " +
                        AdvertisementDBAdapter.AdvertisementColumns.COLUMN_STATE + " = ?) " +
                        "AND " + AdvertisementDBAdapter.AdvertisementColumns.COLUMN_EXPIRE_TIME + " > ?";
                query.args = new String[]{placementId, String.valueOf(Advertisement.READY), String.valueOf(Advertisement.NEW), String.valueOf(System.currentTimeMillis() / 1000l)};
                query.limit = "1";
                query.orderBy = AdvertisementDBAdapter.AdvertisementColumns.COLUMN_STATE + " DESC";
                Cursor cursor = dbHelper.query(query);

                AdvertisementDBAdapter adapter = (AdvertisementDBAdapter) adapters.get(Advertisement.class);
                List<Advertisement> items = new ArrayList<>();
                while (cursor != null && adapter != null && cursor.moveToNext()) {
                    ContentValues values = new ContentValues();
                    DatabaseUtils.cursorRowToContentValues(cursor, values);
                    items.add(adapter.fromContentValues(values));
                }

                if (cursor != null) {
                    cursor.close();
                }

                Advertisement result = items.size() > 0 ? items.get(0) : null;

                Log.i(TAG, result == null ? "Didn't find valid adv" : ("Found valid adv " + result.getId()));

                return result;
            }
        }));
    }

    public <T> FutureResult<List<T>> loadAll(final Class<T> clazz) {

        return new FutureResult<>(backgroundExecutor.submit(new Callable<List<T>>() {
            @Override
            public List<T> call() {
                return loadAllModels(clazz);
            }
        }));
    }

    public @Nullable
    FutureResult<List<Report>> loadAllReportToSend() {
        return new FutureResult<>(backgroundExecutor.submit(new Callable<List<Report>>() {
            @Override
            public List<Report> call() {
                List<Report> sendReports = loadAllModels(Report.class);
                for (Report report : sendReports) {
                    report.setStatus(Report.SENDING);
                    try {
                        saveModel(report);
                    } catch (DatabaseHelper.DBException e) {
                        return null;
                    }
                }
                return sendReports;
            }
        }));
    }

    public @Nullable
    FutureResult<List<Report>> loadReadyOrFailedReportToSend() {
        //First set any existing report to Sending status whose status is Ready/Failed, then update it to Sending

        return new FutureResult<>(backgroundExecutor.submit(new Callable<List<Report>>() {
            @Override
            public List<Report> call() {
                Query query = new Query(ReportDBAdapter.ReportColumns.TABLE_NAME);
                query.selection =
                        COLUMN_REPORT_STATUS + " = ? " + " OR " +
                                COLUMN_REPORT_STATUS + " = ? ";
                query.args = new String[]{String.valueOf(Report.READY), String.valueOf(Report.FAILED)};
                final Cursor cursor = dbHelper.query(query);
                List<Report> sendReports = extractModels(Report.class, cursor);
                for (Report report : sendReports) {
                    report.setStatus(Report.SENDING);
                    try {
                        saveModel(report);
                    } catch (DatabaseHelper.DBException e) {
                        return null;
                    }
                }
                return sendReports;
            }
        }));
    }

    public void updateAndSaveReportState(final String placementId, final String appId, final int statusFrom, final int statusTo) throws DatabaseHelper.DBException {
        runAndWait(new Callable<Void>() {
            @Override
            public Void call() throws Exception {
                ContentValues contentValues = new ContentValues();
                contentValues.put(COLUMN_REPORT_STATUS, statusTo);

                Query query = new Query(ReportDBAdapter.ReportColumns.TABLE_NAME);
                query.selection = ReportDBAdapter.ReportColumns.COLUMN_PLACEMENT_ID + " = ? " + " AND " +
                        COLUMN_REPORT_STATUS + " = ? " + " AND " +
                        ReportDBAdapter.ReportColumns.COLUMN_APP_ID + " = ? ";
                query.args = new String[]{placementId, String.valueOf(statusFrom), appId};

                dbHelper.update(query, contentValues);
                return null;
            }
        });
    }

    public FutureResult<List<AdAsset>> loadAllAdAssets(@NonNull final String adId) {
        return new FutureResult<>(backgroundExecutor.submit(new Callable<List<AdAsset>>() {
            @Override
            public List<AdAsset> call() {
                return loadAllAdAssetModels(adId);
            }
        }));
    }

    private List<AdAsset> loadAllAdAssetModels(@NonNull String adId) {
        Query query = new Query(AdAssetDBAdapter.AdAssetColumns.TABLE_NAME);
        query.selection = AdAssetDBAdapter.AdAssetColumns.COLUMN_AD_ID + " = ? ";
        query.args = new String[]{adId};
        Cursor cursor = dbHelper.query(query);
        return extractModels(AdAsset.class, cursor);
    }

    public <T> void delete(final T r) throws DatabaseHelper.DBException {
        runAndWait(new Callable<Void>() {
            @Override
            public Void call() throws Exception {
                deleteModel(r);
                return null;
            }
        });
    }

    private <T> void deleteModel(Class<T> clazz, String id) throws DatabaseHelper.DBException {
        DBAdapter<T> adapter = adapters.get(clazz);
        Query query = new Query(adapter.tableName());
        query.selection = IdColumns.COLUMN_IDENTIFIER + "=?";
        query.args = new String[]{id};
        dbHelper.delete(query);
    }

    private void deleteAssetForAdId(String adId) throws DatabaseHelper.DBException {
        DBAdapter adapter = adapters.get(AdAsset.class);
        Query query = new Query(adapter.tableName());
        query.selection = AdAssetDBAdapter.AdAssetColumns.COLUMN_AD_ID + "=?";
        query.args = new String[]{adId};
        dbHelper.delete(query);
    }

    private <T> void deleteModel(T model) throws DatabaseHelper.DBException {
        DBAdapter<T> adapter = adapters.get(model.getClass());
        ContentValues contentValues = adapter.toContentValues(model);
        deleteModel(model.getClass(), contentValues.getAsString(IdColumns.COLUMN_IDENTIFIER));
    }

    public void deleteAdvertisement(final String advertisementId) throws DatabaseHelper.DBException {
        runAndWait(new Callable<Void>() {
            @Override
            public Void call() throws Exception {
                deleteAdInternal(advertisementId);
                return null;
            }
        });
    }

    private void deleteAdInternal(String advertisementId) throws DatabaseHelper.DBException {
        if (TextUtils.isEmpty(advertisementId))
            return;

        //First deleting all Ad Assets before deleting the AD
        deleteAssetForAdId(advertisementId);
        deleteModel(Advertisement.class, advertisementId);

        try {
            designer.deleteAssets(advertisementId);
        } catch (IOException e) {
            Log.e(TAG, "IOException ", e);
        }
    }

    /**
     * Makes copy and returns currently valid placements
     *
     * @return currently valid {@link Placement} list
     */
    public FutureResult<Collection<Placement>> loadValidPlacements() {
        return new FutureResult<>(backgroundExecutor.submit(new Callable<Collection<Placement>>() {
            @Override
            public List<Placement> call() {
                synchronized (Repository.this) {
                    Query query = new Query(PlacementDBAdapter.PlacementColumns.TABLE_NAME);
                    query.selection = PlacementDBAdapter.PlacementColumns.COLUMN_IS_VALID + " = ?";
                    query.args = new String[]{"1"};

                    Cursor cursor = dbHelper.query(query);

                    return extractModels(Placement.class, cursor);
                }
            }
        }));
    }

    private List<String> loadValidPlacementIds() {
        Query query = new Query(PlacementDBAdapter.PlacementColumns.TABLE_NAME);
        query.selection = PlacementDBAdapter.PlacementColumns.COLUMN_IS_VALID + " = ?";
        query.args = new String[]{"1"};
        query.columns = new String[]{PlacementDBAdapter.PlacementColumns.COLUMN_IDENTIFIER};
        Cursor cursor = dbHelper.query(query);
        List<String> ids = new ArrayList<>();
        if (cursor != null) {
            try {
                while (cursor != null && cursor.moveToNext()) {
                    ids.add(cursor.getString(cursor.getColumnIndex(PlacementDBAdapter.PlacementColumns.COLUMN_IDENTIFIER)));
                }
            } finally {
                cursor.close();
            }
        }

        return ids;
    }

    public FutureResult<File> getAdvertisementAssetDirectory(final String id) {
        return new FutureResult<>(backgroundExecutor.submit(new Callable<File>() {
            @Override
            public File call() throws Exception {
                return designer.getAssetDirectory(id);
            }
        }));
    }

    /**
     * Makes copy and returns currently valid placements
     *
     * @return currently valid {@link Placement} list
     */
    public FutureResult<Collection<String>> getValidPlacementIds() {
        return new FutureResult<>(backgroundExecutor.submit(new Callable<Collection<String>>() {
            @Override
            public Collection<String> call() throws Exception {
                synchronized (Repository.this) {
                    return loadValidPlacementIds();
                }
            }
        })
        );
    }

    /**
     * Makes copy and returns currently available bid tokens.
     * @param maxBidTokenSize the bid tokens size limitation.
     * @return currently available {@link Advertisement} list which contains bid token.
     */
    public FutureResult<List<String>> getAvailableBidTokens(final int maxBidTokenSize) {
        return new FutureResult<>(backgroundExecutor.submit(new Callable<List<String>>() {
            @Override
            public List<String> call() throws Exception {
                synchronized (Repository.this) {
                    Query query = new Query(AdvertisementDBAdapter.AdvertisementColumns.TABLE_NAME);
                    query.selection = AdvertisementDBAdapter.AdvertisementColumns.COLUMN_BID_TOKEN + " != ''" +
                            " AND " + AdvertisementDBAdapter.AdvertisementColumns.COLUMN_STATE + " = ?" +
                            " AND " + AdvertisementDBAdapter.AdvertisementColumns.COLUMN_EXPIRE_TIME + " > ?";
                    query.columns = new String[]{AdvertisementDBAdapter.AdvertisementColumns.COLUMN_BID_TOKEN};
                    query.args = new String[]{String.valueOf(Advertisement.READY), String.valueOf(System.currentTimeMillis() / 1000L)};
                    query.limit = String.valueOf(maxBidTokenSize);
                    Cursor cursor = dbHelper.query(query);
                    List<String> bidTokens = new ArrayList<>();
                    if (cursor != null) {
                        try {
                            while (cursor.moveToNext()) {
                                int bidTokenColumnIndex = cursor.getColumnIndex(AdvertisementDBAdapter.AdvertisementColumns.COLUMN_BID_TOKEN);
                                String bidToken = cursor.getString(bidTokenColumnIndex);
                                bidTokens.add(bidToken);
                            }
                        } finally {
                            cursor.close();
                        }
                    }
                    return bidTokens;
                }
            }
        })
        );
    }

    /**
     * Sets currently valid placement objects.
     * Verifies current placement and stored.
     * Deletes assets for non-valid placements
     *
     * @param placements to save
     */
    public void setValidPlacements(final @NonNull List<Placement> placements) throws DatabaseHelper.DBException {
        runAndWait(new Callable<Void>() {
            @Override
            public Void call() throws Exception {
                synchronized (Repository.class) {
                    ContentValues contentValues = new ContentValues();
                    contentValues.put(PlacementDBAdapter.PlacementColumns.COLUMN_IS_VALID, false);

                    dbHelper.update(new Query(PlacementDBAdapter.PlacementColumns.TABLE_NAME), contentValues);

                    for (Placement placement : placements) {
                        Placement disk = loadModel(placement.getId(), Placement.class);

                        if (disk != null && disk.isIncentivized() != placement.isIncentivized()) {
                            /// if there was a placement on disk but is not equal to the one from the
                            /// server, overwrite it with the new data. In this case, we also delete any
                            /// assets we have for this placement.
                            Log.w(TAG, "Placements data for " + placement.getId()
                                    + " is different from disc, deleting old");
                            List<String> adIds = getAdsForPlacement(placement.getId());

                            for (String id : adIds) {
                                deleteAdInternal(id);
                            }

                            deleteModel(Placement.class, disk.getId());

                        }

                        //keep non-server values
                        if (disk != null) {
                            placement.setWakeupTime(disk.getWakeupTime());
                            placement.setAdSize(disk.getAdSize());
                        }

                        placement.setValid(true);
                        saveModel(placement);
                    }
                }
                return null;
            }
        });
    }

    public FutureResult<List<String>> findAdsForPlacement(final String placementId) {
        return new FutureResult<>(backgroundExecutor.submit(new Callable<List<String>>() {
            @Override
            public List<String> call() {
                return getAdsForPlacement(placementId);
            }
        }));
    }

    private List<String> getAdsForPlacement(String id) {
        Query query = new Query(AdvertisementDBAdapter.AdvertisementColumns.TABLE_NAME);
        query.columns = new String[]{AdvertisementDBAdapter.AdvertisementColumns.COLUMN_IDENTIFIER};
        query.selection = AdvertisementDBAdapter.AdvertisementColumns.COLUMN_PLACEMENT_ID + "=?";
        query.args = new String[]{id};

        Cursor cursor = dbHelper.query(query);

        List<String> ids = new ArrayList<>();
        while (cursor != null && cursor.moveToNext()) {
            ids.add(cursor.getString(cursor.getColumnIndex(AdvertisementDBAdapter.AdvertisementColumns.COLUMN_IDENTIFIER)));
        }

        if (cursor != null) {
            cursor.close();
        }

        return ids;
    }

    public void clearAllData() {
        dbHelper.dropDb();
        designer.clearCache();
    }


    public interface LoadCallback<T> {
        void onLoaded(T result);
    }

    public interface SaveCallback {
        void onSaved();

        void onError(Exception e);
    }

    /**
     * Processes both {@param advertisement} and {@param placement}
     * in appropriate states.
     * {@link Advertisement#NEW} - Saves advertisement and assigns to placement
     * {@link Advertisement#READY} - Saves advertisement in ready state
     * {@link Advertisement#VIEWING} - Saves advertisement removes from placement
     * {@link Advertisement#DONE} - Deletes advertisement, its assets and removes from placement
     * In all cases placement is saved
     *
     * @param advertisement
     * @param placementId
     * @param state         {@link Advertisement.State} to process
     */
    //Move to Vungle class
    public void saveAndApplyState(@NonNull final Advertisement advertisement,
                                  @NonNull final String placementId,
                                  @Advertisement.State final int state) throws DatabaseHelper.DBException {
        runAndWait(new Callable<Void>() {
            @Override
            public Void call() throws Exception {
                Log.i(TAG, "Setting " + state + " for adv " + advertisement.getId()
                        + " and pl " + placementId);

                advertisement.setState(state);

                switch (state) {
                    case Advertisement.NEW:
                    case Advertisement.READY:
                        advertisement.setPlacementId(placementId);
                        saveModel(advertisement);
                        //No op
                        break;

                    case Advertisement.VIEWING:
                        advertisement.setPlacementId(null);
                        saveModel(advertisement);
                        break;

                    case Advertisement.DONE:
                    case Advertisement.ERROR:
                        deleteAdInternal(advertisement.getId());
                        break;
                }
                return null;
            }
        });
    }

    /**
     * DELETE FROM vision_data WHERE _id <= (SELECT MAX(_id) FROM vision_data) - {@param size}
     */
    public void trimVisionData(final int size) throws DatabaseHelper.DBException {
        runAndWait(new Callable<Void>() {
            @Override
            public Void call() throws Exception {
                Query query = new Query(VisionDataDBAdapter.VisionDataColumns.TABLE_NAME);
                query.selection = VisionDataDBAdapter.VisionDataColumns._ID
                        + " <= ( SELECT MAX( " + VisionDataDBAdapter.VisionDataColumns._ID + " ) FROM "
                        + VisionDataDBAdapter.VisionDataColumns.TABLE_NAME + " ) - ?";
                query.args = new String[]{Integer.toString(size)};
                dbHelper.delete(query);
                return null;
            }
        });
    }

    /**
     * SELECT * FROM vision_data WHERE timestamp >= {@param after} ORDER BY _id DESC
     */
    public FutureResult<VisionAggregationInfo> getVisionAggregationInfo(final long after) {
        return new FutureResult<>(backgroundExecutor.submit(new Callable<VisionAggregationInfo>() {
            @Override
            public VisionAggregationInfo call() {
                Query query = new Query(VisionDataDBAdapter.VisionDataColumns.TABLE_NAME);
                query.selection = VisionDataDBAdapter.VisionDataColumns.COLUMN_TIMESTAMP + " >= ?";
                query.orderBy = VisionDataDBAdapter.VisionDataColumns._ID + " DESC";
                query.args = new String[]{Long.toString(after)};
                Cursor cursor = dbHelper.query(query);
                VisionDataDBAdapter adapter = (VisionDataDBAdapter) adapters.get(VisionData.class);
                if (cursor != null && adapter != null) {
                    try {
                        if (cursor.moveToFirst()) {
                            ContentValues values = new ContentValues();
                            DatabaseUtils.cursorRowToContentValues(cursor, values);
                            VisionData data = adapter.fromContentValues(values);
                            return new VisionAggregationInfo(cursor.getCount(), data.creative);
                        }
                    } finally {
                        cursor.close();
                    }
                }
                return null;
            }
        }));
    }

    /**
     * SELECT COUNT(*) as viewCount, MAX(timestamp), {@param filter} FROM vision_data WHERE timestamp >= {@param after} GROUP By {@param filter} ORDER BY _id DESC limit {@param limit}
     */
    public FutureResult<List<VisionAggregationData>> getVisionAggregationData(final long after, final int limit, final String filter) {
        return new FutureResult<>(backgroundExecutor.submit(new Callable<List<VisionAggregationData>>() {
            @Override
            public List<VisionAggregationData> call() {
                List<VisionAggregationData> list = new ArrayList<>();
                if (!VisionDataDBAdapter.VisionDataColumns.COLUMN_ADVERTISER.equals(filter)
                        && !VisionDataDBAdapter.VisionDataColumns.COLUMN_CAMPAIGN.equals(filter)
                        && !VisionDataDBAdapter.VisionDataColumns.COLUMN_CREATIVE.equals(filter))
                    return list;
                final String viewCount = "viewCount";
                final String lastTimeStamp = "lastTimeStamp";
                Query query = new Query(VisionDataDBAdapter.VisionDataColumns.TABLE_NAME);
                query.columns = new String[]{
                        "COUNT ( * ) as " + viewCount,
                        "MAX ( " + VisionDataDBAdapter.VisionDataColumns.COLUMN_TIMESTAMP + " ) as " + lastTimeStamp,
                        filter};
                query.selection = VisionDataDBAdapter.VisionDataColumns.COLUMN_TIMESTAMP + " >= ?";
                query.groupBy = filter;
                query.orderBy = VisionDataDBAdapter.VisionDataColumns._ID + " DESC";
                query.limit = Integer.toString(limit);
                query.args = new String[]{Long.toString(after)};
                Cursor cursor = dbHelper.query(query);
                if (cursor != null) {
                    try {
                        while (cursor.moveToNext()) {
                            ContentValues values = new ContentValues();
                            DatabaseUtils.cursorRowToContentValues(cursor, values);
                            list.add(new VisionAggregationData(
                                    values.getAsString(filter),
                                    values.getAsInteger(viewCount),
                                    values.getAsLong(lastTimeStamp)));
                        }
                    } finally {
                        cursor.close();
                    }
                }
                return list;
            }
        }));
    }

    private static class VungleDatabaseCreator implements DatabaseHelper.DatabaseFactory {

        private final Context context;

        public VungleDatabaseCreator(Context context) {
            this.context = context;
        }

        @Override
        public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
            //report table had status field added
            if (oldVersion < 2) {
                db.execSQL("ALTER TABLE " + ReportDBAdapter.ReportColumns.TABLE_NAME +
                        " ADD COLUMN " + COLUMN_REPORT_STATUS + " INTEGER DEFAULT " + Report.READY);
            }

            if (oldVersion < 3) {
                db.execSQL(VisionDataDBAdapter.CREATE_VISION_TABLE_QUERY);

                db.execSQL("ALTER TABLE " + ReportDBAdapter.ReportColumns.TABLE_NAME +
                        " ADD COLUMN " + ReportDBAdapter.ReportColumns.COLUMN_AD_SIZE + " TEXT ");

                db.execSQL("ALTER TABLE " + AdvertisementDBAdapter.AdvertisementColumns.TABLE_NAME +
                        " ADD COLUMN " + COLUMN_TT_DOWNLOAD + " NUMERIC DEFAULT " + -1);

                db.execSQL("ALTER TABLE " + PlacementDBAdapter.PlacementColumns.TABLE_NAME +
                        " ADD COLUMN " + COLUMN_AD_SIZE + " TEXT ");
                db.execSQL("ALTER TABLE " + PlacementDBAdapter.PlacementColumns.TABLE_NAME +
                        " ADD COLUMN " + COLUMN_REFRESH_DURATION + " NUMERIC DEFAULT " + 0);
                db.execSQL("ALTER TABLE " + PlacementDBAdapter.PlacementColumns.TABLE_NAME +
                        " ADD COLUMN " + COLUMN_SUPPORTED_TEMPLATE_TYPES + " NUMERIC DEFAULT " + TYPE_DEFAULT);
            }

            if (oldVersion < 4) {
                db.execSQL("ALTER TABLE " + PlacementDBAdapter.PlacementColumns.TABLE_NAME +
                        " ADD COLUMN " + PlacementDBAdapter.PlacementColumns.COLUMN_HEADERBIDDING + " SHORT ");
                db.execSQL("ALTER TABLE " + ReportDBAdapter.ReportColumns.TABLE_NAME +
                        " ADD COLUMN " + ReportDBAdapter.ReportColumns.COLUMN_HEADERBIDDING + " SHORT ");
            }

            if (oldVersion < 5) {
                db.execSQL("ALTER TABLE " + PlacementDBAdapter.PlacementColumns.TABLE_NAME +
                        " ADD COLUMN " + PlacementDBAdapter.PlacementColumns.COLUMN_AUTOCACHE_PRIORITY + " NUMERIC DEFAULT " + AdLoader.Priority.LOWEST);

                db.execSQL("ALTER TABLE " + AdvertisementDBAdapter.AdvertisementColumns.TABLE_NAME +
                        " ADD COLUMN " + AdvertisementDBAdapter.AdvertisementColumns.COLUMN_ASSET_DOWNLOAD_TIMESTAMP + " NUMERIC DEFAULT " + 0);

                db.execSQL("ALTER TABLE " + AdvertisementDBAdapter.AdvertisementColumns.TABLE_NAME +
                        " ADD COLUMN " + AdvertisementDBAdapter.AdvertisementColumns.COLUMN_ASSET_DOWNLOAD_DURATION + " NUMERIC DEFAULT " + 0);

                db.execSQL("ALTER TABLE " + AdvertisementDBAdapter.AdvertisementColumns.TABLE_NAME +
                        " ADD COLUMN " + AdvertisementDBAdapter.AdvertisementColumns.COLUMN_AD_REQUEST_START_TIMESTAMP + " NUMERIC DEFAULT " + 0);

                db.execSQL("ALTER TABLE " + ReportDBAdapter.ReportColumns.TABLE_NAME +
                        " ADD COLUMN " + ReportDBAdapter.ReportColumns.COLUMN_INIT_TIMESTAMP + " NUMERIC DEFAULT " + 0);

                db.execSQL("ALTER TABLE " + ReportDBAdapter.ReportColumns.TABLE_NAME +
                        " ADD COLUMN " + ReportDBAdapter.ReportColumns.COLUMN_ASSET_DOWNLOAD_DURATION + " NUMERIC DEFAULT " + 0);
            }
        }

        @Override
        public void create(SQLiteDatabase db) {
            dropOldFilesData();

            db.execSQL(AdvertisementDBAdapter.CREATE_ADVERTISEMENT_TABLE_QUERY);
            db.execSQL(PlacementDBAdapter.CREATE_PLACEMENT_TABLE_QUERY);
            db.execSQL(CookieDBAdapter.CREATE_COOKIE_TABLE_QUERY);
            db.execSQL(ReportDBAdapter.CREATE_REPORT_TABLE_QUERY);
            db.execSQL(AdAssetDBAdapter.CREATE_ASSET_TABLE_QUERY);
            db.execSQL(VisionDataDBAdapter.CREATE_VISION_TABLE_QUERY);
        }

        @Override
        public void deleteData(SQLiteDatabase db) {
            db.execSQL("DROP TABLE IF EXISTS " + AdvertisementDBAdapter.AdvertisementColumns.TABLE_NAME);
            db.execSQL("DROP TABLE IF EXISTS " + CookieDBAdapter.CookieColumns.TABLE_NAME);
            db.execSQL("DROP TABLE IF EXISTS " + PlacementDBAdapter.PlacementColumns.TABLE_NAME);
            db.execSQL("DROP TABLE IF EXISTS " + ReportDBAdapter.ReportColumns.TABLE_NAME);
            db.execSQL("DROP TABLE IF EXISTS " + AdAssetDBAdapter.AdAssetColumns.TABLE_NAME);
            db.execSQL("DROP TABLE IF EXISTS " + VisionDataDBAdapter.VisionDataColumns.TABLE_NAME);
        }

        @Override
        public void onDowngrade(SQLiteDatabase db, int oldVersion, int newVersion) {
            List<String> names = new ArrayList<>();
            Cursor cursor = db.rawQuery("SELECT * FROM sqlite_master WHERE type='table'", null);
            while (cursor != null && cursor.moveToNext()) {
                String tableName = cursor.getString(1);
                if (!tableName.equals("android_metadata") && !tableName.startsWith("sqlite_")) {
                    names.add(tableName);
                }
            }
            if (cursor != null) {
                cursor.close();
            }
            for (String name : names) {
                db.execSQL("DROP TABLE IF EXISTS " + name);
            }

            create(db);
        }

        private void deleteDatabase(String dbName) {
            context.deleteDatabase(dbName);
        }

        private void dropOldFilesData() {
            //Delete old database if exist for v5.2.x
            deleteDatabase("vungle");                       //'vungle' is the db name of SDK v5.3.x

            //Deleting external dir data for v5.2.x
            final File external = context.getExternalFilesDir(null);
            boolean canUseExternal = ((Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT
                    || (PermissionChecker.checkCallingOrSelfPermission(context, Manifest.permission.WRITE_EXTERNAL_STORAGE) == PermissionChecker.PERMISSION_GRANTED)))
                    && Environment.getExternalStorageState().equals(Environment.MEDIA_MOUNTED) && (external != null);
            if (canUseExternal && external.exists()) {
                File oldData = new File(external, ".vungle");       //'.vungle' is the assets directory name of SDK v5.3.x
                try {
                    FileUtility.delete(oldData);
                } catch (IOException e) {
                    Log.e(TAG, "IOException ", e);
                }
            }

            File file = context.getFilesDir();
            if (file.exists()) {
                File oldData = new File(file, "vungle");
                try {
                    FileUtility.delete(oldData);
                } catch (IOException e) {
                    Log.e(TAG, "IOException ", e);
                }
            }

            //Delete cache dir downloads_vungle
            try {
                FileUtility.delete(new File(context.getCacheDir() + File.separator + "downloads_vungle"));
            } catch (IOException e) {
                Log.e(TAG, "IOException ", e);
            }
        }
    }

    private void runAndWait(Callable<Void> callable) throws DatabaseHelper.DBException {
        try {
            backgroundExecutor.submit(callable).get();
        } catch (ExecutionException e) {
            if (e.getCause() instanceof DatabaseHelper.DBException) {
                throw (DatabaseHelper.DBException) e.getCause();
            }
            e.printStackTrace();
        } catch (InterruptedException e) {
            Log.e(TAG, "InterruptedException ", e);
            Thread.currentThread().interrupt();
        }
    }

    @VisibleForTesting
    public void setMockDBHelper(DatabaseHelper helper) {
        this.dbHelper = helper;
    }
}