package com.vungle.warren.persistence;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.SQLException;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;
import androidx.annotation.NonNull;


public class DatabaseHelper extends SQLiteOpenHelper {

    public static class DBException extends Exception {
        public DBException(String message) {
            super(message);
        }
    }

    public static final String DB_NAME = "vungle_db";
    private final DatabaseFactory databaseFactory;
    public DatabaseHelper(@NonNull Context context, int version, @NonNull DatabaseFactory databaseFactory) {
        // Make sure we use app context to avoid db not closed warnings. When app is terminated system should close connections
        super(context.getApplicationContext(), DB_NAME, null, version);
        this.databaseFactory = databaseFactory;
    }

    @Override
    public synchronized void onCreate(SQLiteDatabase db) {
        databaseFactory.create(db);
    }

    @Override
    public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
        databaseFactory.onUpgrade(db, oldVersion, newVersion);
        //todo
    }

    @Override
    public void onDowngrade(SQLiteDatabase db, int oldVersion, int newVersion) {
        databaseFactory.onDowngrade(db, oldVersion, newVersion);
    }

    public long insertWithConflict(String table, ContentValues values, int strategy) throws DBException {
        try {
            //insertWithOnConflict?
            return loadWritableDB().insertWithOnConflict(table, null, values, strategy);
        } catch (SQLException e) {
            throw new DBException(e.getMessage());
        }
    }

    public long update(Query query, ContentValues values) throws DBException {
        try {
            return loadWritableDB().update(query.tableName, values, query.selection, query.args);
        } catch (SQLException e) {
            throw new DBException(e.getMessage());
        }
    }

    public Cursor query(Query query) {
        return loadWritableDB().query(
                query.tableName,
                query.columns,
                query.selection,
                query.args,
                query.groupBy,
                query.having,
                query.orderBy,
                query.limit
        );
    }

    public Cursor queryRaw(String sql, String[] selectionArgs) {
        return loadWritableDB().rawQuery(sql, selectionArgs);
    }


    public void delete(Query query) throws DBException {
        try {
            loadWritableDB().delete(query.tableName, query.selection, query.args);
        } catch (SQLException e) {
            throw new DBException(e.getMessage());
        }
    }

    public void execSQL(String s) throws DBException {
        try {
            loadWritableDB().execSQL(s);
        } catch (SQLException e) {
            throw new DBException(e.getMessage());
        }
    }

    public synchronized void dropDb() {
        databaseFactory.deleteData(loadWritableDB());
        close();
        onCreate(loadWritableDB());
    }

    public void init() {
        loadWritableDB();
    }

    private synchronized SQLiteDatabase loadWritableDB() {
        return getWritableDatabase();
    }

    public interface DatabaseFactory {

        void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion);

        void create(SQLiteDatabase database);

        void deleteData(SQLiteDatabase database);

        void onDowngrade(SQLiteDatabase db, int oldVersion, int newVersion);
    }
}
