package com.vungle.warren.persistence;

import android.Manifest;
import android.content.Context;
import android.content.SharedPreferences;
import android.os.Build;
import android.os.Environment;
import android.os.FileObserver;
import android.os.StatFs;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.content.PermissionChecker;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * Selects between {@link Context#getExternalFilesDir(String)} and {@link Context#getFilesDir()} with priority to the first.
 * On every {@link CacheManager#getCache()} checks the existence of current cache directory,
 * observers external files dir and calls {@link Listener#onCacheChanged()} on any change.
 */
public class CacheManager {

    public interface Listener {
        void onCacheChanged();
    }

    private static final String COM_VUNGLE_SDK = "com.vungle.sdk";
    private static final String PATH_ID = "cache_path";
    private static final String PATH_IDS = "cache_paths";
    private static final String VUNGLE_DIR = "vungle_cache";
    private final Context context;
    private final SharedPreferences prefs;
    private Set<Listener> listeners = new HashSet<>();
    private File current;
    private List<File> old = new ArrayList<>();
    private boolean changed;
    private List<FileObserver> observers = new ArrayList<>();

    public CacheManager(@NonNull Context context) {
        this.context = context;
        prefs = context.getSharedPreferences(COM_VUNGLE_SDK, Context.MODE_PRIVATE);
    }

    private synchronized void selectFileDest() {
        if (current == null) {
            String path = prefs.getString(PATH_ID, null);
            current = (path != null ? new File(path) : null);
        }

        final File external = context.getExternalFilesDir(null);
        final File internal = context.getFilesDir();
        boolean canUseExternal = ((Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT
                || (PermissionChecker.checkCallingOrSelfPermission(context, Manifest.permission.WRITE_EXTERNAL_STORAGE) == PermissionChecker.PERMISSION_GRANTED)))
                && Environment.getExternalStorageState().equals(Environment.MEDIA_MOUNTED) && (external != null);

        File result = null;
        boolean success, created = false;

        for (File dir : Arrays.asList(new File((canUseExternal ? external : internal), VUNGLE_DIR), new File(internal, VUNGLE_DIR))) {
            //If cache dir is not dir delete it and create new dir
            if (dir.exists() && dir.isFile() && !dir.delete()) {
                break;
            }

            if (dir.exists()) {
                success = dir.isDirectory() && dir.canWrite();
            } else {
                created = success = dir.mkdirs();
            }

            if (success) {
                result = dir;
                break;
            }
        }

        final File obsoleted = context.getCacheDir();
        Set<String> known = prefs.getStringSet(PATH_IDS, new HashSet<String>());

        if (result != null) {
            known.add(result.getPath());
        }
        known.add(obsoleted.getPath());
        prefs.edit().putStringSet(PATH_IDS, known).apply();

        old.clear();
        for (String path : known) {
            if (result == null || !result.getPath().equals(path)) {
                old.add(new File(path));
            }
        }

        if (created || (result != null && !result.equals(current)) || (current != null && !current.equals(result))) {
            current = result;
            if (current != null) {
                prefs.edit().putString(PATH_ID, current.getPath()).apply();
            }

            for (Listener l : listeners) {
                l.onCacheChanged();
            }
            changed = true;
        }

        observeDirectory(external);
    }

    private void check() {
        if (current == null || !current.exists() || !current.isDirectory() || !current.canWrite()) {
            selectFileDest();
        }
    }

    private synchronized void observeDirectory(File root) {
        if (root == null) //keep observing removed sdcard to handle inserting
            return;
        observers.clear();
        observers.add(new FileObserver(root.getPath(), FileObserver.DELETE_SELF) {
            @Override
            public void onEvent(int event, @Nullable String path) {
                stopWatching();
                selectFileDest();
            }
        });
        while (root.getParent() != null) {
            final String dirName = root.getName();
            observers.add(new FileObserver(root.getParent(), FileObserver.CREATE) {
                @Override
                public void onEvent(int event, @Nullable String path) {
                    if (dirName.equals(path)) {
                        //force switching back to external
                        stopWatching();
                        selectFileDest();
                    }
                }
            });
            root = root.getParentFile();
        }
        for (FileObserver observer : observers) {
            observer.startWatching();
        }
    }

    @Nullable
    public synchronized File getCache() {
        check();
        return current;
    }

    public synchronized List<File> getOldCaches() {
        check();
        return old;
    }

    public synchronized void addListener(Listener listener) {
        check();
        listeners.add(listener);
        if (changed) {
            listener.onCacheChanged();
        }
    }

    public synchronized void removeListener(Listener listener) {
        listeners.remove(listener);
    }

    public long getBytesAvailable() {
        File dir = getCache();
        long bytesAvailable = -1;
        if (dir == null) {
            return bytesAvailable;
        }

        StatFs stats = new StatFs(dir.getPath());
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR2) {
            bytesAvailable = stats.getBlockSizeLong() * stats.getAvailableBlocksLong();
        } else {
            bytesAvailable = ((long) stats.getBlockSize()) * ((long) stats.getAvailableBlocks());
        }

        return bytesAvailable;
    }
}
