package com.vungle.warren.model;

import androidx.annotation.IntDef;
import androidx.annotation.NonNull;
import android.util.Log;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.vungle.warren.AdConfig;
import com.vungle.warren.AdLoader;

/**
 * A Placement is an object that represents a single ad configuration. Publishers use placements to
 * test ads in different locations within their app, or to try different kinds of campaigns for
 * A/B testing. The identifier is generated by the Vungle dashboard, and also contains a collection of
 * assets.
 */
public class Placement {
    private static final String TAG = "Placement";
    /**
     * Unique identifier for the placement. This is also called the placement reference ID, which is
     * generated by the Vungle dashboard when the publisher creates a placement.
     */
    String identifier;

    /**
     * Whether or not this placement is auto cached. If true, the SDK will automatically download new
     * assets for this placement after the current assets have expired for whatever reason.
     */
    boolean autoCached;


    /**
     * Whether ot not this placement is incentivized. If true, the user will be prompted to finish
     * watching the ad if they try to exit. Watching the whole ad gives them a reward of some sort.
     */
    boolean incentivized;

    /**
     * The time, in milliseconds since epoch, when the placement will be "awake" again. Before this
     * time, requesting advertisements for this placement should be blocked. This value is set
     * by the ad server when requesting an advertisement for a particular placement.
     */
    long wakeupTime;

    /**
     * Time in seconds in which a banner ad should refresh
     * 0 - No refresh
     *
     */
    int adRefreshDuration;

    /**
     * Priority for which assets should be downloaded
     * the lower the number the higher the priority
     *
     * Default Value - Integer.MAX
     */
    int autoCachePriority;

    /**
     * Whether or not this placement is support header bidding.
     */
    boolean headerBidding;

    boolean isValid;
    /**
     * Supported template types, whether template is banner or not a banner. If "banner" SDK will not
     * automatically download assets for this placement for the first time.
     */
    @Placement.PlacementAdType
    int placementAdType = TYPE_DEFAULT;

    @IntDef({TYPE_DEFAULT, TYPE_VUNGLE_BANNER})
    public @interface PlacementAdType {
    }

    public static final int TYPE_DEFAULT = 0;
    public static final int TYPE_VUNGLE_BANNER = 1;

    /**
     * Selected banner Ad Size for this placement.
     */
    protected AdConfig.AdSize adSize;
//    /**
//     * Tracks the number of advertisements that have been played during this application's lifecycle.
//     * It is stored along with the placement and extracted and deleted when creating the advertisement
//     * report. This variable is set by the publisher because it is possible for them to have multiple
//     * advertisement providers.
//     *
//     * This variable has a sentinel value of -1.
//     */
//    private int ordinal = -1;

    /**
     * Basic constructor, accepts an identifier for the placement.
     *
     * @param identifier The unique reference id of the placement.
     */
    public Placement(String identifier) {
        this.identifier = identifier;
        autoCached = false;
        incentivized = false;
        headerBidding = false;
    }

    Placement() {

    }

    /**
     * Json-parsing constructor. Parses a placement in JSON format from the ad server into a {@link Placement}
     * object. The json object is of the format:
     * <pre><code>
     *
     *  {
     *      "id": "58f8f64fcf684f7f4b0000bc",
     *      "reference_id": "DEFAULT35839",
     *      "is_auto_cached": true,
     *      "header_bidding": true,
     *      "is_incentivized": true
     *  }
     * </code></pre>
     *
     * @param jsonObject The JSON of the placement object.
     */
    public Placement(JsonObject jsonObject) throws IllegalArgumentException {
        if (jsonObject.has("reference_id")) {
            identifier = jsonObject.get("reference_id").getAsString();
        } else {
            /// Placement reference ID is necessary for a placement to be used. Missing this value
            /// causes the creation of this placement to fail.
            throw new IllegalArgumentException("Missing placement reference ID, cannot use placement!");
        }

        /// If we do not know that this placement is autocached, default to false.
        autoCached = jsonObject.has("is_auto_cached") && jsonObject.get("is_auto_cached").getAsBoolean();

        if (jsonObject.has("cache_priority") && autoCached) {
            try {
                autoCachePriority = jsonObject.get("cache_priority").getAsInt();
                if (autoCachePriority < 1) {
                    autoCachePriority = AdLoader.Priority.LOWEST;
                }
            } catch (Exception e) {
                autoCachePriority = AdLoader.Priority.LOWEST;
            }
        } else {
            autoCachePriority = AdLoader.Priority.LOWEST;
        }

        /// If we do not know if this placement is incentivized or not, default to false.
        incentivized = jsonObject.has("is_incentivized") && jsonObject.get("is_incentivized").getAsBoolean();

        adRefreshDuration = jsonObject.has("ad_refresh_duration") ? jsonObject.get("ad_refresh_duration").getAsInt() : 0;

        /// If we do not know if this placement is header bidding or not, default to false.
        headerBidding = jsonObject.has("header_bidding") && jsonObject.get("header_bidding").getAsBoolean();

        /**
         * Checking if "supported_template_types" field is available in Placement or not. If yes
         * check if "supported_template_types" is banner or not.
         */
        if (JsonUtil.hasNonNull(jsonObject, "supported_template_types")) {
            JsonArray supportedTemplatesTypesArray = jsonObject.getAsJsonArray("supported_template_types");
            for (JsonElement jsonElement : supportedTemplatesTypesArray) {
                Log.d("PlacementModel", "SupportedTemplatesTypes : " + jsonElement.getAsString());
                if (jsonElement.getAsString().equals("banner")) {
                    placementAdType = TYPE_VUNGLE_BANNER;
                    break;
                } else {
                    placementAdType = TYPE_DEFAULT;
                }
            }
        }
    }

    /**
     * Set the placement to sleep for the given amount of time. Until this time elapses, the SDK
     * will not request more advertisements to fill this placement.
     *
     * @param sleepTime The number of seconds to sleep for.
     */
    public void snooze(long sleepTime) {
        wakeupTime = System.currentTimeMillis() + sleepTime * 1000;
    }

    /**
     * @return wakeup time at which point placement is active,
     */
    public long getWakeupTime() {
        return wakeupTime;
    }

    public void setWakeupTime(long wakeupTime) {
        this.wakeupTime = wakeupTime;
    }

    @NonNull
    public String getId() {
        return identifier;
    }

    public void setValid(boolean valid) {
        isValid = valid;
    }

    public @Placement.PlacementAdType
    int getPlacementAdType() {
        return placementAdType;
    }

    public AdConfig.AdSize getAdSize() {
        return adSize == null ? AdConfig.AdSize.VUNGLE_DEFAULT : adSize;
    }

    public void setAdSize(AdConfig.AdSize adSize) {
        this.adSize = adSize;
    }

    public int getAdRefreshDuration() {
        if (adRefreshDuration <= 0) {
            return 0;
        }
        return adRefreshDuration;
    }

    public int getAutoCachePriority() {
        return autoCachePriority;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        Placement placement = (Placement) o;

        if (autoCached != placement.autoCached) return false;
        if (incentivized != placement.incentivized) return false;
        if (headerBidding != placement.headerBidding) return false;
        if (wakeupTime != placement.wakeupTime) return false;
        if (isValid != placement.isValid) return false;
        if (adRefreshDuration != placement.adRefreshDuration) return false;
        if (getAdSize() != placement.getAdSize()) return false;
        if (identifier != null ? !identifier.equals(placement.identifier) : placement.identifier != null)
            return false;

        return true;
    }

    @Override
    public int hashCode() {
        int result = identifier != null ? identifier.hashCode() : 0;
        result = 31 * result + (autoCached ? 1 : 0);
        result = 31 * result + (incentivized ? 1 : 0);
        result = 31 * result + (headerBidding ? 1 : 0);
        result = 31 * result + (int) (wakeupTime ^ (wakeupTime >>> 32));
        result = 31 * result + (adRefreshDuration ^ (adRefreshDuration >>> 32));
        result = 31 * result + (getAdSize().hashCode());
        return result;
    }

    public boolean isAutoCached() {
        if (AdConfig.AdSize.isBannerAdSize(adSize)) {
            return true;
        } else {
            return autoCached;
        }
    }

    public boolean isIncentivized() {
        return incentivized;
    }

    public boolean isHeaderBidding() {
        return headerBidding;
    }

    @Override
    public String toString() {
        return "Placement{" +
                "identifier='" + identifier + '\'' +
                ", autoCached=" + autoCached +
                ", incentivized=" + incentivized +
                ", headerBidding=" + headerBidding +
                ", wakeupTime=" + wakeupTime +
                ", refreshTime=" + adRefreshDuration +
                ", adSize=" + getAdSize().getName() +
                ", autoCachePriority=" + autoCachePriority +
                '}';
    }
}
