package com.vungle.warren.model;

import androidx.annotation.IntDef;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.StringDef;
import android.text.TextUtils;
import android.util.Log;
import android.util.Pair;
import android.webkit.URLUtil;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.annotations.SerializedName;
import com.vungle.warren.AdConfig;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import androidx.annotation.VisibleForTesting;
import okhttp3.HttpUrl;

/**
 * The Advertisement class contains all the data about a vungle advertisement. It has the download
 * links for the videos, the business logic for how the advertisement should behave, and the urls
 * that we need to hit in order to track viewability and report ad plays.
 */
public class Advertisement {

    private static final String TAG = "Advertisement";

    private static final String FILE_SCHEME = "file://";
    static final String START_MUTED = "START_MUTED";
    private static final String UNKNOWN = "unknown";

    /**
     * Enumeration of ad types. It is cheaper and faster to use an IntDef instead of an enum as it
     * reduces method count and has faster lookup, not to mention it is simpler to serialize.
     */
    @IntDef({TYPE_VUNGLE_LOCAL, TYPE_VUNGLE_MRAID})
    public @interface AdType {
    }

    public static final int TYPE_VUNGLE_LOCAL = 0;
    public static final int TYPE_VUNGLE_MRAID = 1;

    @IntDef({PORTRAIT, LANDSCAPE})
    public @interface Orientation {
    }

    public static final int PORTRAIT = 0;
    public static final int LANDSCAPE = 1;

    @StringDef({KEY_POSTROLL, KEY_VIDEO, KEY_TEMPLATE})
    public @interface CacheKey {
    }

    public static final String KEY_POSTROLL = "postroll";
    public static final String KEY_VIDEO = "video";
    public static final String KEY_TEMPLATE = "template";

    @IntDef({NEW, READY, VIEWING, DONE, ERROR})
    public @interface State {
    }

    public static final int NEW = 0;
    public static final int READY = 1;
    public static final int VIEWING = 2;
    public static final int DONE = 3;
    public static final int ERROR = 4;

    /**
     * The type of advertisement this is. We have vungle local ads, which are stored videos and
     * MRAID ads, which are reactive javascript ads. These are always streaming at the time of the
     * ad playing, but their metadata can be loaded ahead of time.
     */
    @AdType
    int adType;

    /**
     * Ad identifier, this is a unique value across all advertisements in the vungle system.
     */
    String identifier;

    /**
     * Legacy field, the app ID also contains the event ID of this advertisement.
     */
    String appID;

    /**
     * In seconds since epoch, the date at which this advertisement expires.
     */
    long expireTime;

    /**
     * Linked list of checkpoints allows for fast insertion and iteration when playing an ad.
     */
    List<Checkpoint> checkpoints;

    /**
     * The url to hit when the user mutes the advertisement.
     */
    String[] muteUrls;

    /**
     * The url to hit when the user unmutes the advertisement.
     */
    String[] unmuteUrls;

    /**
     * The url to hit when the user closes the advertisement.
     */
    String[] closeUrls;

    /**
     * The url to hit then the user clicks on the postroll.
     */
    String[] postRollClickUrls;

    /**
     * The url to hit when the postroll is viewed.
     */
    String[] postRollViewUrls;

    /**
     * The urls to hit when the user clicks on the the download link of an advertisement.
     */
    String[] clickUrls;

    /**
     * Urls to hit when the user click on the video.
     */
    String[] videoClickUrls;

    /**
     * The number of seconds to delay playing an ad in the placement this advertisement was played in.
     */
    int delay;

    /**
     * Advertisement campaign identifier. This is required for reporting back to the ad server.
     */
    String campaign;

    /**
     * The number of seconds to wait before showing the close button after the ad has started playing.
     */
    int showCloseDelay;

    /**
     * The number of seconds to wait before showing the close button adter the ad has started playing,
     * if the placement for this advertisement is incentivized.
     */
    int showCloseIncentivized;

    /**
     * Seconds to start the countdown at, for the skip button to be enabled.
     */
    int countdown;

    /**
     * The download URL of the video in this advertisement.
     */
    String videoUrl;

    /**
     * Width of the video, in pixels.
     */
    int videoWidth;

    /**
     * Height of the video, in pixels.
     */
    int videoHeight;

    /**
     * Checksum of the video file that we should compare against to ensure the video was downloaded
     * correctly.
     */
    String md5;

    /**
     * The URL where the postroll bundle website is stored.
     */
    String postrollBundleUrl;

    /**
     * If true, the call to action overlay should be displayed
     */
    boolean ctaOverlayEnabled;

    boolean ctaClickArea = true;

    /**
     * Call To Action URL, this will take the user to the download page of the advertised app
     * or to the store page of the advertised brand.
     */
    String ctaDestinationUrl;

    /**
     * Call to Action Tracking URL. We ping this URL when the user clicks on the call to action button
     * in order to receive credit for it.
     */
    String ctaUrl;

    AdConfig adConfig;

    /**
     * The number of times to retry downloading the assets for this advertisement before giving
     * up.
     */
    int retryCount;

    /**
     * A unique identifier for this advertisement.
     */
    String adToken;

    /**
     * The unique identifier for the video contained within the ad.
     */
    String videoIdentifier;

    /**
     * MRAID-o  nly field, the url from which to download the website.
     */
    String templateUrl;

    Map<String, String> templateSettings;
    Map<String, String> mraidFiles = new HashMap<>();
    //<name, Pair<url, extension>>
    Map<String, Pair<String, String>> cacheableAssets = new HashMap<>();

    String templateId;

    String templateType;

    /**
     * Moat Third Party Tracking:
     * Whether Moat is enabled on this ad
     */
    boolean enableMoat;

    /**
     * Moat Third Party Tracking:
     * Moat Vast Tags to be passed
     */
    String moatExtraVast;

    /**
     * Determines if CTA link needs a non market install
     */
    boolean requiresNonMarketInstall;

    /**
     * The market ID , bundle or the package name that uniquely identifies the App in a store
     * Currently this is used for DirectDownload feature
     */
    String adMarketId;

    /**
     * The String token to be used for Header bidding participation.
     */
    String bidToken;

    /**
     * State of the adv
     */
    int state = NEW;

    String placementId;

    /**
     * Time in milliseconds spent downloading all the ad unit's assets
     */
    long ttDownload;

    /**
     * Time we begin tracking assets
     */
    long assetDownloadStartTime;

    /**
     * Duration it takes to download assets (includes clever cache fetch time)
     */
    long assetDownloadDuration;

    /**
     *
     */
    long adRequestStartTime;

    Advertisement() {
    }

    public long getTtDownload() {
        return ttDownload;
    }

    public String getPlacementId() {
        return placementId;
    }

    public void setPlacementId(String placementId) {
        this.placementId = placementId;
    }

    public void setAssetDownloadStartTime(long timeStamp) {
        assetDownloadStartTime = timeStamp;
    }

    public void setFinishedDownloadingTime(long timeStamp) {
        assetDownloadDuration = timeStamp - assetDownloadStartTime;
        ttDownload = timeStamp - adRequestStartTime;
    }

    public void setAdRequestStartTime(long timeStamp) {
        adRequestStartTime = timeStamp;
    }

    public long getAssetDownloadDuration() {
        return assetDownloadDuration;
    }

    /**
     * @return true if for local ads, cta overlay is enabled
     */
    public boolean isCtaOverlayEnabled() {
        return ctaOverlayEnabled;
    }

    /**
     * @return VideoView click area enable or not
     * i.e. 1 means the entire VideoView clickable, 0 means not clickable,
     */
    public boolean getCtaClickArea() {
        return ctaClickArea;
    }

    /**
     * @return true if this url is a non-app-store url, which can be direct downloaded.
     */
    public boolean isRequiresNonMarketInstall() {
        return requiresNonMarketInstall;
    }

    /**
     * Create an Advertisement object from a JSON formatted advertisement blob.
     *
     * @param json The JSON-formatted blob that contains the data for this advertisement.
     * @throws IllegalArgumentException if the JSON parameter is malformed.
     */
    public Advertisement(@NonNull JsonObject json) throws IllegalArgumentException {
        if (!JsonUtil.hasNonNull(json, "ad_markup"))
            throw new IllegalArgumentException("JSON does not contain ad markup!");
        JsonObject adMarkup = json.getAsJsonObject("ad_markup");

        String tempVideoUrl = "";

        /// Process the ad type. Unknown ad types cannot be used by the sdk and should throw errors.
        if (JsonUtil.hasNonNull(adMarkup, "adType")) {
            String adTypeRaw = adMarkup.get("adType").getAsString();
            switch (adTypeRaw) {
                case "vungle_local":
                    adType = TYPE_VUNGLE_LOCAL;
                    /// Postroll Bundle
                    postrollBundleUrl = (JsonUtil.hasNonNull(adMarkup, "postBundle")) ?
                            adMarkup.get("postBundle").getAsString() : "";

                    /// Url for the video for local Ad
                    if (JsonUtil.hasNonNull(adMarkup, "url")) {
                        tempVideoUrl = adMarkup.get("url").getAsString();
                    }

                    /// Initialize the MRAID-only fields as empty string
                    templateSettings = new HashMap<>();
                    templateUrl = "";
                    templateId = "";
                    templateType = "";
                    break;
                case "vungle_mraid":
                    adType = TYPE_VUNGLE_MRAID;
                    postrollBundleUrl = "";

                    /// Template
                    if (JsonUtil.hasNonNull(adMarkup, "templateSettings")) {
                        templateSettings = new HashMap<>();
                        JsonObject templateJson = adMarkup.getAsJsonObject("templateSettings");
                        if (JsonUtil.hasNonNull(templateJson, "normal_replacements")) {
                            JsonObject normalReplacements = templateJson.getAsJsonObject("normal_replacements");
                            for (Map.Entry<String, JsonElement> element : normalReplacements.entrySet()) {
                                if (TextUtils.isEmpty(element.getKey()))
                                    continue;

                                String value = element.getValue() == null || element.getValue().isJsonNull() ? null : element.getValue().getAsString();
                                templateSettings.put(element.getKey(), value);
                            }
                        }

                        if (JsonUtil.hasNonNull(templateJson, "cacheable_replacements")) {
                            JsonObject cacheable = templateJson.getAsJsonObject("cacheable_replacements");
                            for (Map.Entry<String, JsonElement> element : cacheable.entrySet()) {
                                if (TextUtils.isEmpty(element.getKey()))
                                    continue;

                                if (element.getValue() == null)
                                    continue;

                                if (JsonUtil.hasNonNull(element.getValue(), "url") &&
                                        JsonUtil.hasNonNull(element.getValue(), "extension")) {
                                    String url = element.getValue().getAsJsonObject().get("url").getAsString();
                                    String ext = element.getValue().getAsJsonObject().get("extension").getAsString();
                                    cacheableAssets.put(element.getKey(), new Pair<>(url, ext));

                                    /// Url for the video for MRAID Ad
                                    if (element.getKey().equalsIgnoreCase("MAIN_VIDEO")) {
                                        tempVideoUrl = url;
                                    }
                                }
                            }
                        }
                    } else {
                        throw new IllegalArgumentException("Missing template adConfig!");
                    }

                    if (JsonUtil.hasNonNull(adMarkup, "templateId")) {
                        templateId = adMarkup.get("templateId").getAsString();
                    } else {
                        throw new IllegalArgumentException("Missing templateID!");
                    }

                    if (JsonUtil.hasNonNull(adMarkup, "template_type")) {
                        templateType = adMarkup.get("template_type").getAsString();
                    } else {
                        throw new IllegalArgumentException("Template Type missing!");
                    }

                    if (JsonUtil.hasNonNull(adMarkup, "templateURL")) {
                        templateUrl = adMarkup.get("templateURL").getAsString();
                    } else {
                        throw new IllegalArgumentException("Template URL missing!");
                    }
                    break;
                default:
                    throw new IllegalArgumentException("Unknown Ad Type " + adTypeRaw + "! Please add this ad type");
            }
        } else {
            throw new IllegalArgumentException("Advertisement did not contain an adType!");
        }

        /// Url for the video
        if (!TextUtils.isEmpty(tempVideoUrl)) {
            videoUrl = tempVideoUrl;
        } else {
            //not a required field by videoUrl
            videoUrl = "";
        }

        /// Process the ad identifier
        if (JsonUtil.hasNonNull(adMarkup, "id")) {
            identifier = adMarkup.get("id").getAsString();
        } else {
            /// If no ad identifier is provided, we cannot report this ad, so this is a fatal error
            /// when creating an ad.
            throw new IllegalArgumentException("Missing identifier, cannot process advertisement!");
        }

        /// Campaign
        if (JsonUtil.hasNonNull(adMarkup, "campaign")) {
            campaign = adMarkup.get("campaign").getAsString();
        } else {
            /// Campaign is necessary for ad reporting, so if this field is missing it is a fatal error.
            throw new IllegalArgumentException("Missing campaign information, cannot process advertisement!");
        }

        /// App ID
        if (JsonUtil.hasNonNull(adMarkup, "app_id")) {
            appID = adMarkup.get("app_id").getAsString();
        } else {
            /// App ID is necessary for ad reporting, so if this field is missing it is a fatal error.
            throw new IllegalArgumentException("Missing app Id, cannot process advertisement!");
        }

        /// Expiry Date
        if (JsonUtil.hasNonNull(adMarkup, "expiry") && !adMarkup.get("expiry").isJsonNull()) {
            long expire = adMarkup.get("expiry").getAsLong();
            if (expire > 0) {
                expireTime = expire;
            } else {
                expireTime = System.currentTimeMillis() / 1000;
            }
        } else {
            /// According to the Platform Team, the only ads that do not include an expire time are
            /// streaming ads, which should auto-expire automatically. Therefore we set the expire
            /// time here to be the current time.
            expireTime = System.currentTimeMillis() / 1000; /// expireTime is stored in seconds
        }

        /// Third-Party Ad Tracking  (optional)
        if (JsonUtil.hasNonNull(adMarkup, "tpat")) {
            JsonObject tpat = adMarkup.getAsJsonObject("tpat");

            /// Play Percentage reporting - Different between MRAID and Local
            checkpoints = new ArrayList<>(5);
            switch (adType) {
                case TYPE_VUNGLE_LOCAL:
                    if (JsonUtil.hasNonNull(tpat, "play_percentage")) {
                        JsonArray checkpointData = tpat.getAsJsonArray("play_percentage");

                        /// Sort the checkpoints in order based on the percentage at which they should be
                        /// reported.
                        for (int x = 0; x < checkpointData.size(); x++) {
                            if (checkpointData.get(x) == null)
                                continue;

                            checkpoints.add(new Checkpoint(checkpointData.get(x).getAsJsonObject()));
                        }
                        Collections.sort(checkpoints);
                    }

                    break;
                case TYPE_VUNGLE_MRAID:
                    String checkpoint = null;
                    for (int x = 0; x < 5; x++) {
                        int percent = x * 25;
                        checkpoint = String.format(Locale.ENGLISH, "checkpoint.%d", percent);
                        Checkpoint cpoint = null;
                        if (JsonUtil.hasNonNull(tpat, checkpoint)) {
                            cpoint = new Checkpoint(tpat.getAsJsonArray(checkpoint), (byte) percent);

                        }
                        checkpoints.add(x, cpoint);
                    }
                    break;
                default:
                    throw new IllegalArgumentException("Unknown Ad Type!");
            }

            //Feature does not exist anywhere on dashboard, or in history of local ad
            //Currently only for MRAID, do not remove
            /// MRAID ads also have a clickUrl
            if (JsonUtil.hasNonNull(tpat, "clickUrl")) {
                JsonArray clickUrlsJon = tpat.getAsJsonArray("clickUrl");

                clickUrls = new String[clickUrlsJon.size()];
                int x = 0;
                for (JsonElement element : clickUrlsJon) {
                    clickUrls[x++] = element.getAsString();
                }
            } else {
                clickUrls = new String[0];
            }

            /// MOAT
            if (JsonUtil.hasNonNull(tpat, "moat")) {
                JsonObject moatObject = tpat.getAsJsonObject("moat");
                enableMoat = moatObject.get("is_enabled").getAsBoolean();
                moatExtraVast = moatObject.get("extra_vast").getAsString();
            } else {
                enableMoat = false;
                moatExtraVast = "";
            }

            if (JsonUtil.hasNonNull(tpat, "video_click")) {
                JsonArray urls = tpat.getAsJsonArray("video_click");
                videoClickUrls = new String[urls.size()];
                for (int x = 0; x < urls.size(); x++) {
                    if (null == urls.get(x) || ("null").equalsIgnoreCase(urls.get(x).toString())) {
                        videoClickUrls[x] = "";
                    } else {
                        videoClickUrls[x] = urls.get(x).getAsString();
                    }
                }
            } else {
                videoClickUrls = new String[0];
            }

            /// Different keys for MRAID/Local
            String muteKey, unmuteKey, closeKey, clickKey, viewKey;
            switch (adType) {
                case TYPE_VUNGLE_LOCAL:
                    muteKey = "mute";
                    unmuteKey = "unmute";
                    closeKey = "video_close";
                    clickKey = "postroll_click";
                    viewKey = "postroll_view";
                    break;
                case TYPE_VUNGLE_MRAID:
                    muteKey = "video.mute";
                    unmuteKey = "video.unmute";
                    closeKey = "video.close";
                    clickKey = "postroll.click";
                    viewKey = "postroll.view";
                    break;
                default:
                    throw new IllegalArgumentException("Unknown AdType!");
            }
            /// User Action Events
            if (JsonUtil.hasNonNull(tpat, muteKey)) {
                JsonArray urls = tpat.getAsJsonArray(muteKey);
                muteUrls = new String[urls.size()];
                for (int x = 0; x < urls.size(); x++) {
                    if (null == urls.get(x) || ("null").equalsIgnoreCase(urls.get(x).toString())) {
                        muteUrls[x] = "";
                    } else {
                        muteUrls[x] = urls.get(x).getAsString();
                    }
                }
            } else {
                muteUrls = new String[0];
            }

            if (JsonUtil.hasNonNull(tpat, unmuteKey)) {
                JsonArray urls = tpat.getAsJsonArray(unmuteKey);
                unmuteUrls = new String[urls.size()];
                for (int x = 0; x < urls.size(); x++) {
                    if (null == urls.get(x) || ("null").equalsIgnoreCase(urls.get(x).toString())) {
                        unmuteUrls[x] = "";
                    } else {
                        unmuteUrls[x] = urls.get(x).getAsString();
                    }
                }
            } else {
                unmuteUrls = new String[0];
            }

            if (JsonUtil.hasNonNull(tpat, closeKey)) {
                JsonArray urls = tpat.getAsJsonArray(closeKey);
                closeUrls = new String[urls.size()];
                for (int x = 0; x < urls.size(); x++) {
                    if (null == urls.get(x) || ("null").equalsIgnoreCase(urls.get(x).toString())) {
                        closeUrls[x] = "";
                    } else {
                        closeUrls[x] = urls.get(x).getAsString();
                    }
                }
            } else {
                closeUrls = new String[0];
            }

            if (JsonUtil.hasNonNull(tpat, clickKey)) {
                JsonArray urls = tpat.getAsJsonArray(clickKey);
                postRollClickUrls = new String[urls.size()];
                for (int x = 0; x < urls.size(); x++) {
                    if (null == urls.get(x) || ("null").equalsIgnoreCase(urls.get(x).toString())) {
                        postRollClickUrls[x] = "";
                    } else {
                        postRollClickUrls[x] = urls.get(x).getAsString();
                    }
                }
            } else {
                postRollClickUrls = new String[0];
            }

            if (JsonUtil.hasNonNull(tpat, viewKey)) {
                JsonArray urls = tpat.getAsJsonArray(viewKey);
                postRollViewUrls = new String[urls.size()];
                for (int x = 0; x < urls.size(); x++) {
                    if (null == urls.get(x) || ("null").equalsIgnoreCase(urls.get(x).toString())) {
                        postRollViewUrls[x] = "";
                    } else {
                        postRollViewUrls[x] = urls.get(x).getAsString();
                    }
                }
            } else {
                postRollViewUrls = new String[0];
            }

        } else {
            checkpoints = new ArrayList<>();
            muteUrls = new String[0];
            closeUrls = new String[0];
            unmuteUrls = new String[0];
            postRollViewUrls = new String[0];
            postRollClickUrls = new String[0];
            clickUrls = new String[0];
            videoClickUrls = new String[0];
            enableMoat = false;
            moatExtraVast = "";
        }

        /// Ad Delay (?)
        if (JsonUtil.hasNonNull(adMarkup, "delay")) {
            delay = adMarkup.get("delay").getAsInt();
        } else {
            delay = 0;
        }

        /// Close Parameters
        if (JsonUtil.hasNonNull(adMarkup, "showClose")) {
            showCloseDelay = adMarkup.get("showClose").getAsInt();
        } else {
            showCloseDelay = 0;
        }
        if (JsonUtil.hasNonNull(adMarkup, "showCloseIncentivized")) {
            showCloseIncentivized = adMarkup.get("showCloseIncentivized").getAsInt();
        } else {
            showCloseIncentivized = 0;
        }

        /// Countdown
        if (JsonUtil.hasNonNull(adMarkup, "countdown")) {
            countdown = adMarkup.get("countdown").getAsInt();
        } else {
            countdown = 0;
        }

        /// Video Metadata
        if (JsonUtil.hasNonNull(adMarkup, "videoWidth")) {
            videoWidth = adMarkup.get("videoWidth").getAsInt();
        } else {
            throw new IllegalArgumentException("Missing video width!");
        }
        if (JsonUtil.hasNonNull(adMarkup, "videoHeight")) {
            videoHeight = adMarkup.get("videoHeight").getAsInt();
        } else {
            throw new IllegalArgumentException("Missing video height!");
        }

        /// Video MD5 Checksum (optional)
        if (JsonUtil.hasNonNull(adMarkup, "md5")) {
            md5 = adMarkup.get("md5").getAsString();
        } else {
            md5 = "";
        }

        /// Call to action parameters
        if (JsonUtil.hasNonNull(adMarkup, "cta_overlay")) {
            JsonObject cta = adMarkup.getAsJsonObject("cta_overlay");
            if (JsonUtil.hasNonNull(cta, "enabled")) {
                ctaOverlayEnabled = cta.get("enabled").getAsBoolean();
            } else {
                ctaOverlayEnabled = false;
            }
            if (JsonUtil.hasNonNull(cta, "click_area") && !cta.get("click_area").getAsString().isEmpty() && cta.get("click_area").getAsDouble() == 0) {
                ctaClickArea = false;
            }
        } else {
            /// I think we should be able to continue even without this data as long as the defaults
            /// are okay with the product team. TODO: Verify default behaviour with product.
            ctaOverlayEnabled = false;
        }

        /// Call to Action Destination URL
        ctaDestinationUrl = (JsonUtil.hasNonNull(adMarkup, "callToActionDest")) ?
                adMarkup.get("callToActionDest").getAsString() : "";

        /// Call to Action URL - the tracker for cta action
        ctaUrl = (JsonUtil.hasNonNull(adMarkup, "callToActionUrl")) ?
                adMarkup.get("callToActionUrl").getAsString() : "";

        /// Retry Count
        if (JsonUtil.hasNonNull(adMarkup, "retryCount")) {
            retryCount = adMarkup.get("retryCount").getAsInt();
        } else {
            retryCount = 1;
        }

        /// Ad Token - Because more identifiers are better.
        if (JsonUtil.hasNonNull(adMarkup, "ad_token")) {
            adToken = adMarkup.get("ad_token").getAsString();
        } else {
            throw new IllegalArgumentException("AdToken missing!");
        }

        /// Video Object ID - So that we don't duplicate videos
        if (JsonUtil.hasNonNull(adMarkup, "video_object_id")) {
            videoIdentifier = adMarkup.get("video_object_id").getAsString();
        } else {
            videoIdentifier = "";
        }

        /// Non Market Installs
        if (JsonUtil.hasNonNull(adMarkup, "requires_sideloading")) {
            requiresNonMarketInstall = adMarkup.get("requires_sideloading").getAsBoolean();
        } else {
            requiresNonMarketInstall = false;
        }

        if (JsonUtil.hasNonNull(adMarkup, "ad_market_id")) {
            adMarketId = adMarkup.get("ad_market_id").getAsString();
        } else {
            adMarketId = "";
        }

        if (JsonUtil.hasNonNull(adMarkup, "bid_token")) {
            bidToken = adMarkup.get("bid_token").getAsString();
        } else {
            bidToken = "";
        }

        /**
         * Initialize adConfig
         */
        adConfig = new AdConfig();
    }

    public @AdType
    int getAdType() {
        return adType;
    }

    public List<Checkpoint> getCheckpoints() {
        return checkpoints;
    }

    /**
     * Apply runtime adConfig to this Advertisement. These are set by the publisher application in
     * order to enable different ad behavior programmatically.
     *
     * @param settings bitmask that applies adConfig such as <code>MUTED</code>
     * @see
     */
    public void configure(AdConfig settings) {
        if (settings == null) {
            adConfig = new AdConfig();
        } else {
            adConfig = settings;
        }
    }

    public AdConfig getAdConfig() {
        return adConfig;
    }

    public @Orientation
    int getOrientation() {
        if (videoWidth > videoHeight) {
            return LANDSCAPE;
        } else {
            return PORTRAIT;
        }
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) return true;
        if (!(obj instanceof Advertisement)) return false;

        Advertisement wrap = (Advertisement) obj;

        if (wrap.adType != adType) return false;
        if (wrap.delay != delay) return false;
        if (wrap.showCloseDelay != showCloseDelay) return false;
        if (wrap.showCloseIncentivized != showCloseIncentivized) return false;
        if (wrap.countdown != countdown) return false;
        if (wrap.videoWidth != videoWidth) return false;
        if (wrap.videoHeight != videoHeight) return false;
        if (wrap.ctaOverlayEnabled != ctaOverlayEnabled) return false;
        if (wrap.ctaClickArea != ctaClickArea) return false;
        if (wrap.retryCount != retryCount) return false;
        if (wrap.enableMoat != enableMoat) return false;
        if (wrap.requiresNonMarketInstall != requiresNonMarketInstall) return false;
        if (wrap.state != state) return false;

        if (wrap.identifier == null || identifier == null) return false;
        if (!wrap.identifier.equals(identifier)) return false;
        if (!wrap.campaign.equals(campaign)) return false;
        if (!wrap.videoUrl.equals(videoUrl)) return false;
        if (!wrap.md5.equals(md5)) return false;
        if (!wrap.postrollBundleUrl.equals(postrollBundleUrl)) return false;
        if (!wrap.ctaDestinationUrl.equals(ctaDestinationUrl)) return false;
        if (!wrap.ctaUrl.equals(ctaUrl)) return false;
        if (!wrap.adToken.equals(adToken)) return false;
        if (!wrap.videoIdentifier.equals(videoIdentifier)) return false;
        if (!wrap.moatExtraVast.equals(moatExtraVast)) return false;
        if (!wrap.adMarketId.equals(adMarketId)) return false;
        if (!wrap.bidToken.equals(bidToken)) return false;

        if (wrap.checkpoints.size() != checkpoints.size()) return false;
        for (int x = 0; x < checkpoints.size(); x++) {
            if (!wrap.checkpoints.get(x).equals(checkpoints.get(x))) return false;
        }

        if (wrap.muteUrls.length != muteUrls.length) return false;
        for (int x = 0; x < muteUrls.length; x++) {
            if (!wrap.muteUrls[x].equals(muteUrls[x])) return false;
        }

        if (wrap.unmuteUrls.length != unmuteUrls.length) return false;
        for (int x = 0; x < unmuteUrls.length; x++) {
            if (!wrap.unmuteUrls[x].equals(unmuteUrls[x])) return false;
        }

        if (wrap.closeUrls.length != closeUrls.length) return false;
        for (int x = 0; x < closeUrls.length; x++) {
            if (!wrap.closeUrls[x].equals(closeUrls[x])) return false;
        }

        if (wrap.postRollClickUrls.length != postRollClickUrls.length) return false;
        for (int x = 0; x < postRollClickUrls.length; x++) {
            if (!wrap.postRollClickUrls[x].equals(postRollClickUrls[x])) return false;
        }

        if (wrap.postRollViewUrls.length != postRollViewUrls.length) return false;
        for (int x = 0; x < postRollViewUrls.length; x++) {
            if (!wrap.postRollViewUrls[x].equals(postRollViewUrls[x])) return false;
        }

        if (wrap.videoClickUrls.length != videoClickUrls.length) return false;
        for (int x = 0; x < videoClickUrls.length; x++) {
            if (!wrap.videoClickUrls[x].equals(videoClickUrls[x])) return false;
        }

        return true;
    }

    @Override
    public int hashCode() {
        int result = adType;
        result = 31 * result + identifier.hashCode();
        result = 31 * result + checkpoints.hashCode();
        result = 31 * result + Arrays.hashCode(muteUrls);
        result = 31 * result + Arrays.hashCode(unmuteUrls);
        result = 31 * result + Arrays.hashCode(closeUrls);
        result = 31 * result + Arrays.hashCode(postRollClickUrls);
        result = 31 * result + Arrays.hashCode(postRollViewUrls);
        result = 31 * result + Arrays.hashCode(videoClickUrls);
        result = 31 * result + delay;
        result = 31 * result + campaign.hashCode();
        result = 31 * result + showCloseDelay;
        result = 31 * result + showCloseIncentivized;
        result = 31 * result + countdown;
        result = 31 * result + videoUrl.hashCode();
        result = 31 * result + videoWidth;
        result = 31 * result + videoHeight;
        result = 31 * result + md5.hashCode();
        result = 31 * result + postrollBundleUrl.hashCode();
        result = 31 * result + (ctaOverlayEnabled ? 1 : 0);
        result = 31 * result + (ctaClickArea ? 1 : 0);
        result = 31 * result + ctaDestinationUrl.hashCode();
        result = 31 * result + ctaUrl.hashCode();
        result = 31 * result + retryCount;
        result = 31 * result + adToken.hashCode();
        result = 31 * result + videoIdentifier.hashCode();
        result = 31 * result + (enableMoat ? 1 : 0);
        result = 31 * result + moatExtraVast.hashCode();
        result = 31 * result + (requiresNonMarketInstall ? 1 : 0);
        result = 31 * result + adMarketId.hashCode();
        result = 31 * result + bidToken.hashCode();
        result = 31 * result + state;
        return result;
    }

    public String[] getTpatUrls(@NonNull String event) {
        switch (adType) {
            case TYPE_VUNGLE_LOCAL:
                switch (event) {
                    case "postroll_view":
                        return postRollViewUrls.clone();
                    case "postroll_click":
                        return postRollClickUrls.clone();
                    case "mute":
                        return muteUrls.clone();
                    case "unmute":
                        return unmuteUrls.clone();
                    case "video_close":
                        return closeUrls.clone();
                    case "click_url":
                        return clickUrls.clone();
                    case "video_click":
                        return videoClickUrls.clone();
                    default:
                        throw new IllegalArgumentException("Unknown TPAT Event " + event);
                }
            case TYPE_VUNGLE_MRAID:
                if (event.startsWith("checkpoint")) {
                    String[] ret = new String[0];
                    int percent = Integer.parseInt(event.split("\\.")[1]);
                    Checkpoint cpoint = checkpoints.get(percent / 25);
                    if (cpoint != null) {
                        ret = cpoint.getUrls();
                    }
                    return ret;
                }
                switch (event) {
                    case "video.close": {
                        return closeUrls.clone();
                    }
                    case "postroll.view": {
                        return postRollViewUrls.clone();
                    }
                    case "postroll.click": {
                        return postRollClickUrls.clone();
                    }
                    case "clickUrl": {
                        return clickUrls.clone();
                    }
                    case "video.mute": {
                        return muteUrls.clone();
                    }
                    case "video.unmute": {
                        return unmuteUrls.clone();
                    }
                    case "video_click": {
                        return videoClickUrls.clone();
                    }
                    default:
                        throw new IllegalArgumentException("Unknown TPAT Event " + event);
                }
            default:
                throw new IllegalStateException("Unknown Advertisement Type!");
        }
    }

    @NonNull
    public String getId() {
        if (identifier == null) {
            return "";
        }
        return identifier;
    }

    public String getAdToken() {
        return adToken;
    }

    public String getAppID() {
        return appID;
    }

    String getUrl() {
        return videoUrl;
    }

    public String getCampaign() {
        return campaign;
    }

    String getTemplateId() {
        return templateId;
    }

    public String getTemplateType() {
        return templateType;
    }

    /**
     * The number of milliseconds to delay showing the close button after an advertisement has started.
     *
     * @param incentivized If <code>true</code>, use the incentivized version of the close delay, otherwise
     *                     use the default.
     * @return Milliseconds of delay
     */
    public int getShowCloseDelay(boolean incentivized) {
        if (incentivized) {
            return showCloseIncentivized * 1000;
        }
        return showCloseDelay * 1000;
    }

    public boolean getMoatEnabled() {
        return enableMoat;
        //&& Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN;
    }

    public String getMoatVastExtra() {
        return moatExtraVast;
    }

    /**
     * The time, in milliseconds since epoch, at which this advertisement will expire. After this time,
     * the advertisement can no longer be shown to a user and its assets can be deleted safely.
     *
     * @return Unix time for ad expiration
     */
    public long getExpireTime() {
        return expireTime * 1000;
    }

    public JsonObject createMRAIDArgs() {
        if (templateSettings == null) {
            throw new IllegalArgumentException("Advertisement does not have MRAID Arguments!");
        }

        Map<String, String> resultMap = new HashMap<>(templateSettings);    // adding templateSettings

        for (Map.Entry<String, Pair<String, String>> entry : cacheableAssets.entrySet()) {
            resultMap.put(entry.getKey(), entry.getValue().first);            // adding cacheableAssets with http urls
        }

        if (!mraidFiles.isEmpty()) {                                        // replacing cacheableAssets with file urls
            resultMap.putAll(mraidFiles);
        }

        if (!"true".equalsIgnoreCase(resultMap.get(START_MUTED))) {
            resultMap.put(START_MUTED, ((getAdConfig().getSettings() & AdConfig.MUTED) != 0) ? "true" : "false");
        }

        JsonObject ret = new JsonObject();
        for (Map.Entry<String, String> entry : resultMap.entrySet()) {
            ret.addProperty(entry.getKey(), entry.getValue());
        }

        return ret;
    }

    /**
     * Get the URL to navigate to when the user clicks on the download button during the postroll
     * of an advertisement.
     *
     * @param tpat If <code>true</code> this method will return the TPAT Hit URL, otherwise return
     *             the destination url.
     * @return A URL to download the app being advertised, or null if there is no download link.
     */
    @Nullable
    public String getCTAURL(boolean tpat) {
        switch (adType) {
            case TYPE_VUNGLE_LOCAL:
                return tpat ? ctaUrl : ctaDestinationUrl;
            case TYPE_VUNGLE_MRAID:
                return ctaUrl;
            default:
                throw new IllegalArgumentException("Unknown AdType " + adType);
        }
    }

    /**
     * True if this advertisement has a postroll bundle. False otherwise.
     *
     * @return true if this advertisement has a postroll bundle, false otherwise.
     */
    public boolean hasPostroll() {
        return !TextUtils.isEmpty(postrollBundleUrl);
    }

    public Map<String, String> getDownloadableUrls() {
        HashMap<String, String> ret = new HashMap<>();
        switch (adType) {
            case TYPE_VUNGLE_LOCAL:
                ret.put(KEY_VIDEO, videoUrl);
                /// Postroll Bundle is optional, only include it if it exists
                if (!TextUtils.isEmpty(postrollBundleUrl)) {
                    ret.put(KEY_POSTROLL, postrollBundleUrl);
                }
                break;
            case TYPE_VUNGLE_MRAID:
                ret.put(KEY_TEMPLATE, templateUrl);

                // Adding http urls from cacheableAssets to download urls map,
                // concating ext with key to save file name with ext
                for (Map.Entry<String, Pair<String, String>> entry : cacheableAssets.entrySet()) {
                    String httpUrl = entry.getValue().first;

                    if (isValidUrl(httpUrl)) {
                        String fileName = URLUtil.guessFileName(httpUrl, null, null);
                        ret.put(fileName, httpUrl);
                    }
                }
                break;
            default:
                throw new IllegalStateException("Advertisement created without adType!");
        }

        return ret;
    }

    private boolean isValidUrl(String httpUrl) {
        return !TextUtils.isEmpty(httpUrl) && HttpUrl.parse(httpUrl) != null;
    }

    /**
     * Assign the directory where the assets are stored for an MRAID advertisement. It expects a
     * video, app icon, and vungle logo assets. These complete paths are then set as the
     * asset location in the MRAID arguments. If any of the asset files are missing, their default
     * values will remain unchanged in the MRAID arguments.
     *
     * @param dir The directory where all these files are stored.
     */
    public void setMraidAssetDir(File dir) {
        for (Map.Entry<String, Pair<String, String>> entry : cacheableAssets.entrySet()) {
            String httpUrl = entry.getValue().first;

            if (isValidUrl(httpUrl)) {
                String fileName = URLUtil.guessFileName(httpUrl, null, null);

                File file = new File(dir, fileName);
                //Saving file urls to mraidFiles map if file exists

                if (file.exists()) {
                    mraidFiles.put(entry.getKey(), FILE_SCHEME + file.getPath());
                }
            }
        }
    }

    public void setState(@State int state) {
        this.state = state;
    }

    public @State
    int getState() {
        return state;
    }

    public String getAdMarketId() {
        return adMarketId;
    }


    public String getBidToken() {
        return bidToken;
    }

    /**
     * Inner class to hold information about a checkpoint and to make them sortable so we can access
     * them sequentially rather than by key matching.
     */
    public static class Checkpoint implements Comparable<Checkpoint> {

        @SerializedName("percentage")
        private byte percentage;
        @SerializedName("urls")
        private String[] urls;

        public Checkpoint(JsonObject json) throws IllegalArgumentException {
            if (JsonUtil.hasNonNull(json, "checkpoint")) {
                percentage = (byte) (json.get("checkpoint").getAsFloat() * 100);
            } else {
                throw new IllegalArgumentException("Checkpoint missing percentage!");
            }

            if (JsonUtil.hasNonNull(json, "urls")) {
                JsonArray urlsArray = json.getAsJsonArray("urls");
                urls = new String[urlsArray.size()];
                for (int x = 0; x < urlsArray.size(); x++) {
                    if (null == urlsArray.get(x) || ("null").equalsIgnoreCase(urlsArray.get(x).toString())) {
                        urls[x] = "";
                    } else {
                        urls[x] = urlsArray.get(x).getAsString();
                    }
                }
            } else {
                throw new IllegalArgumentException("Checkpoint missing reporting URL!");
            }
        }

        public Checkpoint(JsonArray urlsArray, byte percentage) {
            if (urlsArray.size() == 0) throw new IllegalArgumentException("Empty URLS!");

            urls = new String[urlsArray.size()];
            for (int x = 0; x < urlsArray.size(); x++) {
                urls[x] = urlsArray.get(x).getAsString();
            }

            this.percentage = percentage;
        }

        public String[] getUrls() {
            return urls.clone();
        }

        public byte getPercentage() {
            return percentage;
        }

        @Override
        public int compareTo(@NonNull Checkpoint o) {
            return Float.compare(percentage, o.percentage);
        }

        @Override
        public boolean equals(Object obj) {
            if (this == obj) return true;
            if (!(obj instanceof Checkpoint)) return false;

            Checkpoint wrap = (Checkpoint) obj;

            if (wrap.percentage != percentage) return false;
            if (wrap.urls.length != urls.length) return false;

            for (int x = 0; x < urls.length; x++) {
                if (!wrap.urls[x].equals(urls[x])) return false;
            }

            return true;
        }

        @Override
        public int hashCode() {
            int result = (int) percentage;
            result = 31 * result + urls.length;
            result = 31 * result + Arrays.hashCode(urls);
            return result;
        }
    }

    @Override
    public String toString() {
        return "Advertisement{" +
                "adType=" + adType +
                ", identifier='" + identifier + '\'' +
                ", appID='" + appID + '\'' +
                ", expireTime=" + expireTime +
                ", checkpoints=" + checkpoints +
                ", muteUrls=" + Arrays.toString(muteUrls) +
                ", unmuteUrls=" + Arrays.toString(unmuteUrls) +
                ", closeUrls=" + Arrays.toString(closeUrls) +
                ", postRollClickUrls=" + Arrays.toString(postRollClickUrls) +
                ", postRollViewUrls=" + Arrays.toString(postRollViewUrls) +
                ", videoClickUrls=" + Arrays.toString(videoClickUrls) +
                ", clickUrls=" + Arrays.toString(clickUrls) +
                ", delay=" + delay +
                ", campaign='" + campaign + '\'' +
                ", showCloseDelay=" + showCloseDelay +
                ", showCloseIncentivized=" + showCloseIncentivized +
                ", countdown=" + countdown +
                ", videoUrl='" + videoUrl + '\'' +
                ", videoWidth=" + videoWidth +
                ", videoHeight=" + videoHeight +
                ", md5='" + md5 + '\'' +
                ", postrollBundleUrl='" + postrollBundleUrl + '\'' +
                ", ctaOverlayEnabled=" + ctaOverlayEnabled +
                ", ctaClickArea=" + ctaClickArea +
                ", ctaDestinationUrl='" + ctaDestinationUrl + '\'' +
                ", ctaUrl='" + ctaUrl + '\'' +
                ", adConfig=" + adConfig +
                ", retryCount=" + retryCount +
                ", adToken='" + adToken + '\'' +
                ", videoIdentifier='" + videoIdentifier + '\'' +
                ", templateUrl='" + templateUrl + '\'' +
                ", templateSettings=" + templateSettings +
                ", mraidFiles=" + mraidFiles +
                ", cacheableAssets=" + cacheableAssets +
                ", templateId='" + templateId + '\'' +
                ", templateType='" + templateType + '\'' +
                ", enableMoat=" + enableMoat +
                ", moatExtraVast='" + moatExtraVast + '\'' +
                ", requiresNonMarketInstall=" + requiresNonMarketInstall +
                ", adMarketId='" + adMarketId + '\'' +
                ", bidToken='" + bidToken + '\'' +
                ", state=" + state +
                '}';
    }


    public String getAdvertiserAppId() {
        String advertiserAppId = getAppID();
        final String rawAppIdJson = getAppID();
        if (rawAppIdJson != null && rawAppIdJson.length() > 3) {
            try {
                JSONObject appIdJson = new JSONObject(rawAppIdJson.substring(3));
                advertiserAppId = (appIdJson.isNull("app_id") ? null : appIdJson.optString("app_id", null));
            } catch (JSONException e) {
                Log.e(TAG, "JsonException : ", e);
            }
        }
        return TextUtils.isEmpty(advertiserAppId) ? UNKNOWN : advertiserAppId;
    }

    public String getCampaignId() {
        String campaignId = null;
        final String campaign = getCampaign();
        if (!TextUtils.isEmpty(campaign)) {
            String[] campaignArr = campaign.split("\\|");
            if (campaignArr.length >= 1) {
                campaignId = campaignArr[0];
            }
        }
        return TextUtils.isEmpty(campaignId) ? UNKNOWN : campaignId;
    }

    public String getCreativeId() {
        String creativeId = null;
        final String campaign = getCampaign();
        if (!TextUtils.isEmpty(campaign)) {
            String[] campaignArr = campaign.split("\\|");
            if (campaignArr.length >= 2) {
                creativeId = campaignArr[1];
            }
        }
        return TextUtils.isEmpty(creativeId) ? UNKNOWN : creativeId;
    }
}