package com.vungle.warren.error;

import androidx.annotation.IntDef;
import androidx.annotation.Keep;

/**
 * Internal error detection class for Vungle-specific business errors that the SDk should handle,
 * but be aware are errors. For public-facing error conditions that must be handled by the publisher
 * app, use {@link VungleException}.
 */
@Keep
public class VungleError extends Error {

    @Keep
    @IntDef({NO_AD_AVAILABLE, SLEEP, ID_NOT_FOUND, UNKNOWN_ERROR, MRAID_FILE_MISSING, INDEX_HTML_MISSING,
            WILL_PLAY_AD_DISABLED, CONFIG_FAILED, MALFORMED_AD_RESPONSE, INVALID_URL})
    @interface ErrorCode {}

    public static final int NO_AD_AVAILABLE = 0;
    static final int SLEEP = 1;
    public static final int ID_NOT_FOUND = 2;
    static final int UNKNOWN_ERROR = 3;
    public static final int MRAID_FILE_MISSING = 4;
    public static final int INDEX_HTML_MISSING = 5;
    public static final int WILL_PLAY_AD_DISABLED = 6;
    public static final int CONFIG_FAILED = 7;
    public static final int MALFORMED_AD_RESPONSE = 9;
    public static final int INVALID_URL = 10;

    @ErrorCode
    private int code;

    public VungleError(@ErrorCode int cause) {
        this.code = cause;
    }

    @VungleError.ErrorCode
    public int getErrorCode() {
        return code;
    }

    @Override
    public String getLocalizedMessage() {
        switch (code) {
            case ID_NOT_FOUND:
                return "No object for the given identifier was found";
            case NO_AD_AVAILABLE:
                return "No Advertisement is currently available";
            case SLEEP:
                return "Sleep Code Returned";
            case INDEX_HTML_MISSING:
                return "index.html was not found. 404!";
            case MRAID_FILE_MISSING:
                return "mraid.js was not found. 404!";
            case UNKNOWN_ERROR:
                return "Unknown Error encountered.";
            case WILL_PLAY_AD_DISABLED:
                return "/config has disabled /will_play_ad";
            case CONFIG_FAILED:
                return "/config returned an error";
            case MALFORMED_AD_RESPONSE:
                return "invalid ad response";
            case INVALID_URL:
                return "invalid url";
            default:
                throw new IllegalArgumentException("Error Code " + code + " is not recognized!");
        }
    }
}