package com.vungle.warren.downloader;

import androidx.annotation.IntDef;
import androidx.annotation.Keep;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.util.Pair;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.ReentrantLock;

@Keep
class DownloadRequestMediator {

    public final String url;
    public final String filePath;
    public final String metaPath;
    public final boolean isCacheable;
    public final String key;

    private AtomicInteger statusAtomic = new AtomicInteger(Status.NEW);
    private Map<String, Pair<DownloadRequest, AssetDownloadListener>> children = new ConcurrentHashMap<>(1);
    private final ReentrantLock lock = new ReentrantLock();

    private AtomicBoolean connectedAtomic = new AtomicBoolean(true);
    private AtomicReference<Runnable> runnable = new AtomicReference<>();

    DownloadRequestMediator(@NonNull DownloadRequest downloadRequest,
                            @Nullable AssetDownloadListener downloadListener,
                            @NonNull String filePath,
                            @NonNull String metaPath,
                            boolean isCacheable,
                            String key) {
        this.url = downloadRequest.url;
        this.filePath = filePath;
        this.metaPath = metaPath;
        this.isCacheable = isCacheable;
        this.key = key;
        children.put(downloadRequest.id, new Pair<>(downloadRequest, downloadListener));
    }

    synchronized void add(DownloadRequest downloadRequest, AssetDownloadListener downloadListener) {
        children.put(downloadRequest.id, new Pair<>(downloadRequest, downloadListener));
    }

    synchronized Pair<DownloadRequest, AssetDownloadListener> remove(DownloadRequest downloadRequest) {
        return children.remove(downloadRequest.id);
    }

    synchronized List<Pair<DownloadRequest, AssetDownloadListener>> values() {
        return new ArrayList<>(children.values());
    }

    synchronized List<DownloadRequest> requests() {
        List<Pair<DownloadRequest, AssetDownloadListener>> pairs = values();
        List<DownloadRequest> requests = new ArrayList<>();
        for (Pair<DownloadRequest, AssetDownloadListener> pair : pairs) {
            requests.add(pair.first);
        }

        return requests;
    }

    void lock() {
        lock.lock();
    }

    void unlock() {
        lock.unlock();
    }

    int getPriority() {
        int priority = DownloadRequest.Priority.LOWEST;
        for (Pair<DownloadRequest, AssetDownloadListener> pair : values()) {
            if (pair.first == null)
                continue;

            priority = Math.min(priority, pair.first.getPriority());
        }

        return priority;
    }

    boolean isPausable() {

        for (Pair<DownloadRequest, AssetDownloadListener> pair : values()) {
            if (pair.first == null)
                continue;

            if (pair.first.pauseOnConnectionLost)
                return true;
        }

        return false;
    }


    public @Status
    int getStatus() {
        return statusAtomic.get();
    }

    boolean is(@Status int status) {
        return this.statusAtomic.get() == status;
    }

    void set(@Status int status) {
        if (statusAtomic.get() == Status.CANCELLED)
            return;

        this.statusAtomic.set(status);
    }

    boolean isConnected() {
        return connectedAtomic.get();
    }

    void setConnected(boolean value) {
        connectedAtomic.set(value);
    }

    public void setRunnable(Runnable runnable) {
        this.runnable.set(runnable);

    }

    public Runnable getRunnable() {
        return runnable.get();
    }

    @IntDef(flag = true, value = {Status.NEW, Status.IN_PROGRESS,
            Status.PAUSED, Status.CANCELLED, Status.DONE, Status.ERROR,
            Status.PUBLISHED})
    @Keep
    @interface Status {
        int NEW = 0;
        int IN_PROGRESS = 1;
        int PAUSED = 2;
        int CANCELLED = 3;
        int DONE = 4;
        int ERROR = 5;
        int PUBLISHED = 6;
    }
}
