package com.vungle.warren.analytics;

import android.text.TextUtils;
import android.util.Log;
import android.util.Pair;
import android.widget.VideoView;

import com.moat.analytics.mobile.vng.MoatAdEvent;
import com.moat.analytics.mobile.vng.MoatAdEventType;
import com.moat.analytics.mobile.vng.ReactiveVideoTracker;
import com.vungle.warren.model.Advertisement;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.Map;
import java.util.Queue;

import androidx.annotation.NonNull;
import androidx.annotation.VisibleForTesting;

public class MoatTracker implements AnalyticsVideoTracker {

    private static final String TAG = MoatTracker.class.getSimpleName();
    public static final String VUNGLE_ID = "vunglenativevideo893259554489";
    private static final String UNKNOWN = "unknown";

    private VideoView videoView;
    private boolean isMoatEnabled;
    private boolean configured;
    private ReactiveVideoTracker tracker;
    private Map<String, String> adIds = new HashMap<>();

    /**
     * Queue to store View Percentages MoatTracking
     * Integer represents Percentage 0-100
     */
    private Queue<Pair<Integer, MoatAdEventType>> moatQuartileTrackers;

    private MoatTracker(@NonNull VideoView videoView, boolean isMoatEnabled) {
        this.isMoatEnabled = isMoatEnabled;
        this.videoView = videoView;
    }

    public void configure(@NonNull String placementId, @NonNull Advertisement advertisement, @NonNull String appId, @NonNull ReactiveVideoTracker tracker) {
        this.tracker = tracker;
        this.isMoatEnabled = isMoatEnabled && !TextUtils.isEmpty(placementId) && advertisement != null && advertisement.getMoatEnabled();

        //Initialize moat for this advertisement if config.moat is true and ad.moat is true
        if (isMoatEnabled) {

            //Create Tuple Queue to store MoatAdEvents <Percent(0-100), MoatAdEventType>
            //Works only once per creation of activity. If we want to play multiple ads on our
            //videoview, this will need to change for quartile tracking

            moatQuartileTrackers = new LinkedList<>();

            moatQuartileTrackers.add(new Pair<>(0, MoatAdEventType.AD_EVT_START));
            moatQuartileTrackers.add(new Pair<>(25, MoatAdEventType.AD_EVT_FIRST_QUARTILE));
            moatQuartileTrackers.add(new Pair<>(50, MoatAdEventType.AD_EVT_MID_POINT));
            moatQuartileTrackers.add(new Pair<>(75, MoatAdEventType.AD_EVT_THIRD_QUARTILE));

            if (!advertisement.getMoatVastExtra().isEmpty()) {
                adIds.put("zMoatVASTIDs", advertisement.getMoatVastExtra());
            }

            adIds.put("level1", advertisement.getAdvertiserAppId());
            adIds.put("level2", advertisement.getCampaignId());
            adIds.put("level3", advertisement.getCreativeId());
            adIds.put("level4", TextUtils.isEmpty(placementId) ? UNKNOWN : placementId);

            if (!TextUtils.isEmpty(appId)) {
                adIds.put("slicer1", appId);
            }

            configured = true;
        }

        configured = configured && isMoatEnabled;
    }

    public static MoatTracker connect(@NonNull VideoView videoView, boolean isEnabled) {
        return new MoatTracker(videoView, isEnabled);
    }

    @Override
    public void onProgress(int progress) {
        if (configured) {
            if (progress >= 100) {
                //have to send current video position in millis to moat or else view is not counted
                tracker.dispatchEvent(new MoatAdEvent(MoatAdEventType.AD_EVT_COMPLETE, videoView.getCurrentPosition()));
                tracker.stopTracking();
            } else {
                //Moat Quartile Tracking. Queue Polls for every completed quartile
                if (!moatQuartileTrackers.isEmpty() && progress >= moatQuartileTrackers.peek().first) {
                    MoatAdEvent event = new MoatAdEvent(moatQuartileTrackers.poll().second, progress);
                    tracker.dispatchEvent(event);
                }
            }
        }
    }

    @Override
    public void start(int videoDuration) {
        if (configured) {
            Log.d(TAG, "start");
            tracker.trackVideoAd(adIds, videoDuration, videoView);
        }
    }

    @Override
    public void stop() {
        if (configured) {
            /// If user did not finish video, send moat analytics.
            /// This must be done on the same thread as the WebView was created in.
            int currentVideoPosition = (videoView != null) ? videoView.getCurrentPosition() : 0;
            Log.d(TAG, "stopViewabilityTracker: " + currentVideoPosition);
            tracker.dispatchEvent(new MoatAdEvent(MoatAdEventType.AD_EVT_STOPPED, currentVideoPosition));
            tracker.stopTracking();
            Log.d(TAG, "stopViewabilityTracker: Success !!");
        }
    }

    @Override
    public void setPlayerVolume(boolean muted) {
        if (configured) {
            Log.d(TAG, "setPlayerVolume muted: " + muted);
            if(muted) {
                tracker.setPlayerVolume(MoatAdEvent.VOLUME_MUTED);
            } else {
                tracker.setPlayerVolume(MoatAdEvent.VOLUME_UNMUTED);
            }
        }
    }

    @VisibleForTesting
    Map<String, String> getAdIds() {
        return adIds;
    }
}
