package com.vungle.warren;

import android.content.Context;

/**
 * Vungle's setting for
 * {@link Vungle#init(String, Context, InitCallback, VungleSettings)}
 */
public final class VungleSettings {

    private static final long MEGABYTE = 1024L * 1024L;

    private final long minimumSpaceForInit;
    private final long minimumSpaceForAd;
    private final boolean androidIdOptedOut;
    private final long maximumStorageForCleverCache;
    private final boolean disableRefresh;

    /**
     * Builder for {@link VungleSettings}
     */
    public static class Builder {
        private long minimumSpaceForInit = 51 * MEGABYTE;
        private long minimumSpaceForAd = 50 * MEGABYTE;
        private boolean androidIdOptedOut;
        private long maximumStorageForCleverCache = 100 * MEGABYTE;
        private boolean disableRefresh;

        public VungleSettings build() {
            return new VungleSettings(this);
        }

        /**
         * Sets minimum amount of free storage required for initialization.
         * Default value is 50 megabytes.
         *
         * @param minimumSpaceForInit minimum storage in bytes.
         * @return {@link VungleSettings.Builder}
         */
        public Builder setMinimumSpaceForInit(long minimumSpaceForInit) {
            this.minimumSpaceForInit = minimumSpaceForInit;
            return this;
        }

        /**
         * Sets minimum amount of free storage required for ad loading.
         * Default value is 51 megabytes.
         *
         * @param minimumSpaceForAd minimum storage in bytes.
         * @return {@link VungleSettings.Builder}
         */
        public Builder setMinimumSpaceForAd(long minimumSpaceForAd) {
            this.minimumSpaceForAd = minimumSpaceForAd;
            return this;
        }

        /**
         * Enables/disables automatic fallback collection of Android ID in case ifa is not available.
         * Default value is false.
         *
         * @param disabled if true disables the collection.
         * @return {@link VungleSettings.Builder}
         */
        public Builder setAndroidIdOptOut(boolean disabled) {
            this.androidIdOptedOut = disabled;
            return this;
        }

        public Builder disableBannerRefresh() {
            this.disableRefresh = true;
            return this;
        }

        /**
         * Sets maximum amount of storage that clever cache can take.
         * Default value is 100 megabytes.
         *
         * @param maximumStorageForCleverCache minimum storage in bytes.
         * @return {@link VungleSettings.Builder}
         */
        public Builder setMaximumStorageForCleverCache(long maximumStorageForCleverCache) {
            this.maximumStorageForCleverCache = maximumStorageForCleverCache;
            return this;
        }
    }

    private VungleSettings(Builder builder) {
        this.minimumSpaceForAd = builder.minimumSpaceForAd;
        this.minimumSpaceForInit = builder.minimumSpaceForInit;
        this.androidIdOptedOut = builder.androidIdOptedOut;
        this.disableRefresh = builder.disableRefresh;
        this.maximumStorageForCleverCache = builder.maximumStorageForCleverCache;
    }

    public long getMinimumSpaceForAd() {
        return minimumSpaceForAd;
    }

    public long getMinimumSpaceForInit() {
        return minimumSpaceForInit;
    }

    public boolean getAndroidIdOptOut() {
        return androidIdOptedOut;
    }

    public boolean getBannerRefreshDisabled() {
        return disableRefresh;
    }

    public long getMaximumStorageForCleverCache() {
        return maximumStorageForCleverCache;
    }
}
