package com.vungle.warren;

import android.Manifest;
import android.annotation.SuppressLint;
import android.app.Application;
import android.app.UiModeManager;
import android.content.ContentResolver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.pm.PackageManager;
import android.content.res.Configuration;
import android.media.AudioManager;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.os.BatteryManager;
import android.os.Build;
import android.os.Environment;
import android.os.Handler;
import android.os.Looper;
import android.os.PowerManager;
import android.provider.Settings;
import android.security.NetworkSecurityPolicy;
import android.telephony.TelephonyManager;
import android.text.TextUtils;
import android.util.DisplayMetrics;
import android.util.Log;
import android.view.WindowManager;
import android.webkit.URLUtil;
import android.webkit.WebSettings;

import com.google.android.gms.ads.identifier.AdvertisingIdClient;
import com.google.android.gms.common.ConnectionResult;
import com.google.android.gms.common.GoogleApiAvailabilityLight;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.moat.analytics.mobile.vng.MoatAnalytics;
import com.moat.analytics.mobile.vng.MoatOptions;
import com.vungle.warren.error.VungleException;
import com.vungle.warren.model.Cookie;
import com.vungle.warren.model.JsonUtil;
import com.vungle.warren.network.APIFactory;
import com.vungle.warren.network.Call;
import com.vungle.warren.network.Response;
import com.vungle.warren.network.VungleApi;
import com.vungle.warren.persistence.CacheManager;
import com.vungle.warren.persistence.DatabaseHelper;
import com.vungle.warren.persistence.Repository;
import com.vungle.warren.utility.ViewUtility;

import java.io.File;
import java.io.IOException;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.TimeZone;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import androidx.annotation.Keep;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;
import androidx.annotation.StringDef;
import androidx.annotation.VisibleForTesting;
import androidx.core.content.PermissionChecker;
import okhttp3.HttpUrl;
import okhttp3.Interceptor;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.Protocol;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.ResponseBody;
import okio.Buffer;
import okio.BufferedSink;
import okio.GzipSink;
import okio.Okio;

import static com.vungle.warren.error.VungleException.CONFIGURATION_ERROR;
import static com.vungle.warren.model.Cookie.CCPA_COOKIE;
import static com.vungle.warren.model.Cookie.CONSENT_COOKIE;
import static com.vungle.warren.model.Cookie.USER_AGENT_ID_COOKIE;

/**
 * The HTTP Client for communicating with the Ad Server. All HTTP Requests should be routed through
 * this class.
 */
public class VungleApiClient {


    private static final String TAG = VungleApiClient.class.getCanonicalName();

    /**
     * Application context. Weak reference because we don't want to cause memory leaks if the
     * application is exiting.
     */
    private Context context;

    /**
     * Retrofit API client. The HTTP requests go through here ultimately.
     */
    private VungleApi api;

    /**
     * The paths of the endpoints that are defined by the response of the /config call.
     */
    private String newEndpoint, requestAdEndpoint, reportAdEndpoint, willPlayAdEndpoint, riEndpoint;

    /**
     * Value of the "device" field that is sent on every POST request to the ad server.
     */
    private JsonObject deviceBody;

    /**
     * Value of the "app" field that is sent on every POST request to the server.
     */
    private JsonObject appBody;

    /**
     * If true, the ad server would like a chance to replace the ad that is about to be played and
     * /willPlayAd should be called during the playAd() flow. If false, this part of the flow is
     * skipped.
     */
    private boolean willPlayAdEnabled;

    /**
     * The number of milliseconds to set as the timeout for the /willPlayAd response.
     */
    private int willPlayAdTimeout;

    /**
     * The HTTP Client that makes the requests. We need to keep a reference to this in order to
     * create per-request clones. The per-request clones share the connection pool, dispatcher, and
     * configuration of the original client.
     */
    private OkHttpClient client;

    /**
     * Retrofit API Client that enforces a strict timeout on its requests. This is primarily used
     * to manage the willPlayAd timeout logic. This field is only populated if willPlayAdEnabled
     * true.
     * It is otherwise <code>null</code>.
     */
    private VungleApi timeoutApi;

    /**
     * Retrofit API client with GZIP encoding of request body.
     */
    private VungleApi gzipApi;

    /**
     * Config if we should globally initialize MOAT on this device to track
     * viewability. If this is true we initialize MoatAnalytics.
     * enableMoat should override moat requests on ad responses.
     */
    private boolean enableMoat;

    /**
     * Designer that manages the cache directory. The API requires that we send the number of
     * available bytes in every request, which the designer knows.
     */
    private CacheManager cacheManager;

    static final String MANUFACTURER_AMAZON = "Amazon";

    public static String HEADER_UA = MANUFACTURER_AMAZON.equals(Build.MANUFACTURER) ?
            "VungleAmazon/" + BuildConfig.VERSION_NAME : "VungleDroid/" + BuildConfig.VERSION_NAME;

    private static String BASE_URL = "https://ads.api.vungle.com/";

    protected static WrapperFramework WRAPPER_FRAMEWORK_SELECTED;

    /**
     * Value of the "user" field that is send on every POST request to the server.
     */
    private JsonObject userBody;

    private Map<String, Long> retryAfterDataMap = new ConcurrentHashMap<>();

    /**
     * Andorid ID Collection Disabled as Fallback
     */
    private boolean defaultIdFallbackDisabled;

    private Repository repository;

    /**
     * String to keep track of User-Agent String
     */
    private String uaString = System.getProperty("http.agent");

    private final boolean okHttpSupported;

    /**
     * Private no-arg constructor ensure that we always have an api client.
     */
    VungleApiClient(@NonNull final Context context, @NonNull CacheManager cacheManager, @NonNull Repository repository) {
        this.cacheManager = cacheManager;
        this.context = context.getApplicationContext();
        this.repository = repository;

        /// Response Interceptor for Retry-After header value
        Interceptor responseInterceptor = new Interceptor() {

            @Override
            public okhttp3.Response intercept(Chain chain) throws IOException {
                Request request = chain.request();
                okhttp3.Response response;

                String urlPath = request.url().encodedPath();

                Long retryExpireTime = retryAfterDataMap.get(urlPath);
                if (retryExpireTime != null) {
                    long currentTimeStamp = System.currentTimeMillis();
                    long newRetryAfter = TimeUnit.MILLISECONDS.toSeconds(retryExpireTime - currentTimeStamp);
                    if (newRetryAfter > 0) {
                        return new okhttp3.Response.Builder()
                                .request(request)
                                .addHeader("Retry-After", String.valueOf(newRetryAfter))
                                .code(500)
                                .protocol(Protocol.HTTP_1_1)
                                .message("Server is busy")
                                .body(ResponseBody.create(MediaType.parse("application/json; charset=utf-8"), "{\"Error\":\"Retry-After\"}"))
                                .build();
                    } else {
                        retryAfterDataMap.remove(urlPath);            //Retry - After time passed, Remove url from MAP, allow request
                    }
                }

                response = chain.proceed(request);
                if (response != null) {
                    int responseCode = response.code();
                    if (responseCode == 429 || responseCode == 500 || responseCode == 502 || responseCode == 503) {                            //Checking for error code as per API docs
                        String retryAfterTimeStr = response.headers().get("Retry-After");
                        if (!TextUtils.isEmpty(retryAfterTimeStr)) {
                            try {
                                long retryAfterTimeValue = Long.parseLong(retryAfterTimeStr);
                                if (retryAfterTimeValue > 0) {
                                    retryAfterDataMap.put(urlPath, (retryAfterTimeValue * 1000) + System.currentTimeMillis());
                                }
                            } catch (NumberFormatException e) {
                                Log.d(TAG, "Retry-After value is not an valid value");
                            }
                        }
                    }
                }
                return response;
            }
        };

        /// Create the OkHttp Client
        OkHttpClient.Builder builder = new OkHttpClient.Builder()
                .addInterceptor(responseInterceptor);

        if (BuildConfig.DEBUG) {
            for (Interceptor interceptor : logInterceptors) {
                builder.addInterceptor(interceptor);                //SDK File logger works with addInterceptor()
            }

            for (Interceptor interceptor : networkInterceptors) {
                builder.addNetworkInterceptor(interceptor);         //Stetho requires addNetworkInterceptor()
            }
        }

        try {
            client = builder.build();
        } catch (ExceptionInInitializerError | NoClassDefFoundError e) {
            Log.e(TAG, "Can't init OKHttp", e);
            okHttpSupported = false;
            return;
        }
        okHttpSupported = true;

        OkHttpClient gzipClient = builder.addInterceptor(new GzipRequestInterceptor()).build();

        api = new APIFactory(client, BASE_URL).createAPI();
        gzipApi = new APIFactory(gzipClient, BASE_URL).createAPI();
    }

    public boolean platformIsNotSupported() {
        return !okHttpSupported;
    }

    public void init(String appId) {
        init(context, appId);
    }

    static class GzipRequestInterceptor implements Interceptor {
        private static final String CONTENT_ENCODING = "Content-Encoding";
        private static final String GZIP = "gzip";

        @NonNull
        @Override
        public okhttp3.Response intercept(@NonNull Chain chain) throws IOException {
            okhttp3.Request originalRequest = chain.request();
            if (originalRequest.body() == null
                    || originalRequest.header(CONTENT_ENCODING) != null) {
                return chain.proceed(originalRequest);
            }

            okhttp3.Request compressedRequest = originalRequest.newBuilder()
                    .header(CONTENT_ENCODING, GZIP)
                    .method(originalRequest.method(), gzip(originalRequest.body()))
                    .build();
            return chain.proceed(compressedRequest);
        }

        private RequestBody gzip(final RequestBody requestBody) throws IOException {
            final Buffer output = new Buffer();
            BufferedSink gzipSink = Okio.buffer(new GzipSink(output));
            requestBody.writeTo(gzipSink);
            gzipSink.close();
            return new RequestBody() {
                @Override
                public MediaType contentType() {
                    return requestBody.contentType();
                }

                @Override
                public long contentLength() {
                    return output.size();
                }

                @Override
                public void writeTo(@NonNull BufferedSink sink) throws IOException {
                    sink.write(output.snapshot());
                }
            };
        }
    }

    /**
     * Method to disable automatic fallback collection of Android ID in case ifa is not available.
     */
    public void setDefaultIdFallbackDisabled(boolean disabled) {
        defaultIdFallbackDisabled = disabled;
    }

    /**
     * Initializes the Vungle API Client and creates the request body parts.
     *
     * @param context Application context
     * @param appID   The publisher application ID, as shown in the dashboard.
     */
    private synchronized void init(final Context context, String appID) {

        /// Create the device and app objects, they don't change through the lifecycle of the SDK App
        JsonObject app = new JsonObject();
        app.addProperty("id", appID);
        app.addProperty("bundle", context.getPackageName());
        String versionName = null;
        try {
            versionName = context.getPackageManager().getPackageInfo(context.getPackageName(), 0).versionName;
        } catch (PackageManager.NameNotFoundException e) {
            /// Unable to retrieve the application version, will default to 1.0
        }
        app.addProperty("ver", versionName != null ? versionName : "1.0");

        /// Device
        JsonObject device = new JsonObject();
        device.addProperty("make", Build.MANUFACTURER);
        device.addProperty("model", Build.MODEL);
        device.addProperty("osv", Build.VERSION.RELEASE);
        device.addProperty("carrier", ((TelephonyManager) context.getSystemService(Context.TELEPHONY_SERVICE)).getNetworkOperatorName());
        device.addProperty("os", MANUFACTURER_AMAZON.equals(Build.MANUFACTURER) ? "amazon" : "android");
        DisplayMetrics dm = new DisplayMetrics();
        WindowManager windowManager = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
        windowManager.getDefaultDisplay().getMetrics(dm);
        device.addProperty("w", dm.widthPixels);
        device.addProperty("h", dm.heightPixels);

        /// Device Extension
        JsonObject ext = new JsonObject();
        JsonObject vungle = new JsonObject();
        ext.add("vungle", vungle);
        device.add("ext", ext);

        /// User-Agent. Before API 17, this code can only run on the UI thread. It is the only part
        /// of initialization that has this requirement, so we implement this not-elegant workaround
        /// to be defensive in this case.
        try {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {

                //Lets first check in Cookie
                uaString = getUserAgentFromCookie();

                //might it has been updated, re-init UA in background. Since it should not block Vungle Executor or to avoid dead-lock
                initUserAgentLazy();

            } else if (Looper.getMainLooper() == Looper.myLooper()) {       /// We are running on the main thread, we can do it simply

                uaString = ViewUtility.getWebView(context.getApplicationContext()).getSettings().getUserAgentString();

            } else {                                                        /// We are not running on the main thread, so we call out to the main thread

                final CountDownLatch latch = new CountDownLatch(1);         /// Use CountDownLatch to block until we have a user-agent
                Handler handler = new Handler(Looper.getMainLooper());
                handler.post(new Runnable() {
                    @Override
                    public void run() {

                        try {
                            uaString = ViewUtility.getWebView(context.getApplicationContext()).getSettings().getUserAgentString();
                        } catch (InstantiationException ex) {
                            Log.e(TAG, "Cannot Get UserAgent. Setting Default Device UserAgent." + ex.getLocalizedMessage());
                        }

                        latch.countDown();
                    }
                });

                /// If no uaString available in time, populate with Dalvik VM user agent  Dalvik/2.1.0 (Linux; U; Android 9; Pixel 2 Build/PPR1.180610.009)
                /// however this is not the preferred user agent
                if (!latch.await(2, TimeUnit.SECONDS)) {// Wait for 2 seconds to get user agent
                    Log.e(TAG, "Unable to get User Agent String in specified time");
                }
            }
        } catch (Exception ex) {        /// Adding Generic Exception to avoid any WebView related crash
            Log.e(TAG, "Cannot Get UserAgent. Setting Default Device UserAgent." + ex.getLocalizedMessage());
        }

        device.addProperty("ua", uaString);
        deviceBody = device;

        /// Assign the values to the singleton instance.
        appBody = app;
    }

    @RequiresApi(api = Build.VERSION_CODES.JELLY_BEAN_MR1)
    private void initUserAgentLazy() {
        //Not possible being in main thread here, since in VungleExecutor
        new Thread(new Runnable() {

            @Override
            public void run() {
                try {
                    uaString = WebSettings.getDefaultUserAgent(context);
                    deviceBody.addProperty("ua", uaString);
                    addUserAgentInCookie(uaString);
                } catch (Exception ex) {/// Adding Generic Exception to avoid any WebView related crash
                    Log.e(TAG, "Cannot Get UserAgent. Setting Default Device UserAgent." + ex.getLocalizedMessage());
                }
            }
        }).start();
    }

    /**
     * Requests a configuration from the server. This will set the relevant fields within the API
     * Client for the dynamic endpoints, then callback.
     */
    public Response config() throws VungleException, IOException {

        JsonObject body = new JsonObject();
        body.add("device", getDeviceBody());
        body.add("app", appBody);
        body.add("user", getUserBody());

        Response<JsonObject> response = api.config(HEADER_UA, body).execute();

        if (!response.isSuccessful()) {
            /// Immediately propagate the response up to the caller. We cannot extract
            /// any useful information here, but perhaps some business logic can.
            return response;
        }

        JsonObject jsonObject = response.body();
        Log.d(TAG, "Config Response: " + jsonObject);
        if (JsonUtil.hasNonNull(jsonObject, "sleep")) {
            String errorMessage = JsonUtil.hasNonNull(jsonObject, "info") ? jsonObject.get("info").getAsString() : "";
            Log.e(TAG, "Error Initializing Vungle. Please try again. " + errorMessage);
            throw new VungleException(CONFIGURATION_ERROR);
        }

        //endpoints call be null and crash out application
        if (!JsonUtil.hasNonNull(jsonObject, "endpoints")) {
            Log.e(TAG, "Error Initializing Vungle. Please try again. ");
            throw new VungleException(CONFIGURATION_ERROR);
        }

        /// Parse out the endpoints
        JsonObject endpoints = jsonObject.getAsJsonObject("endpoints");

        HttpUrl newUrl = HttpUrl.parse(endpoints.get("new").getAsString());
        HttpUrl adsUrl = HttpUrl.parse(endpoints.get("ads").getAsString());
        HttpUrl willPlayAdUrl = HttpUrl.parse(endpoints.get("will_play_ad").getAsString());
        HttpUrl reportAdUrl = HttpUrl.parse(endpoints.get("report_ad").getAsString());
        HttpUrl reportIncentivized = HttpUrl.parse(endpoints.get("ri").getAsString());

        if (newUrl == null || adsUrl == null || willPlayAdUrl == null || reportAdUrl == null || reportIncentivized == null) {
            Log.e(TAG, "Error Initializing Vungle. Please try again. ");
            throw new VungleException(CONFIGURATION_ERROR);
        }

        newEndpoint = newUrl.toString();
        requestAdEndpoint = adsUrl.toString();
        willPlayAdEndpoint = willPlayAdUrl.toString();
        reportAdEndpoint = reportAdUrl.toString();
        riEndpoint = reportIncentivized.toString();

        /// Parse out the configuration variables
        JsonObject willPlayAd = jsonObject.getAsJsonObject("will_play_ad");
        willPlayAdTimeout = willPlayAd.get("request_timeout").getAsInt();
        willPlayAdEnabled = willPlayAd.get("enabled").getAsBoolean();

        //parse out moat viewabiilty
        JsonObject moatViewability = jsonObject.getAsJsonObject("viewability");
        enableMoat = moatViewability.get("moat").getAsBoolean();


        if (willPlayAdEnabled) {
            /// Make a clone of the original client in order to override the timeout value only for
            /// this endpoint. It is not currently possible to override this value per
            /// request or per call. Followed best practices according to OkHTTP:
            /// https://github.com/square/okhttp/wiki/Recipes#per-call-configuration
            Log.v(TAG, "willPlayAd is enabled, generating a timeout client.");
            OkHttpClient timeoutClient = client.newBuilder()
                    .readTimeout(willPlayAdTimeout, TimeUnit.MILLISECONDS)
                    .build();
            APIFactory timeoutRetro = new APIFactory(timeoutClient,
                    "https://api.vungle.com/");
            timeoutApi = timeoutRetro.createAPI();
        }

        //init Moat with logging
        if (getMoatEnabled()) {
            MoatOptions options = new MoatOptions();
            options.disableAdIdCollection = true;
            options.disableLocationServices = true;
            options.loggingEnabled = true;

            MoatAnalytics.getInstance().start(options, (Application) context.getApplicationContext());
        }
        return response;
    }

    /**
     * Only called the first time the SDK runs per application, informs the server that there is a new
     * installation.
     *
     * @return {@link Call} which will pass back the response body to the caller.
     * @throws IllegalStateException if this method is called before the Api Client has been initialized.
     */
    public Call<JsonObject> reportNew() throws IllegalStateException {
        if (newEndpoint == null) {
            throw new IllegalStateException("API Client not configured yet! Must call /config first.");
        }
        HashMap<String, String> query = new HashMap<>(2);
        JsonElement idElement = appBody.get("id");
        JsonElement ifaElement = deviceBody.get("ifa");
        query.put("app_id", idElement != null ? idElement.getAsString() : "");
        query.put("ifa", ifaElement != null ? ifaElement.getAsString() : "");
        return api.reportNew(HEADER_UA, newEndpoint, query);
    }

    /**
     * Request the ad information for a particular placement.
     *
     * @param placement             The identifier for the placement whose assets are being requested.
     * @param adSize                Size of Ad
     * @param isHeaderBiddingEnable Status of header bidding feature.
     * @param vision                Vision payload
     * @return {@link Call} which will pass back the response body to the caller.
     * @throws IllegalStateException If called before the API Client has been initialized
     */
    public Call<JsonObject> requestAd(String placement, String adSize, boolean isHeaderBiddingEnable, @Nullable JsonObject vision) throws IllegalStateException {
        if (requestAdEndpoint == null) {
            throw new IllegalStateException("API Client not configured yet! Must call /config first.");
        }

        JsonObject body = new JsonObject();
        body.add("device", getDeviceBody());
        body.add("app", appBody);
        JsonObject userBody = getUserBody();
        if (vision != null) {
            userBody.add(VisionController.VISION, vision);
        }
        body.add("user", userBody);

        /// Create the request body
        JsonObject request = new JsonObject();
        JsonArray placementsArray = new JsonArray();
        placementsArray.add(placement);
        request.add("placements", placementsArray);
        request.addProperty("header_bidding", isHeaderBiddingEnable);

        if (!TextUtils.isEmpty(adSize)) {
            request.addProperty("ad_size", adSize);
        }

        body.add("request", request);

        /// Hack to work around the server not giving me ads.
        return gzipApi.ads(HEADER_UA, requestAdEndpoint, body);
    }

    /**
     * Inform the server that we are about to play an ad, gives the back-end a chance to change the ad
     * that will be played if there is a better one available.
     *
     * @param adToken     The token identifying the advertisement bundle that is about to be played
     * @param autoCached  Whether or not this placement is auto-cached
     * @param placementID The placement identifier.
     * @return {@link Call} which will pass back the response body to the caller.
     */
    Call<JsonObject> willPlayAd(String placementID, boolean autoCached, String adToken) {
        JsonObject body = new JsonObject();
        body.add("device", getDeviceBody());
        body.add("app", appBody);
        body.add("user", getUserBody());

        /// Create the request body
        JsonObject request = new JsonObject();
        JsonObject placement = new JsonObject();
        placement.addProperty("reference_id", placementID);
        placement.addProperty("is_auto_cached", autoCached);

        request.add("placement", placement);
        request.addProperty("ad_token", adToken);

        body.add("request", request);

        return timeoutApi.willPlayAd(HEADER_UA, willPlayAdEndpoint, body);
    }

    boolean canCallWillPlayAd() {
        return willPlayAdEnabled && !TextUtils.isEmpty(willPlayAdEndpoint);
    }

    /**
     * Report that an ad has been played to the server.
     *
     * @param request The request body
     * @return A {@link Call} object which can be used to execute or enqueue the request. It is expected
     * that the caller will handle the error scenarios.
     */
    public Call<JsonObject> reportAd(JsonObject request) {
        if (reportAdEndpoint == null) {
            throw new IllegalStateException("API Client not configured yet! Must call /config first.");
        }
        JsonObject body = new JsonObject();
        body.add("device", getDeviceBody());
        body.add("app", appBody);
        body.add("request", request);
        body.add("user", getUserBody());

        return gzipApi.reportAd(HEADER_UA, reportAdEndpoint, body);
    }

    /**
     * "request": {
     * "placement_reference_id": "string", // reserved for utilize placement information to reward callback
     *         "app_id": "string",     
     * "adStartTime": 0,            // Make sure its value is same as the one in /report_ad request payload     
     * "user": "string",     
     * "name": "string"
     * }
     *
     * @param request
     * @return
     */

    public Call<JsonObject> ri(JsonObject request) {
        if (riEndpoint == null) {
            throw new IllegalStateException("API Client not configured yet! Must call /config first.");
        }
        JsonObject body = new JsonObject();
        body.add("device", getDeviceBody());
        body.add("app", appBody);
        body.add("request", request);

        return api.ri(HEADER_UA, riEndpoint, body);
    }

    /**
     * Ping TPAT
     *
     * @param url - TPAT Url
     * @return false if request failed but can retry
     */
    public boolean pingTPAT(final String url) throws ClearTextTrafficException, MalformedURLException {
        if (!TextUtils.isEmpty(url) && HttpUrl.parse(url) != null) {    //Url is empty or invalid, No need to hit tpat
            /*
             * From Android M, developer can use clearTextTraffic flag to set the clearTextTraffic allow or not.
             * Below Android M, clearTextTraffic is always allowed and developer can not disable it.
             *
             * By default allow clearTextTrafficEnabled is true, below api Android P and upto Android M
             * By default allow clearTextTrafficEnabled is false, for api Android P and above
             *
             * So the below condition is used to check whether cleartext network traffic allowed or not for App
             * https://developer.android.com/reference/android/security/NetworkSecurityPolicy.html#isCleartextTrafficPermitted()
             */
            boolean clearTextTrafficPermitted;
            String host;                                                //Checking host
            try {
                host = (new URL(url)).getHost();
            } catch (MalformedURLException e) {
                throw new MalformedURLException("Invalid URL : " + url);
            }

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {       //Checking clearTextTrafficPermitted
                clearTextTrafficPermitted = NetworkSecurityPolicy.getInstance().isCleartextTrafficPermitted(host);  //Above Android N check with host
            } else if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                clearTextTrafficPermitted = NetworkSecurityPolicy.getInstance().isCleartextTrafficPermitted();      //Android M check normally
            } else {
                clearTextTrafficPermitted = true;                                                                   //Older always permitted
            }

            if (!clearTextTrafficPermitted && URLUtil.isHttpUrl(url)) {                                             //If not permitted and url is http throw exc
                throw new ClearTextTrafficException("Clear Text Traffic is blocked");
            }

            try {
               api.pingTPAT(uaString, url).execute();
            } catch (IOException e) {
                Log.d(TAG, "Error on pinging TPAT");
                return false;
            }
        } else {
            throw new MalformedURLException("Invalid URL : " + url);
        }
        return true;
    }

    /**
     * Method that generates the device body. The device body is not because it includes up-to-date
     * battery and network information. Example:
     * <pre><code>
     *     "android_id": "68f19b937a5ef8da",
     *     "battery_level": 1,
     *     "battery_saver_enabled": 0,
     *     "battery_state": "BATTERY_PLUGGED_USB",
     *     "connection_type": "WIFI",
     *     "connection_type_detail": "WIFI",
     *     "data_saver_status": "NOT_APPLICABLE",
     *     "gaid": "68f19b937a5ef8da",
     *     "language": "en",
     *     "locale": "en_US",
     *     "network_metered": 0,
     *     "sd_card_available": 1,
     *     "sound_enabled": 0,
     *     "time_zone": "America/Los_Angeles",
     *     "volume_level": 0
     * </code></pre>
     *
     * @return JsonObject that includes the up-to-date android device information.
     */
    @SuppressLint("HardwareIds")
    @SuppressWarnings("squid:S5322")
    private JsonObject getDeviceBody() throws IllegalStateException {
        JsonObject android = new JsonObject();

        /// Advertising Identifier
        String advertId = null;
        boolean limitAdTracking = true;

        try {
            AdvertisingIdClient.Info idInfo = null;
            if (MANUFACTURER_AMAZON.equals(Build.MANUFACTURER)) {
                try {
                    ContentResolver cr = context.getContentResolver();
                    // load user's tracking preference
                    limitAdTracking = (Settings.Secure.getInt(cr, "limit_ad_tracking") == 1);
                    // load advertising
                    advertId = Settings.Secure.getString(cr, "advertising_id");
                } catch (Settings.SettingNotFoundException ex) {
                    // not supported
                    Log.w(TAG, "Error getting Amazon advertising info", ex);
                }
            } else {
                try {
                    idInfo = AdvertisingIdClient.getAdvertisingIdInfo(context);
                    if (idInfo != null) {
                        advertId = idInfo.getId();
                        limitAdTracking = (idInfo.isLimitAdTrackingEnabled());
                        deviceBody.addProperty("ifa", advertId);
                    } else {
                        /// Advertising ID was not available through the Google Play Services. Panic!
                        //                    advertisingID = "PANIC!";
                    }
                } catch (NoClassDefFoundError ex) {
                    Log.e(TAG, "Play services Not available: " + ex.getLocalizedMessage());
                    ContentResolver cr = context.getContentResolver();
                    advertId = Settings.Secure.getString(cr, "advertising_id");
                }

            }
        } catch (Exception ex) {
            Log.e(TAG, "Cannot load Advertising ID");
        }

        if (advertId != null) {
            android.addProperty(MANUFACTURER_AMAZON.equals(Build.MANUFACTURER) ? "amazon_advertising_id" : "gaid", advertId);
            deviceBody.addProperty("ifa", advertId);
        } else {
            /// If the google advertising ID is not available, we fall back to the android_id
            String androidID = Settings.Secure.getString(context.getContentResolver(), Settings.Secure.ANDROID_ID);
            deviceBody.addProperty("ifa", defaultIdFallbackDisabled ? ""
                    : !TextUtils.isEmpty(androidID) ? androidID : "");
            if (!TextUtils.isEmpty(androidID) && !defaultIdFallbackDisabled) {
                android.addProperty("android_id", androidID);
            }

        }

        //this lmt value is legally required to be passed upward
        deviceBody.addProperty("lmt", limitAdTracking ? 1 : 0);

        android.addProperty("is_google_play_services_available", isGooglePlayServicesAvailable(context));

        /// Battery
        Intent batteryStatus = (context != null) ?
                context.registerReceiver(null, new IntentFilter(Intent.ACTION_BATTERY_CHANGED)) : null;
        String batteryState;
        if (batteryStatus != null) {
            int level = 0;
            level = batteryStatus.getIntExtra(BatteryManager.EXTRA_LEVEL, -1);
            int scale = batteryStatus.getIntExtra(BatteryManager.EXTRA_SCALE, -1);
            if (level > 0 && scale > 0) {
                android.addProperty("battery_level", level / (float) scale);
            }
            int status = batteryStatus.getIntExtra(BatteryManager.EXTRA_STATUS, -1);

            if (status == -1) {
                batteryState = "UNKNOWN";
            } else if (status == BatteryManager.BATTERY_STATUS_CHARGING || status == BatteryManager.BATTERY_STATUS_FULL) {
                switch (batteryStatus.getIntExtra(BatteryManager.EXTRA_PLUGGED, -1)) {
                    case BatteryManager.BATTERY_PLUGGED_USB:
                        batteryState = "BATTERY_PLUGGED_USB";
                        break;
                    case BatteryManager.BATTERY_PLUGGED_AC:
                        batteryState = "BATTERY_PLUGGED_AC";
                        break;
                    case BatteryManager.BATTERY_PLUGGED_WIRELESS:
                        batteryState = "BATTERY_PLUGGED_WIRELESS";
                        break;
                    default:
                        batteryState = "BATTERY_PLUGGED_OTHERS";
                }
            } else {
                batteryState = "NOT_CHARGING";
            }
        } else {
            batteryState = "UNKNOWN";
        }

        android.addProperty("battery_state", batteryState);

        /// Battery saver (only available from Lollipop onward)
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            PowerManager powerManager = (PowerManager) context.getSystemService(Context.POWER_SERVICE);
            android.addProperty("battery_saver_enabled", (powerManager != null && powerManager.isPowerSaveMode()) ? 1 : 0);
        }

        /// Network Connection
        if (PermissionChecker.checkCallingOrSelfPermission(context, Manifest.permission.ACCESS_NETWORK_STATE) == PermissionChecker.PERMISSION_GRANTED) {
            String connectionType = "NONE";
            String connectionTypeDetail = ConnectionTypeDetail.UNKNOWN;

            ConnectivityManager cm = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
            if (cm != null) {
                NetworkInfo info = cm.getActiveNetworkInfo();
                if (info != null) {
                    switch (info.getType()) {
                        case ConnectivityManager.TYPE_BLUETOOTH:
                            connectionType = "BLUETOOTH";
                            break;
                        case ConnectivityManager.TYPE_ETHERNET:
                            connectionType = "ETHERNET";
                            break;
                        case ConnectivityManager.TYPE_MOBILE:
                            connectionType = "MOBILE";
                            connectionTypeDetail = getConnectionTypeDetail(info.getSubtype());
                            break;
                        case ConnectivityManager.TYPE_WIFI:
                        case ConnectivityManager.TYPE_WIMAX:
                            connectionType = "WIFI";
                            break;
                        default:
                            connectionType = "UNKNOWN";
                    }
                }
            }

            android.addProperty("connection_type", connectionType);
            android.addProperty("connection_type_detail", connectionTypeDetail);

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                if (cm.isActiveNetworkMetered()) {
                    String dataSaverStatus;
                    switch (cm.getRestrictBackgroundStatus()) {
                        case ConnectivityManager.RESTRICT_BACKGROUND_STATUS_ENABLED:
                            // Background data usage is blocked for this app. If
                            // possible, sdk should use less data in the foreground.
                            dataSaverStatus = "ENABLED";
                            break;
                        case ConnectivityManager.RESTRICT_BACKGROUND_STATUS_WHITELISTED:
                            // Current app is whitelisted. If possible, sdk should use
                            // less data in the foreground and background.
                            dataSaverStatus = "WHITELISTED";
                            break;
                        case ConnectivityManager.RESTRICT_BACKGROUND_STATUS_DISABLED:
                            // Data Saver is disabled. Since the device is on a metered
                            // network, if possible, the SDK should use less data.
                            dataSaverStatus = "DISABLED";
                            break;
                        default:
                            dataSaverStatus = "UNKNOWN";
                            break;
                    }
                    android.addProperty("data_saver_status", dataSaverStatus);
                    android.addProperty("network_metered", 1);
                } else {
                    android.addProperty("data_saver_status", "NOT_APPLICABLE");
                    android.addProperty("network_metered", 0);
                }
            }
        }

        /// Language/Locale
        android.addProperty("locale", Locale.getDefault().toString());
        android.addProperty("language", Locale.getDefault().getLanguage()); /// ISO-639-1-alpha-2
        android.addProperty("time_zone", TimeZone.getDefault().getID());

        /// Audio Values
        AudioManager audio = (AudioManager) context.getSystemService(Context.AUDIO_SERVICE);
        int max, current;
        float vol;
        if (audio != null) {
            max = audio.getStreamMaxVolume(AudioManager.STREAM_MUSIC);
            current = audio.getStreamVolume(AudioManager.STREAM_MUSIC);
            vol = (float) current / (float) max;
            android.addProperty("volume_level", vol);
            android.addProperty("sound_enabled", current > 0 ? 1 : 0);
        }

        /// Storage Values
        final File cacheDirectory = cacheManager.getCache();
        final String cachePath = cacheDirectory.getPath();

        if (cacheDirectory.exists() && cacheDirectory.isDirectory()) {
            android.addProperty("storage_bytes_available", cacheManager.getBytesAvailable());
        }

        /// TV Values
        boolean isTV;
        if (MANUFACTURER_AMAZON.equals(Build.MANUFACTURER)) {
            final String AMAZON_FEATURE_FIRE_TV = "amazon.hardware.fire_tv";
            isTV = context.getApplicationContext().getPackageManager().hasSystemFeature(AMAZON_FEATURE_FIRE_TV);
        } else {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                // in later API versions, this is better check as some handheld devices could be connected to TV screen
                // and run in TV mode
                UiModeManager uiModeManager = (UiModeManager) context.getSystemService(Context.UI_MODE_SERVICE);
                isTV = (uiModeManager.getCurrentModeType() == Configuration.UI_MODE_TYPE_TELEVISION);
            } else {
                final String FEATURE_ANDROID_TV = "com.google.android.tv";
                final String FEATURE_HW_TOUCHSCREEN = "android.hardware.touchscreen";
                //has feature flag for Android TV OR Does Not have H/W Feature Touchscreen
                isTV = context.getApplicationContext().getPackageManager().hasSystemFeature(FEATURE_ANDROID_TV) ||
                        !context.getApplicationContext().getPackageManager().hasSystemFeature(FEATURE_HW_TOUCHSCREEN);
            }
        }
        android.addProperty("is_tv", isTV);
        android.addProperty("os_api_level", Build.VERSION.SDK_INT);

        /// Non Market Install Values
        boolean canInstallNonMarket = false;
        try {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                if (PackageManager.PERMISSION_GRANTED ==
                        context.checkCallingOrSelfPermission(Manifest.permission.REQUEST_INSTALL_PACKAGES)) {
                    canInstallNonMarket = context.getApplicationContext().getPackageManager().canRequestPackageInstalls();
                }
            } else {
                canInstallNonMarket = (Settings.Secure.getInt(context.getContentResolver(), Settings.Secure.INSTALL_NON_MARKET_APPS) == 1);
            }
        } catch (Settings.SettingNotFoundException e) {
            Log.e(TAG, "isInstallNonMarketAppsEnabled Settings not found", e);
        }
        android.addProperty("is_sideload_enabled", canInstallNonMarket);

        boolean isSDPresent = Environment.getExternalStorageState().equals(Environment.MEDIA_MOUNTED);
        android.addProperty("sd_card_available", isSDPresent ? 1 : 0);
        android.addProperty("os_name", Build.FINGERPRINT);


        android.addProperty("vduid", "");

        deviceBody.getAsJsonObject("ext").getAsJsonObject("vungle").add(MANUFACTURER_AMAZON.equals(Build.MANUFACTURER) ? "amazon" : "android", android);

        return deviceBody;
    }

    /**
     * Find and returns the sub type of mobile connection detail as per the of mobile network
     *
     * @return String sub type of mobile connection like GPRS, EVDO, LTE, etc.
     */
    private String getConnectionTypeDetail(int type) {
        switch (type) {
            case TelephonyManager.NETWORK_TYPE_1xRTT:
                return ConnectionTypeDetail.CDMA_1XRTT;
            case TelephonyManager.NETWORK_TYPE_CDMA:
                return ConnectionTypeDetail.WCDMA;
            case TelephonyManager.NETWORK_TYPE_EDGE:
                return ConnectionTypeDetail.EDGE;
            case TelephonyManager.NETWORK_TYPE_EHRPD:
                return ConnectionTypeDetail.HRPD;
            case TelephonyManager.NETWORK_TYPE_EVDO_0:
                return ConnectionTypeDetail.CDMA_EVDO_0;
            case TelephonyManager.NETWORK_TYPE_EVDO_A:
                return ConnectionTypeDetail.CDMA_EVDO_A;
            case TelephonyManager.NETWORK_TYPE_EVDO_B:
                return ConnectionTypeDetail.CDMA_EVDO_B;
            case TelephonyManager.NETWORK_TYPE_GPRS:
                return ConnectionTypeDetail.GPRS;
            case TelephonyManager.NETWORK_TYPE_HSDPA:
                return ConnectionTypeDetail.HSDPA;
            case TelephonyManager.NETWORK_TYPE_HSUPA:
                return ConnectionTypeDetail.HSUPA;
            case TelephonyManager.NETWORK_TYPE_LTE:
                return ConnectionTypeDetail.LTE;

            case TelephonyManager.NETWORK_TYPE_UNKNOWN:
            default:
                return ConnectionTypeDetail.UNKNOWN;
        }
    }

    /**
     * Generates and returns the user portion of the request body, which contains data-gathering
     * consent information and other user state.
     *
     * @return JsonObject which includes all the user information required to make the request.
     */
    private JsonObject getUserBody() {
        JsonObject userBody = new JsonObject();

        /// The consent status is saved on disk in a special cookie, retrieve it and extract the data.
        String status, source, messageVersion;
        long timestamp;

        Cookie consentCookie = repository.load(CONSENT_COOKIE, Cookie.class).get();

        //return status even if not gdpr
        if (consentCookie != null) {
            status = consentCookie.getString("consent_status");
            source = consentCookie.getString("consent_source");
            timestamp = consentCookie.getLong("timestamp");
            messageVersion = consentCookie.getString("consent_message_version");
        } else {
            /// If we have no consent status, default to unknown values. We use lack of data to infer
            /// this status.
            status = "unknown";
            source = "no_interaction";
            timestamp = 0L;
            //return empty string for non updated consent status/vungle default
            //returning a number will break jaeger
            messageVersion = "";
        }

        JsonObject gdpr = new JsonObject();

        /*
         * Status of consent regarding General Data Protection Regulation (GDPR). There are three
         * options for this field: ‘opted_in’ which means the user allows the use of personal data,
         * ‘opted_out’ which means the user denied the use of personal data, and ‘unknown’ which means
         * the user did not specify the use of personal data.
         */
        gdpr.addProperty("consent_status", status);

        /*
         * Source of consent regarding General Data Protection Regulation (GDPR). There are three
         * options for this field: ‘publisher’ which means the consent came from the publisher,
         * ‘vungle_modal’ which means the consent came from our vungle modal, and ‘no_interaction’
         * which means there was no interaction from the user to the vungle modal.
         */
        gdpr.addProperty("consent_source", source);

        /*
         * Timestamp in unix epoch seconds at the moment that consent was recorded.
         */
        gdpr.addProperty("consent_timestamp", timestamp);

        gdpr.addProperty("consent_message_version", TextUtils.isEmpty(messageVersion) ? "" : messageVersion);

        /// Include the GDPR state in the user model
        userBody.add("gdpr", gdpr);

        // The ccpa status is saved on disk in a special cookie, retrieve it and extract the data.
        String ccpaStatus;
        Cookie ccpaCookie = repository.load(CCPA_COOKIE, Cookie.class).get();
        //return status even if not ccpa
        if (ccpaCookie != null) {
            ccpaStatus = ccpaCookie.getString(Cookie.CCPA_CONSENT_STATUS);
        } else {
            // If we have no ccpa status, default to opted_in values. We use lack of data to infer this status.
            ccpaStatus = Cookie.CONSENT_STATUS_OPTED_IN;
        }

        JsonObject ccpa = new JsonObject();

        /*
         * Status of consent regarding California Consumer Privacy Act (CCPA). There are two
         * options for this field: ‘opted_in’ which means the user allows the use of personal data,
         * ‘opted_out’ which means the user denied the use of personal data.
         */
        ccpa.addProperty("status", ccpaStatus);

        // Include the CCPA state in the user model
        userBody.add("ccpa", ccpa);

        return userBody;
    }

    /**
     * If Moat is enabled. Hard to keep this clean without creating another class.
     * , marshalling requests through here or pushing all of this logic elsewhere. We don't have a
     * constant class. Considered a number of options, but this is the lowest impact.
     *
     * @return - if true, setup moat
     * TODO: Next cleanup
     */
    public boolean getMoatEnabled() {
        return enableMoat && Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN;
    }

    private String getUserAgentFromCookie() {
        String tempUserAgent;

        Cookie cookie = repository.load(USER_AGENT_ID_COOKIE, Cookie.class).get();
        if (cookie == null) {                            //Very first time or after clearing app data
            tempUserAgent = System.getProperty("http.agent");
        } else {
            tempUserAgent = cookie.getString("userAgent");
            if (TextUtils.isEmpty(tempUserAgent)) {        //Generally not possible but for safer side
                tempUserAgent = System.getProperty("http.agent");
            }
        }

        return tempUserAgent;
    }


    private void addUserAgentInCookie(String userAgent) throws DatabaseHelper.DBException {
        Cookie cookie = new Cookie(USER_AGENT_ID_COOKIE);
        cookie.putValue("userAgent", userAgent);

        repository.save(cookie);
    }

    public long getRetryAfterHeaderValue(Response response) {
        String header = response.headers().get("Retry-After");
        try {
            return Long.parseLong(header) * 1000;
        } catch (NumberFormatException ex) {
            return 0;
        }
    }

    /**
     * External framework wrapping this SDK.
     */
    @Keep
    public enum WrapperFramework {
        admob,
        air,
        cocos2dx,
        corona,
        dfp,
        heyzap,
        marmalade,
        mopub,
        unity,
        fyber,
        ironsource,
        upsight,
        appodeal,
        aerserv,
        adtoapp,
        tapdaq,
        vunglehbs,
        none
    }

    @StringDef(value = {
            ConnectionTypeDetail.UNKNOWN,
            ConnectionTypeDetail.CDMA_1XRTT,
            ConnectionTypeDetail.WCDMA,
            ConnectionTypeDetail.EDGE,
            ConnectionTypeDetail.HRPD,
            ConnectionTypeDetail.CDMA_EVDO_0,
            ConnectionTypeDetail.CDMA_EVDO_A,
            ConnectionTypeDetail.CDMA_EVDO_B,
            ConnectionTypeDetail.GPRS,
            ConnectionTypeDetail.HSDPA,
            ConnectionTypeDetail.HSUPA,
            ConnectionTypeDetail.LTE
    })
    @Retention(RetentionPolicy.SOURCE)
    public @interface ConnectionTypeDetail {
        String UNKNOWN = "unknown";
        String CDMA_1XRTT = "cdma_1xrtt";
        String WCDMA = "wcdma";
        String EDGE = "edge";
        String HRPD = "hrpd";
        String CDMA_EVDO_0 = "cdma_evdo_0";
        String CDMA_EVDO_A = "cdma_evdo_a";
        String CDMA_EVDO_B = "cdma_evdo_b";
        String GPRS = "gprs";
        String HSDPA = "hsdpa";
        String HSUPA = "hsupa";
        String LTE = "LTE";
    }

    private static Set<Interceptor> networkInterceptors = new HashSet<>();

    private static Set<Interceptor> logInterceptors = new HashSet<>();

    public static class ClearTextTrafficException extends IOException {

        ClearTextTrafficException(String message) {
            super(message);
        }
    }

    @VisibleForTesting
    void overrideApi(VungleApi api) {
        this.api = api;
    }

    public static boolean isGooglePlayServicesAvailable(Context context) {
        try {
            GoogleApiAvailabilityLight googleApiAvailabilityLight = GoogleApiAvailabilityLight.getInstance();
            if (googleApiAvailabilityLight != null) {
                return googleApiAvailabilityLight.isGooglePlayServicesAvailable(context) == ConnectionResult.SUCCESS;
            }
        } catch (NoClassDefFoundError error) {
            Log.w(TAG, "Play services Not available");
        }
        return false;
    }
}
