package com.vungle.warren;

import android.app.Application;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.os.Build;
import android.text.TextUtils;
import android.util.Base64;
import android.util.Log;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.vungle.warren.downloader.DownloadRequest;
import com.vungle.warren.downloader.Downloader;
import com.vungle.warren.error.VungleException;
import com.vungle.warren.model.Advertisement;
import com.vungle.warren.model.Cookie;
import com.vungle.warren.model.JsonUtil;
import com.vungle.warren.model.Placement;
import com.vungle.warren.network.Call;
import com.vungle.warren.network.Callback;
import com.vungle.warren.network.HttpException;
import com.vungle.warren.network.Response;
import com.vungle.warren.persistence.CacheManager;
import com.vungle.warren.persistence.DatabaseHelper;
import com.vungle.warren.persistence.FutureResult;
import com.vungle.warren.persistence.Repository;
import com.vungle.warren.tasks.CleanupJob;
import com.vungle.warren.tasks.JobRunner;
import com.vungle.warren.tasks.ReconfigJob;
import com.vungle.warren.tasks.SendReportsJob;
import com.vungle.warren.ui.VungleActivity;
import com.vungle.warren.ui.VungleFlexViewActivity;
import com.vungle.warren.ui.contract.AdContract.AdvertisementBus;
import com.vungle.warren.ui.view.VungleNativeView;
import com.vungle.warren.utility.ActivityManager;
import com.vungle.warren.utility.Executors;
import com.vungle.warren.utility.TimeoutProvider;
import com.vungle.warren.vision.VisionConfig;

import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;

import androidx.annotation.Keep;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import androidx.localbroadcastmanager.content.LocalBroadcastManager;

import static com.vungle.warren.error.VungleException.APPLICATION_CONTEXT_REQUIRED;
import static com.vungle.warren.error.VungleException.CONFIGURATION_ERROR;
import static com.vungle.warren.error.VungleException.DB_ERROR;
import static com.vungle.warren.error.VungleException.INCORRECT_DEFAULT_API_USAGE;
import static com.vungle.warren.error.VungleException.INVALID_SIZE;
import static com.vungle.warren.error.VungleException.MISSING_REQUIRED_ARGUMENTS_FOR_INIT;
import static com.vungle.warren.error.VungleException.NO_SERVE;
import static com.vungle.warren.error.VungleException.NO_SPACE_TO_INIT;
import static com.vungle.warren.error.VungleException.OPERATION_ONGOING;
import static com.vungle.warren.error.VungleException.PLACEMENT_NOT_FOUND;
import static com.vungle.warren.error.VungleException.SERVER_RETRY_ERROR;
import static com.vungle.warren.error.VungleException.UNKNOWN_ERROR;
import static com.vungle.warren.error.VungleException.VUNGLE_NOT_INTIALIZED;
import static com.vungle.warren.model.Advertisement.ERROR;
import static com.vungle.warren.model.Advertisement.NEW;
import static com.vungle.warren.model.Advertisement.READY;
import static java.lang.Boolean.TRUE;

/**
 * Main interface with the Vungle ad network.
 */
@Keep
public class Vungle {
    static final Vungle _instance = new Vungle();

    private static final String TAG = Vungle.class.getCanonicalName();
    private static final String COM_VUNGLE_SDK = "com.vungle.sdk";


    /**
     * Consent to be maintained if publisher set the consent before calling init.
     * Also to avoid blocking call for {@link Vungle#updateConsentStatus(Consent, String)}
     */
    private final AtomicReference<Consent> consent = new AtomicReference<>();

    /**
     * Consent version to be maintained
     */
    private volatile String consentVersion;

    /**
     * CCPAStatus to be maintained if publisher set the ccpa status before calling init.
     * Also to avoid blocking call for {@link Vungle#updateCCPAStatus(Consent)}
     */
    private volatile Consent ccpaStatus;

    /**
     * State tracking field which contains true if a current placement is playing an ad.
     */
    private Map<String, Boolean> playOperations = new ConcurrentHashMap<>();

    /**
     * The current application ID. We need to store this in order to reconfigure the SDK as necessary.
     */
    private volatile String appID;

    /**
     * A weak reference to the application context. Once this goes away, we can consider the SDK to be
     * dead, since the host application has lost application context. Holding a strong reference to
     * the context would be a memory leak.
     */
    private Context context;

    private static volatile boolean isInitialized;
    private static AtomicBoolean isInitializing = new AtomicBoolean(false);
    private static AtomicBoolean isDepInit = new AtomicBoolean(false);

    private Gson gson = new GsonBuilder().create();

    //Header Bidding ordinal view count
    private AtomicInteger hbpOrdinalViewCount = new AtomicInteger(0);

    /**
     * Private no-arg constructor to prevent instance generation of this class.
     */
    private Vungle() {
    }

    static Context getAppContext() {
        return _instance.context;
    }

    /**
     * Initialize the Vungle SDK
     * Please use {@link #init(String, Context, InitCallback)}
     * This method exists for backward compatibility. Placements pass in this method are ignored by the SDK.
     * Placements will be filled in from the dashboard to get a list of Valid Placements, call
     * {@link #getValidPlacements()} after Vungle is initialized successfully
     *
     * @param placements Placements pass here will no longer be passed into SDK. Placements will be passed in from the dashboard.
     * @param appId      The identifier for the publisher application. This can be found in the vungle
     *                   dashboard.
     * @param context    Application context.
     * @param callback   Callback that will be triggered once initialization has completed, or if any errors occur.
     * @throws IllegalArgumentException if InitCallback is null
     */
    @Deprecated
    public static void init(@NonNull final Collection<String> placements,
                            @NonNull String appId,
                            @NonNull Context context,
                            @NonNull InitCallback callback) throws IllegalArgumentException {
        init(appId, context, callback, new VungleSettings.Builder().build());
    }

    /**
     * Initialize the Vungle SDK
     *
     * @param appId    The identifier for the publisher application. This can be found in the vungle
     *                 dashboard.
     * @param context  Application context.
     * @param callback Callback that will be triggered once initialization has completed, or if any errors occur.
     * @throws IllegalArgumentException if InitCallback is null
     */
    public static void init(@NonNull final String appId,
                            @NonNull final Context context,
                            @NonNull final InitCallback callback) throws IllegalArgumentException {
        init(appId, context, callback, new VungleSettings.Builder().build());
    }

    /**
     * Initialize the Vungle SDK
     *
     * @param appId    The identifier for the publisher application. This can be found in the vungle
     *                 dashboard.
     * @param context  Application context.
     * @param callback Callback that will be triggered once initialization has completed, or if any errors occur.
     * @param settings Vungle's settings
     * @throws IllegalArgumentException if InitCallback is null
     */
    @SuppressWarnings("squid:S2583")
    public static void init(@NonNull final String appId,
                            @NonNull final Context context,
                            @NonNull final InitCallback callback,
                            @NonNull final VungleSettings settings) throws IllegalArgumentException {

        //String for conforming to Google's versioning logic, DO NOT remove or put this anywhere else
        String VUNGLE_VERSION_STRING = "!SDK-VERSION-STRING!:com.vungle:publisher-sdk-android:" + BuildConfig.VERSION_NAME;

        if (callback == null) {
            throw new IllegalArgumentException("A valid InitCallback required to ensure API calls are being made after initialize is successful");
        }

        if (context == null) {
            callback.onError(new VungleException(MISSING_REQUIRED_ARGUMENTS_FOR_INIT));
            return;
        }

        final RuntimeValues runtimeValues = ServiceLocator.getInstance(context).getService(RuntimeValues.class);
        runtimeValues.settings.set(settings);

        final ServiceLocator serviceLocator = ServiceLocator.getInstance(context);

        Executors sdkExecutors = serviceLocator.getService(Executors.class);

        InitCallback initCallback = callback instanceof InitCallbackWrapper
                ? callback
                : new InitCallbackWrapper(sdkExecutors.getUIExecutor(), callback);

        if ((appId == null || appId.isEmpty())) {
            initCallback.onError(new VungleException(MISSING_REQUIRED_ARGUMENTS_FOR_INIT));
            return;
        }

        if (!(context instanceof Application)) {
            initCallback.onError(new VungleException(APPLICATION_CONTEXT_REQUIRED));
            return;
        }

        if (isInitialized()) {
            Log.d(TAG, "init already complete");
            initCallback.onSuccess();
            return;
        }

        if (isInitializing.getAndSet(true)) {
            Log.d(TAG, "init ongoing");
            initCallback.onError(new VungleException(OPERATION_ONGOING));
            return;
        }

        runtimeValues.initCallback.set(initCallback);

        sdkExecutors.getBackgroundExecutor().execute(new Runnable() {
            @Override
            public void run() {
                if (!isDepInit.getAndSet(true)) {
                    CacheManager cacheManager = serviceLocator.getService(CacheManager.class);
                    final VungleSettings settings = runtimeValues.settings.get();
                    final InitCallback initCallback = runtimeValues.initCallback.get();

                    if (settings != null && cacheManager.getBytesAvailable() < settings.getMinimumSpaceForInit()) {
                        onError(initCallback, new VungleException(NO_SPACE_TO_INIT));
                        deInit();
                        return;
                    }

                    cacheManager.addListener(cacheListener);

                    /// Save the context reference to track application state.
                    _instance.context = context;
                    _instance.appID = appId;

                    //init first
                    Repository repository = serviceLocator.getService(Repository.class);
                    try {
                        repository.init();
                    } catch (DatabaseHelper.DBException e) {
                        onError(initCallback, new VungleException(DB_ERROR));
                        deInit();
                        return;
                    }

                    VungleApiClient vungleApiClient = serviceLocator.getService(VungleApiClient.class);
                    if (vungleApiClient.platformIsNotSupported()) {
                        onError(initCallback, new VungleException(VungleException.OK_HTTP_NOT_SUPPORTED_PLATFORM));
                        deInit();
                        return;
                    }
                    vungleApiClient.init(appId);


                    if (settings != null) {
                        vungleApiClient.setDefaultIdFallbackDisabled(settings.getAndroidIdOptOut());
                    }

                    JobRunner jobRunner = serviceLocator.getService(JobRunner.class);
                    AdLoader adLoader = serviceLocator.getService(AdLoader.class);
                    adLoader.init(jobRunner);

                    //Restore the consent if publisher set it before calling init
                    if (_instance.consent.get() != null) {
                        saveGDPRConsent(repository, _instance.consent.get(), _instance.consentVersion);
                    } else {
                        //Restore it from storage
                        Cookie gdprConsent = repository.load(Cookie.CONSENT_COOKIE, Cookie.class).get();
                        if (gdprConsent == null) {
                            _instance.consent.set(null);
                            _instance.consentVersion = null;
                        } else {
                            _instance.consent.set(getConsent(gdprConsent));
                            _instance.consentVersion = getConsentMessageVersion(gdprConsent);
                        }
                    }

                    //Restore the CCPA if publisher set it before calling init
                    if (_instance.ccpaStatus != null) {
                        updateCCPAStatus(repository, _instance.ccpaStatus);
                    } else {
                        //Restore it from storage
                        Cookie ccpaConsent = repository.load(Cookie.CCPA_COOKIE, Cookie.class).get();
                        _instance.ccpaStatus = getCCPAStatus(ccpaConsent);
                    }

                    Cookie appIdCookie = repository.load(Cookie.APP_ID, Cookie.class).get();
                    if (appIdCookie == null) {
                        appIdCookie = new Cookie(Cookie.APP_ID);
                    }
                    appIdCookie.putValue("appId", appId);
                    try {
                        repository.save(appIdCookie);
                    } catch (DatabaseHelper.DBException e) {
                        if (initCallback != null) {
                            onError(initCallback, new VungleException(NO_SPACE_TO_INIT));
                        }
                        deInit();
                        return;
                    }
                }
                /// Configure the instance.
                _instance.configure(runtimeValues.initCallback.get());

            }
        });
    }

    private static void onError(InitCallback initCallback, VungleException e) {
        if (initCallback != null) {
            initCallback.onError(e);
        }
    }

    static void reConfigure() {
        if (_instance.context == null)
            return;

        ServiceLocator serviceLocator = ServiceLocator.getInstance(_instance.context);
        Executors sdkExecutors = serviceLocator.getService(Executors.class);
        final RuntimeValues runtimeValues = serviceLocator.getService(RuntimeValues.class);

        if (isInitialized()) {
            sdkExecutors.getBackgroundExecutor().execute(new Runnable() {
                @Override
                public void run() {
                    _instance.configure(runtimeValues.initCallback.get());
                }
            });
        } else {
            init(_instance.appID, _instance.context, runtimeValues.initCallback.get());
        }
    }

    /**
     * Request a configuration from the server. This updates the current list of placements and
     * schedules another configuration job in the future. It also has the side-effect of loading the
     * auto_cached ad if it is not downloaded currently.
     *
     * @param callback Callback that will be called when initialization has completed or failed.
     */
    private void configure(@NonNull final InitCallback callback) {
        /// Request a configuration from the server. This happens asynchronously on the network thread.
        try {
            if (context == null)
                throw new IllegalArgumentException("Context is null");

            ServiceLocator serviceLocator = ServiceLocator.getInstance(context);
            VungleApiClient vungleApiClient = serviceLocator.getService(VungleApiClient.class);
            Repository repository = serviceLocator.getService(Repository.class);
            JobRunner jobRunner = serviceLocator.getService(JobRunner.class);

            Response<JsonObject> response = vungleApiClient.config();

            if (response == null) {
                callback.onError(new VungleException(UNKNOWN_ERROR));
                isInitializing.set(false);
                return;
            }

            if (!response.isSuccessful()) {
                long retryAfterHeaderValue = vungleApiClient.getRetryAfterHeaderValue(response);
                if (retryAfterHeaderValue > 0) {
                    jobRunner.execute(ReconfigJob.makeJobInfo(_instance.appID).setDelay(retryAfterHeaderValue));
                    callback.onError(new VungleException(SERVER_RETRY_ERROR));
                    isInitializing.set(false);
                    return;
                }

                callback.onError(new VungleException(CONFIGURATION_ERROR));
                isInitializing.set(false);
                return;
            }

            final SharedPreferences preferences = context.getSharedPreferences(COM_VUNGLE_SDK, Context.MODE_PRIVATE);

            /// If we have not reported this install, fire it off to the ad server
            if (!preferences.getBoolean("reported", false)) {
                vungleApiClient.reportNew().enqueue(new Callback<JsonObject>() {
                    @Override
                    public void onResponse(Call<JsonObject> call, Response<JsonObject> response) {
                        if (response.isSuccessful()) {
                            /// Save the reported state to shared preferences
                            SharedPreferences.Editor editor = preferences.edit();
                            editor.putBoolean("reported", true);
                            editor.apply();
                            Log.d(TAG, "Saving reported state to shared preferences");
                        }
                    }
                    @Override
                    public void onFailure(Call<JsonObject> call, Throwable throwable) {
                        /// Do nothing. Install will be reported on the next init.
                        /// TODO: Should this retryCount more?
                    }
                });
            }

            JsonObject jsonObject = response.body();
            /// Parse out the placements
            JsonArray placementsData = jsonObject.getAsJsonArray("placements");

            if (placementsData == null) {
                callback.onError(new VungleException(CONFIGURATION_ERROR));
                isInitializing.set(false);
                return;
            }

            CleverCacheSettings settings = CleverCacheSettings.fromJson(jsonObject);
            Downloader downloader = serviceLocator.getService(Downloader.class);

            if (settings != null) {
                CleverCacheSettings currentCacheSettings = CleverCacheSettings.deserializeFromString(
                        preferences.getString(CleverCacheSettings.KEY_CLEVER_CACHE, null));

                boolean timestampChanged = currentCacheSettings == null ||
                        currentCacheSettings.getTimestamp() != settings.getTimestamp();

                if (!settings.isEnabled() || timestampChanged) {
                    downloader.clearCache();
                }

                downloader.setCacheEnabled(settings.isEnabled());

                preferences.edit()
                        .putString(CleverCacheSettings.KEY_CLEVER_CACHE, settings.serializeToString())
                        .apply();
            } else {
                downloader.setCacheEnabled(true);
            }

            final AdLoader adLoader = serviceLocator.getService(AdLoader.class);
//            playOperations.clearCache();
//            adLoader.clearCache();

            List<Placement> newPlacements = new ArrayList<>();
            for (JsonElement jsonElement : placementsData) {
                newPlacements.add(new Placement(jsonElement.getAsJsonObject()));
            }

            repository.setValidPlacements(newPlacements);

            if (jsonObject.has("gdpr")) {
                // create cooking regardless of whether user is in CountryDataProtected or not
                Cookie gdprConsent = repository.load(Cookie.CONSENT_COOKIE, Cookie.class).get();
                if (gdprConsent == null) {
                    // Otherwise, create a new one set default values
                    gdprConsent = new Cookie(Cookie.CONSENT_COOKIE);
                    gdprConsent.putValue("consent_status", "unknown");
                    gdprConsent.putValue("consent_source", "no_interaction");
                    gdprConsent.putValue("timestamp", 0L);
                }

                JsonObject gdprJsonObject = jsonObject.getAsJsonObject("gdpr");

                boolean isCountryDataProtected = JsonUtil.hasNonNull(gdprJsonObject, "is_country_data_protected") && gdprJsonObject.get("is_country_data_protected").getAsBoolean();
                String consentTitle = JsonUtil.hasNonNull(gdprJsonObject, "consent_title") ? gdprJsonObject.get("consent_title").getAsString() : "";
                String consentMessage = JsonUtil.hasNonNull(gdprJsonObject, "consent_message") ? gdprJsonObject.get("consent_message").getAsString() : "";
                String consentMessageVersion = JsonUtil.hasNonNull(gdprJsonObject, "consent_message_version") ? gdprJsonObject.get("consent_message_version").getAsString() : "";
                String acceptText = JsonUtil.hasNonNull(gdprJsonObject, "button_accept") ? gdprJsonObject.get("button_accept").getAsString() : "";
                String denyText = JsonUtil.hasNonNull(gdprJsonObject, "button_deny") ? gdprJsonObject.get("button_deny").getAsString() : "";

                gdprConsent.putValue("is_country_data_protected", isCountryDataProtected);

                gdprConsent.putValue("consent_title", TextUtils.isEmpty(consentTitle) ? "Targeted Ads" : consentTitle);

                gdprConsent.putValue("consent_message", TextUtils.isEmpty(consentMessage) ?
                        "To receive more relevant ad content based on your interactions with our ads, click \"I Consent\" below. Either way, you will see the same amount of ads." : consentMessage);

                // what ever SDK get messageVersion from server, SDK saves and pass
                String strGDPRCurrentSource = gdprConsent.getString("consent_source");
                if (!"publisher".equalsIgnoreCase(strGDPRCurrentSource)) {      //Condition for not to override the publisher version with server version
                    gdprConsent.putValue("consent_message_version", TextUtils.isEmpty(consentMessageVersion) ? "" : consentMessageVersion);
                }

                gdprConsent.putValue("button_accept", TextUtils.isEmpty(acceptText) ? "I Consent" : acceptText);

                gdprConsent.putValue("button_deny", TextUtils.isEmpty(denyText) ? "I Do Not Consent" : denyText);

                repository.save(gdprConsent);
            }

            if (jsonObject.has("ri")) {
                Cookie configCookie = repository.load(Cookie.CONFIG_COOKIE, Cookie.class).get();
                if (configCookie == null) {
                    configCookie = new Cookie(Cookie.CONFIG_COOKIE);
                }
                boolean isReportIncentivizedEnabled = jsonObject.getAsJsonObject("ri").get("enabled").getAsBoolean();
                configCookie.putValue("isReportIncentivizedEnabled", isReportIncentivizedEnabled);
                repository.save(configCookie);
            }

            /// Schedule the reconfig job
            if (jsonObject.has("config")) {
                long sleep = jsonObject.getAsJsonObject("config").get("refresh_time").getAsLong();
                jobRunner.execute(ReconfigJob.makeJobInfo(appID).setDelay(sleep));
            }

            try {
                serviceLocator.getService(VisionController.class).setConfig(JsonUtil.hasNonNull(jsonObject, VisionController.VISION) ?
                        gson.fromJson(jsonObject.getAsJsonObject(VisionController.VISION), VisionConfig.class) : new VisionConfig());
            } catch (DatabaseHelper.DBException dbException) {
                Log.e(TAG, "not able to apply vision data config");
            }

            //this is earliest we think that SDK basic init is success
            isInitialized = true;

            /// Inform the publisher that initialization has succeeded.
            callback.onSuccess();

            isInitializing.set(false);

            SessionData sessionData = new SessionData();
            sessionData.setInitTimeStamp(System.currentTimeMillis());

            ServiceLocator.getInstance(context).getService(RuntimeValues.class).sessionData.set(sessionData);

            final Collection<Placement> placements = repository.loadValidPlacements().get();

            /// Clean up the asset and metadata caches
            jobRunner.execute(CleanupJob.makeJobInfo());

            /// Download assets for the auto-cached placement immediately. If assets are already
            /// available, this will do nothing except inform the publisher that the auto-cached
            /// placement is ready.
            /// Download assets for the auto-cached placement immediately. If assets are already
            /// available, this will do nothing except inform the publisher that the auto-cached
            /// placement is ready.

            if (placements != null) {
                final List<Placement> placementList = new ArrayList<>(placements);
                Collections.sort(placementList, new Comparator<Placement>() {
                    @Override
                    public int compare(Placement o1, Placement o2) {
                        return ((Integer) o1.getAutoCachePriority()).compareTo(o2.getAutoCachePriority());
                    }
                });
                Log.d(TAG, "starting jobs for autocached advs");

                ExecutorService uiExecutor = serviceLocator.getService(Executors.class).getUIExecutor();
                uiExecutor.execute(new Runnable() {
                    @Override
                    public void run() {
                        for (final Placement placement : placementList) {
                            if (placement.isAutoCached()) {
                                adLoader.loadEndless(placement, 0);
                            }
                        }
                    }
                });
            }

            /// Send any pending ad reports
            jobRunner.execute(SendReportsJob.makeJobInfo(true));
        } catch (final Throwable throwable) {
            isInitialized = false;
            isInitializing.set(false);
            Log.e(TAG, Log.getStackTraceString(throwable));
            if (throwable instanceof HttpException) {
                callback.onError(new VungleException(CONFIGURATION_ERROR));
            } else if (throwable instanceof DatabaseHelper.DBException) {
                callback.onError(new VungleException(DB_ERROR));
            } else {
                callback.onError(new VungleException(UNKNOWN_ERROR));
            }
        }
    }

    /**
     * Checks if Vungle's SDK is already in initialized state
     *
     * @return true if Vungle SDK is initialized, false otherwise.
     */
    public static boolean isInitialized() {
        // no need to check for number of placements, pubs can init for zero placements
        return isInitialized && (_instance.context != null);
    }

    /**
     * Overrides the previously-set incentivized fields which are used when warning the user if they
     * are attempting to exit an incentivized advertisement prematurely. If there are no local
     * values set, the SDK will use the values set in the Vungle Dashboard as the defaults. This method
     * must be called **before** {@link #playAd(String, AdConfig, PlayAdCallback)} is called.
     *
     * @param title        The dialog box title
     * @param body         The dialog box body text. Values longer than the space allows for will be ellipsised
     *                     at the end.
     * @param keepWatching Continue button text
     * @param close        Close button text.
     */
    public static void setIncentivizedFields(@Nullable final String userID,
                                             @Nullable final String title,
                                             final @Nullable String body,
                                             final @Nullable String keepWatching,
                                             final @Nullable String close) {
        if (_instance.context == null) {
            Log.e(TAG, "Vungle is not initialized, context is null");
            return;
        }

        final ServiceLocator serviceLocator = ServiceLocator.getInstance(_instance.context);

        serviceLocator.getService(Executors.class).getBackgroundExecutor().execute(new Runnable() {
            @Override
            public void run() {
                if (!isInitialized()) {
                    Log.e(TAG, "Vungle is not initialized");
                    return;
                }

                Repository repository = serviceLocator.getService(Repository.class);

                Cookie incentivizedCookie = repository.load(Cookie.INCENTIVIZED_TEXT_COOKIE, Cookie.class).get();
                if (incentivizedCookie == null) {
                    incentivizedCookie = new Cookie(Cookie.INCENTIVIZED_TEXT_COOKIE);
                }

                boolean changed = false;

                if (!TextUtils.isEmpty(title)) {
                    changed = true;
                    incentivizedCookie.putValue("title", title);
                }

                if (!TextUtils.isEmpty(body)) {
                    changed = true;
                    incentivizedCookie.putValue("body", body);
                }

                if (!TextUtils.isEmpty(keepWatching)) {
                    changed = true;
                    incentivizedCookie.putValue("continue", keepWatching);
                }

                if (!TextUtils.isEmpty(close)) {
                    changed = true;
                    incentivizedCookie.putValue("close", close);
                }

                if (!TextUtils.isEmpty(userID)) {
                    changed = true;
                    incentivizedCookie.putValue("userID", userID);
                }

                if (changed) {
                    try {
                        repository.save(incentivizedCookie);
                    } catch (DatabaseHelper.DBException e) {
                        e.printStackTrace();
                    }
                }
            }
        });
    }

    /**
     * Check if we can play an advertisement for the given placement. This method checks out file
     * system to see if there are asset files for the given placement and returns true if we have
     * assets stored which have not expired.
     *
     * @param id The placement identifier.
     * @return true if an advertisement can be played immediately, false otherwise.
     */
    public static boolean canPlayAd(@NonNull final String id) {
        final Context context = _instance.context;

        if (context == null) {
            Log.e(TAG, "Context is null");
            return false;
        }


        ServiceLocator serviceLocator = ServiceLocator.getInstance(context);
        Executors sdkExecutors = serviceLocator.getService(Executors.class);
        TimeoutProvider provider = serviceLocator.getService(TimeoutProvider.class);

        FutureResult futureResult = new FutureResult<>(sdkExecutors.getApiExecutor()
                .submit(new Callable<Boolean>() {
                    @Override
                    public Boolean call() {

                        if (!isInitialized()) {
                            Log.e(TAG, "Vungle is not initialized");
                            return false;
                        }

                        ServiceLocator serviceLocator = ServiceLocator.getInstance(context);
                        Repository repository = serviceLocator.getService(Repository.class);
                        final Advertisement advertisement = repository
                                .findValidAdvertisementForPlacement(id)
                                .get();
                        Placement placement = repository
                                .load(id, Placement.class)
                                .get();

                        if (advertisement == null || placement == null) {
                            return false;
                        }

                        if (placement.getPlacementAdType() != Placement.TYPE_DEFAULT
                                || !AdConfig.AdSize.isDefaultAdSize(placement.getAdSize())
                                && !placement.getAdSize().equals(advertisement.getAdConfig().getAdSize())) {
                            return false;
                        }


                        return canPlayAd(advertisement);
                    }
                })
        );

        return Boolean.TRUE.equals(futureResult.get(provider.getTimeout(), TimeUnit.MILLISECONDS));

    }

    static boolean canPlayAd(final Advertisement advertisement) {
        if (_instance.context == null)
            return false;


        AdLoader adLoader = ServiceLocator.getInstance(_instance.context).getService(AdLoader.class);
        return adLoader.canPlayAd(advertisement);
    }

    /**
     * Play an ad for the given placement ID. If this placement ID is valid and an advertisement is
     * ready to be played, this will cause the {@link VungleActivity} to start and the advertisement
     * will be rendered.
     *
     * @param id       The placement identifier.
     * @param settings Optional settings for playing the advertisement. The full list can be found
     *                 at {@link AdConfig} documentation.
     * @param callback Optional, though strongly encouraged, event listener. This object will be
     *                 notified of the advertisement starting, ending, and any errors that occur
     *                 during rendering.
     */
    public static void playAd(@NonNull final String id, final AdConfig settings, @Nullable final PlayAdCallback callback) {
        if (!isInitialized()) {
            Log.e(TAG, "Locator is not initialized");
            if (callback != null) {
                callback.onError(id, new VungleException(VUNGLE_NOT_INTIALIZED));
            }
            return;
        }

        ServiceLocator serviceLocator = ServiceLocator.getInstance(_instance.context);

        final Executors sdkExecutors = serviceLocator.getService(Executors.class);
        final Repository repository = serviceLocator.getService(Repository.class);
        final AdLoader adLoader = serviceLocator.getService(AdLoader.class);
        final VungleApiClient vungleApiClient = serviceLocator.getService(VungleApiClient.class);

        final PlayAdCallback listener = new PlayAdCallbackWrapper(sdkExecutors.getUIExecutor(), callback);

        sdkExecutors.getBackgroundExecutor().execute(new Runnable() {

            @Override
            public void run() {
                //Don't allow Ad to play when either loading or playing
                if (TRUE.equals(_instance.playOperations.get(id)) || adLoader.isLoading(id)) {
                    listener.onError(id, new VungleException(OPERATION_ONGOING));
                    return;
                }

                final Placement placement = repository.load(id, Placement.class).get();
                if (placement == null) {
                    listener.onError(id, new VungleException(PLACEMENT_NOT_FOUND));
                    return;
                }

                //Todo need to check any other validation needs to be added for invalid size
                if (AdConfig.AdSize.isBannerAdSize(placement.getAdSize())) {
                    // Can not play Banner Ad with PlayAd method
                    listener.onError(id, new VungleException(INVALID_SIZE));
                    return;
                }

                /// Check if an Ad is valid
                boolean streamingOnly = false;

                /// Check if an Ad for this placement exists.
                Advertisement advertisement = repository.findValidAdvertisementForPlacement(id).get();

                try {
                    if (!canPlayAd(advertisement)) {
                        /// Even if we don't have a cached Ad prepared, we can still play a streaming Ad if available.
                        streamingOnly = true;

                        if (advertisement != null && advertisement.getState() == READY) {
                            //assets were somehow deleted
                            repository.saveAndApplyState(advertisement, id, ERROR);
                            if (placement.isAutoCached()) {
                                adLoader.loadEndless(placement, 0);
                            }
                        }
                    } else {
                        /// Ad we have cached is valid and can be used, apply the settings and save them
                        advertisement.configure(settings);
                        repository.save(advertisement);
                    }
                } catch (DatabaseHelper.DBException ignored) {
                    listener.onError(id, new VungleException(DB_ERROR));
                    return;
                }

                if (_instance.context != null) {
                    /// Inform the Ad Server that we are about to play an Ad.
                    /// If there is a better Ad available, it will be substituted here.
                    final boolean finalStreamingOnly = streamingOnly;
                    final Advertisement finalAdvertisement = advertisement;

                    if (vungleApiClient.canCallWillPlayAd()) {
                        vungleApiClient.willPlayAd(placement.getId(), placement.isAutoCached(),
                                streamingOnly ? "" : advertisement.getAdToken()).enqueue(new Callback<JsonObject>() {

                            @Override
                            public void onResponse(Call<JsonObject> call, final Response<JsonObject> response) {
                                //Lets do it in separate thread, since there are few IO calls involved.
                                sdkExecutors.getBackgroundExecutor().execute(new Runnable() {
                                    @Override
                                    public void run() {
                                        Advertisement streamingAd = null;
                                        if (response.isSuccessful()) {
                                            JsonObject responseBody = response.body();
                                            if (responseBody != null && responseBody.has("ad")) {
                                                try {
                                                    JsonObject adJson = responseBody.getAsJsonObject("ad");
                                                    streamingAd = new Advertisement(adJson);

                                                    /// Update the settings for this advertisement
                                                    streamingAd.configure(settings);

                                                    /// If we make it here, it means that there is a replacement streaming ad for
                                                    /// this placement. So we update the metadata and save it so the activity can
                                                    /// use the fresh values.
                                                    repository.saveAndApplyState(streamingAd, id, NEW);
                                                } catch (IllegalArgumentException e) {
                                                    Log.v(TAG, "Will Play Ad did not respond with a replacement. Move on.");
                                                } catch (Exception e) {
                                                    Log.e(TAG, "Error using will_play_ad!", e);
                                                }
                                            }
                                        }

                                        if (finalStreamingOnly) {
                                            if (streamingAd == null) {
                                                listener.onError(id, new VungleException(NO_SERVE));
                                            } else {
                                                renderAd(id, listener, placement, streamingAd);
                                            }
                                        } else {
                                            renderAd(id, listener, placement, finalAdvertisement);
                                        }
                                    }
                                });
                            }

                            @Override
                            public void onFailure(Call<JsonObject> call, Throwable throwable) {
                                //Lets do it in separate thread, since there are few IO calls involved.
                                sdkExecutors.getBackgroundExecutor().execute(new Runnable() {
                                    @Override
                                    public void run() {
                                        if (finalStreamingOnly) {
                                            listener.onError(id, new VungleException(NO_SERVE));
                                        } else {
                                            renderAd(id, listener, placement, finalAdvertisement);
                                        }
                                    }
                                });
                            }
                        });
                    } else {
                        if (finalStreamingOnly) {
                            listener.onError(id, new VungleException(NO_SERVE));
                        } else {
                            renderAd(id, listener, placement, finalAdvertisement);
                        }
                    }
                }
            }
        });
    }

    /**
     * Private helper method to reduce callback hell, this method starts playing the given advertisement
     * in a {@link VungleActivity}. It also creates an event listener in order to control the flow
     * of events once the advertisement ends.
     *
     * @param placementId   The placement identifier
     * @param listener      The optional listener for playback events
     * @param placement     The placement in which the advertisement is being rendered
     * @param advertisement The advertisement metadata for the advertisement that will be played.
     */
    private static synchronized void renderAd(@NonNull final String placementId,
                                              @Nullable final PlayAdCallback listener,
                                              final Placement placement,
                                              final Advertisement advertisement) {
        /// Subscribe to the event bus of the activity before starting the activity, otherwise
        /// the Publisher notices it has no subscribers and does not emit the start value.
        if (!isInitialized()) {
            Log.e(TAG, "Sdk is not initilized");
            return;
        }
        final ServiceLocator serviceLocator = ServiceLocator.getInstance(_instance.context);

        VungleActivity.setEventListener(new AdEventListener(
                placementId,
                _instance.playOperations,
                listener,
                serviceLocator.getService(Repository.class),
                serviceLocator.getService(AdLoader.class),
                serviceLocator.getService(JobRunner.class),
                serviceLocator.getService(VisionController.class),
                placement,
                advertisement
        ) {
            @Override
            protected void onFinished() {
                super.onFinished();
                VungleActivity.setEventListener(null);
            }
        });

        /// Start the activity, and if there are any extras that have been overriden by the application, apply them.
        /// Vungleflex is same as VungleActivity, this is all about android rotation issue in 8.0 and translucent attribute.
        boolean isFlex = advertisement != null && "flexview".equals(advertisement.getTemplateType());
        Intent intent = new Intent(_instance.context, isFlex ? VungleFlexViewActivity.class : VungleActivity.class);
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        intent.putExtra(VungleActivity.PLACEMENT_EXTRA, placementId);

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
            _instance.context.startActivity(intent);
        } else {
            ActivityManager.startWhenForeground(_instance.context, intent);
        }
    }

    /**
     * Request the Vungle SDK to load the assets for an advertisement by the placement identifier.
     * This will cause us to request an ad from the Ad Server, and if the bid is filled, the assets
     * will be downloaded. It is possible for the bid to get no responses and therefore no assets
     * will be loaded. In this case, the SDK will automatically retry at a later time, specified by
     * the Vungle Server. The callback will be notified that the assets are pending download.
     *
     * @param id       The placement identifier for which assets should be loaded.
     * @param callback The optional callback which will be notified when the assets are loaded, or
     *                 if they are deferred. Errors will also be sent through this callback.
     */
    public static void loadAd(@NonNull final String id, @Nullable final LoadAdCallback callback) {
        loadAd(id, new AdConfig(), callback);
    }

    /**
     * Request the Vungle SDK to load the assets for an advertisement by the placement identifier.
     * This will cause us to request an ad from the Ad Server, and if the bid is filled, the assets
     * will be downloaded. It is possible for the bid to get no responses and therefore no assets
     * will be loaded. In this case, the SDK will automatically retry at a later time, specified by
     * the Vungle Server. The callback will be notified that the assets are pending download.
     *
     * @param id       The placement identifier for which assets should be loaded.
     * @param adConfig Optional AdConfig field to set ad size and playback options
     * @param callback The optional callback which will be notified when the assets are loaded, or
     *                 if they are deferred. Errors will also be sent through this callback.
     */
    public static void loadAd(@NonNull final String id,
                              @Nullable final AdConfig adConfig,
                              @Nullable final LoadAdCallback callback) {

        /// Validate SDK State
        if (!isInitialized()) {
            Log.e(TAG, "Vungle is not initialized");
            if (callback != null) {
                callback.onError(id, new VungleException(VUNGLE_NOT_INTIALIZED));
            }
            return;
        }

        if (adConfig != null && !AdConfig.AdSize.isDefaultAdSize(adConfig.getAdSize())) {
            if (callback != null) {
                callback.onError(id, new VungleException(VungleException.INCORRECT_DEFAULT_API_USAGE));
            }
        }
        loadAdInternal(id, adConfig, callback);
    }

    static void loadAdInternal(@NonNull final String id,
                               @Nullable AdConfig adConfig,
                               @Nullable final LoadAdCallback callback) {
        /// Validate SDK State
        if (!isInitialized()) {
            Log.e(TAG, "Vungle is not initialized");
            if (callback != null) {
                callback.onError(id, new VungleException(VUNGLE_NOT_INTIALIZED));
            }
            return;
        }

        final ServiceLocator serviceLocator = ServiceLocator.getInstance(_instance.context);

        final LoadAdCallback listener = new LoadAdCallbackWrapper(serviceLocator
                .getService(Executors.class)
                .getUIExecutor(), callback);

        /*
         * If adConfig set by the publisher is null set to default
         */
        serviceLocator.getService(AdLoader.class).load(id, adConfig == null ? new AdConfig() : adConfig, listener);
    }

    /**
     * Clear the internal caches and files. This triggers a config call to happen again.
     * This method is private but used in test app using reflection.
     */
    private static void clearCache() {
        if (!isInitialized()) {
            Log.e(TAG, "Vungle is not initialized");
            return;
        }

        final ServiceLocator serviceLocator = ServiceLocator.getInstance(_instance.context);
        serviceLocator.getService(Executors.class).getBackgroundExecutor().execute(new Runnable() {
            @Override
            public void run() {
                //Clear the persisted data
                serviceLocator.getService(Downloader.class).cancelAll();
                serviceLocator.getService(AdLoader.class).clear();
                serviceLocator.getService(Repository.class).clearAllData();
                _instance.playOperations.clear();

                //Reset ccpa status due to database dropped.
                _instance.ccpaStatus = null;

                /// Configure again, which will auto-load the auto-cached placement and hydrate our metadata.
                _instance.configure(serviceLocator.getService(RuntimeValues.class).initCallback.get());
            }
        });
    }

    /**
     * Delete all existing advertisement.
     * This method is private but used in test app using reflection.
     */
    private static void clearAdvertisements() {
        if (!isInitialized()) {
            Log.e(TAG, "Vungle is not initialized");
            return;
        }

        final ServiceLocator serviceLocator = ServiceLocator.getInstance(_instance.context);
        serviceLocator.getService(Executors.class).getBackgroundExecutor().execute(new Runnable() {
            @Override
            public void run() {
                //Clear the persisted data
                serviceLocator.getService(Downloader.class).cancelAll();
                serviceLocator.getService(AdLoader.class).clear();
                final Repository repository = serviceLocator.getService(Repository.class);
                Executors sdkExecutors = serviceLocator.getService(Executors.class);
                sdkExecutors.getBackgroundExecutor().execute(new Runnable() {
                    @Override
                    public void run() {
                        List<Advertisement> ads = repository.loadAll(Advertisement.class).get();
                        if (ads != null) {
                            for (Advertisement ad : ads) {
                                try {
                                    repository.deleteAdvertisement(ad.getId());
                                } catch (DatabaseHelper.DBException ignored) {
                                }
                            }
                        }
                    }
                });
            }
        });
    }

    /**
     * VungleNativeAd can be used inside any Android view container. Do not use this method to retrieve a
     * banner placement.
     *
     * @param placementId    - placementId for Native ad Placement configured for the right AdSize
     * @param adConfig       - Set the  {@link AdConfig.AdSize} for Native Ad view corresponding to the Placement configuration
     * @param playAdCallback - Callback to receive onAdStart, onAdEnd, onAdError Callbacks
     * @return VungleNativeAd
     */
    @Nullable
    public static VungleNativeAd getNativeAd(@NonNull String placementId, @Nullable AdConfig adConfig, @Nullable final PlayAdCallback playAdCallback) {
        if (adConfig == null) {
            adConfig = new AdConfig();
        }

        if (AdConfig.AdSize.isDefaultAdSize(adConfig.getAdSize())) {
            return getNativeAdInternal(placementId, adConfig, playAdCallback);
        } else {
            if (playAdCallback != null) {
                Log.e(TAG, "Please use Banners.getBanner(... ) to retrieve Banner Ad");
                playAdCallback.onError(placementId, new VungleException(INCORRECT_DEFAULT_API_USAGE));
            }
            return null;
        }
    }

    @Nullable
    static VungleNativeView getNativeAdInternal(String placementId, AdConfig adConfig, final PlayAdCallback playAdCallback) {
        //todo AND-2511
        if (_instance.context == null) {
            Log.e(TAG, "Vungle is not initialized, returned VungleNativeAd = null");
            if (playAdCallback != null) {
                playAdCallback.onError(placementId, new VungleException(VUNGLE_NOT_INTIALIZED));
            }
            return null;
        }

        final ServiceLocator serviceLocator = ServiceLocator.getInstance(_instance.context);
        final AdLoader adLoader = serviceLocator.getService(AdLoader.class);

        //todo allow same placement on multiple banners for simple integration?
        if (TRUE.equals(_instance.playOperations.get(placementId)) || adLoader.isLoading(placementId)) {
            Log.e(TAG, "Playing or Loading operation ongoing. Playing "
                    + _instance.playOperations.get(placementId)
                    + " Loading: " + adLoader.isLoading(placementId));
            if (playAdCallback != null) {
                playAdCallback.onError(placementId, new VungleException(OPERATION_ONGOING));
            }
            return null;
        }

        return new VungleNativeView(
                _instance.context.getApplicationContext(),
                placementId,
                adConfig,
                serviceLocator.getService(PresentationFactory.class),
                new AdEventListener(
                        placementId,
                        _instance.playOperations,
                        playAdCallback,
                        serviceLocator.getService(Repository.class),
                        adLoader,
                        serviceLocator.getService(JobRunner.class),
                        serviceLocator.getService(VisionController.class),
                        null,
                        null
                )
        );
    }

    /**
     * If Vungle is initialized, will return list of placement identifiers associated with
     * it's corresponding app id
     *
     * @return a list of valid placement identifiers
     */
    public static Collection<String> getValidPlacements() {
        if (!isInitialized()) {
            Log.e(TAG, "Vungle is not initialized return empty placements list");
            return Collections.emptyList();
        }

        ServiceLocator serviceLocator = ServiceLocator.getInstance(_instance.context);
        Repository repository = serviceLocator.getService(Repository.class);
        TimeoutProvider provider = serviceLocator.getService(TimeoutProvider.class);

        Collection<String> placements = repository.getValidPlacementIds().get(provider.getTimeout(), TimeUnit.MILLISECONDS);
        if (placements == null) {
            return Collections.emptyList();
        } else {
            return placements;
        }
    }


    /**
     * If Vungle is initialized, will return list of placement identifiers associated with
     * it's corresponding app id.
     * This method is protected but used in test app using reflection.
     *
     * @return a list of valid placement identifiers
     */
    @VisibleForTesting
    static Collection<Placement> getValidPlacementModels() {
        if (!isInitialized()) {
            Log.e(TAG, "Vungle is not initialized return empty placements list");
            return Collections.emptyList();
        }

        ServiceLocator serviceLocator = ServiceLocator.getInstance(_instance.context);
        Repository repository = serviceLocator.getService(Repository.class);
        TimeoutProvider provider = serviceLocator.getService(TimeoutProvider.class);

        Collection<Placement> placements = repository.loadValidPlacements().get(provider.getTimeout(), TimeUnit.MILLISECONDS);
        if (placements == null) {
            return Collections.emptyList();
        } else {
            return placements;
        }
    }

    /**
     * GDPR or CCPA Consent Status
     */
    @Keep
    public enum Consent {
        OPTED_IN,
        OPTED_OUT
    }

    /**
     * If handling GDPR Consent dialog with own implementation, use this dialog to update Vungle on
     * user consent status.
     * <p>
     * Updates the data-gathering consent status of the user. This is gathered by the publisher
     * and provided to Vungle. It is then stored on disk and used whenever we are sending information
     * to the ad server.
     *
     * @param status                If true, the user has consented to us gathering data about their device.
     * @param consentMessageVersion Optional version string that can be passed indicating the version
     *                              of your shown consent message users acted on
     */
    public static void updateConsentStatus(@NonNull final Consent status, @Nullable final String consentMessageVersion) {
        //Additional check since method is public
        if (status == null) {
            Log.e(TAG, "Cannot set consent with a null consent, please check your code");
            return;
        }

        _instance.consent.set(status);
        _instance.consentVersion = consentMessageVersion;

        if (isInitialized() && isDepInit.get()) {
            ServiceLocator serviceLocator = ServiceLocator.getInstance(_instance.context);
            final Repository repository = serviceLocator.getService(Repository.class);
            saveGDPRConsent(repository, _instance.consent.get(), _instance.consentVersion);
        } else {
            Log.e(TAG, "Vungle is not initialized");
        }

    }

    private static void saveGDPRConsent(@NonNull final Repository repository, @NonNull final Consent status, @Nullable final String consentMessageVersion){
        /// If there is already a cookie on disk for consent status, re-use it.
        repository.load(Cookie.CONSENT_COOKIE, Cookie.class, new Repository.LoadCallback<Cookie>() {
            @Override
            public void onLoaded(Cookie gdprConsent) {
                if (gdprConsent == null) {
                    /// Otherwise, create a new one
                    gdprConsent = new Cookie(Cookie.CONSENT_COOKIE);
                }
                gdprConsent.putValue("consent_status", status == Consent.OPTED_IN ? "opted_in" : "opted_out");
                gdprConsent.putValue("timestamp", System.currentTimeMillis() / 1000); /// Server requires seconds.
                gdprConsent.putValue("consent_source", "publisher");
                gdprConsent.putValue("consent_message_version", consentMessageVersion == null ? "" : consentMessageVersion);
                repository.save(gdprConsent, null);
            }
        });
    }

    /**
     * @return Whether a user for Vungle has Accepted GDPR Consent. Returns null if user has had the opportunity yet to make opt in or out.
     */
    @Nullable
    public static Consent getConsentStatus() {
        return _instance.consent.get();
    }

    /**
     * @return  Whether a user for Vungle has Accepted GDPR Consent Message Version. {@link Vungle#updateConsentStatus(Consent, String)}
     */
    public static String getConsentMessageVersion() {
        return _instance.consentVersion;
    }

    private static Consent getConsent(Cookie gdprConsent) {
        if (gdprConsent == null) {
            return null;
        }
        return "opted_in".equals(gdprConsent.getString("consent_status")) ? Consent.OPTED_IN : Consent.OPTED_OUT;

    }

    private static String getConsentMessageVersion(Cookie gdprConsent) {
        if (gdprConsent == null) {
            return null;
        } else {
            return gdprConsent.getString("consent_message_version");
        }
    }

    /**
     * If handling CCPA Consent dialog with own implementation, use this dialog to update Vungle on
     * user consent status.
     * <p>
     * Updates the data-gathering consent status of the user. This is gathered by the publisher
     * and provided to Vungle. It is then stored on disk and used whenever we are sending information
     * to the ad server.
     *
     * @param status                If true, the user has consented to us gathering data about their device.
     */
    public static void updateCCPAStatus(@NonNull final Consent status) {
        if (status == null) {
            Log.e(TAG, "Unable to update CCPA status, Invalid input parameter.");
            return;
        }

        //To ensure update state should be maintained.
        _instance.ccpaStatus = status;

        if (!isInitialized() || !isDepInit.get()) {
            Log.e(TAG, "Vungle is not initialized");
            return;
        }

        ServiceLocator serviceLocator = ServiceLocator.getInstance(_instance.context);
        Repository repository = serviceLocator.getService(Repository.class);

        updateCCPAStatus(repository, status);
    }

    private static void updateCCPAStatus(@NonNull Repository repository, @NonNull Consent status) {
        Cookie ccpaConsent = repository.load(Cookie.CCPA_COOKIE, Cookie.class).get();
        if (ccpaConsent == null) {
            ccpaConsent = new Cookie(Cookie.CCPA_COOKIE);
        }
        ccpaConsent.putValue(Cookie.CCPA_CONSENT_STATUS, status == Consent.OPTED_OUT ? Cookie.CONSENT_STATUS_OPTED_OUT : Cookie.CONSENT_STATUS_OPTED_IN);
        try {
            repository.save(ccpaConsent);
        } catch (DatabaseHelper.DBException e) {
            Log.e(TAG, "Unable to update CCPA status: Database exception." + e.getLocalizedMessage());
        }
    }

    /**
     * @return Whether a user for Vungle has Accepted CCPA Consent.
     * Returns null if SDK has not been called earlier with {@link Vungle#updateCCPAStatus(Consent)}.
     */
    @Nullable
    public static Consent getCCPAStatus() {
        return _instance.ccpaStatus;
    }

    private static Consent getCCPAStatus(@Nullable Cookie ccpaConsent) {
        if (ccpaConsent == null) {
            return null;
        }
        return Cookie.CONSENT_STATUS_OPTED_OUT.equals(ccpaConsent.getString(Cookie.CCPA_CONSENT_STATUS)) ? Consent.OPTED_OUT : Consent.OPTED_IN;
    }

    /**
     * If Vungle is initialized, will return an encoded string of placements bid token.
     * This method might be called from adapter side.
     * The SDK must be initialized first.
     *
     * @param maxBidTokenSize the bid tokens size limitation.
     * @return an encoded string contains available bid tokens.
     */
    @Nullable
    public static String getAvailableBidTokens(final int maxBidTokenSize) {
        final Context context = _instance.context;
        if (context == null) {
            Log.e(TAG, "Context is null");
            return null;
        }

        ServiceLocator serviceLocator = ServiceLocator.getInstance(context);
        Executors sdkExecutors = serviceLocator.getService(Executors.class);
        TimeoutProvider provider = serviceLocator.getService(TimeoutProvider.class);

        FutureResult<String> futureResult = new FutureResult<>(sdkExecutors.getApiExecutor()
                .submit(new Callable<String>() {
            @Override
            public String call() throws Exception {
                if (!isInitialized()) {
                    Log.e(TAG, "Vungle is not initialized, available bid token is null");
                    return null;
                }

                _instance.hbpOrdinalViewCount.incrementAndGet();
                Repository repository = ServiceLocator.getInstance(_instance.context).getService(Repository.class);
                List<String> availableBidTokens = repository.getAvailableBidTokens(maxBidTokenSize).get();
                if (availableBidTokens == null || availableBidTokens.isEmpty()) {
                    return null;
                }

                String encodeVersion = "2";
                String bidTokens = TextUtils.join(",", availableBidTokens);
                String bidTokensWithOrdinalViewCount = bidTokens + ":" + _instance.hbpOrdinalViewCount.toString();
                byte[] tokenBytes = Base64.encode(bidTokensWithOrdinalViewCount.getBytes(), Base64.NO_WRAP);
                String encodedTokenChain = new String(tokenBytes, Charset.defaultCharset());
                return encodeVersion + ":" + encodedTokenChain;
            }
        }));

        return futureResult.get(provider.getTimeout(), TimeUnit.MILLISECONDS);
    }

    /**
     * Closes currently playing Ad (only for Native FlexView , not supported for the Interstitial Ad formats)
     * Currently ability to close already Playing ad is supported only for Placements
     * that show FlexView ads.
     *
     * @param placementReferenceId placement id.
     * @return True if Vungle will attempt to close this ad. False if placement is invalid or app has state issue.
     * For other errors you will receive an error message, check debug logs for additional detail.
     */
    public static boolean closeFlexViewAd(@NonNull final String placementReferenceId) {
        if (!isInitialized()) {
            Log.e(TAG, "Vungle is not initialized, can't close flex view ad");
            return false;
        }
        /// Use a broadcast receiver to ferry messages to the Activity without needing to reference
        /// it directly.
        Intent broadcast = new Intent(AdvertisementBus.ACTION);
        broadcast.putExtra(AdvertisementBus.PLACEMENT, placementReferenceId);
        broadcast.putExtra(AdvertisementBus.COMMAND, AdvertisementBus.CLOSE_FLEX);
        LocalBroadcastManager.getInstance(_instance.context).sendBroadcast(broadcast);
        return true;
    }

    /**
     * Method to register Header bidding callback.
     *
     * @param headerBiddingCallback {@link HeaderBiddingCallback} instance to get notified about
     *                              bidding token and ad availability.
     */
    public static void setHeaderBiddingCallback(HeaderBiddingCallback headerBiddingCallback) {
        if (_instance.context == null)
            return;

        ServiceLocator serviceLocator = ServiceLocator.getInstance(_instance.context);
        RuntimeValues runtimeValues = serviceLocator.getService(RuntimeValues.class);

        runtimeValues.headerBiddingCallback.set(new HeaderBiddingCallbackWrapper(
                serviceLocator.getService(Executors.class).getUIExecutor(),
                headerBiddingCallback));
    }

    /**
     * Lite version of de-initialization, can be used in Unit Tests and {@link Vungle#init(String, Context, InitCallback)}
     */
    protected static void deInit() {
        if (_instance.context != null) {
            ServiceLocator serviceLocator = ServiceLocator.getInstance(_instance.context);
            if (serviceLocator.isCreated(CacheManager.class)) {
                serviceLocator.getService(CacheManager.class).removeListener(cacheListener);
            }
            if (serviceLocator.isCreated(Downloader.class)) {
                serviceLocator.getService(Downloader.class).cancelAll();
            }
            if (serviceLocator.isCreated(AdLoader.class)) {
                serviceLocator.getService(AdLoader.class).clear();
            }
            _instance.playOperations.clear();
        }

        ServiceLocator.deInit();

        isInitialized = false;
        isDepInit.set(false);
        isInitializing.set(false);
    }

    private static CacheManager.Listener cacheListener = new CacheManager.Listener() {
        @Override
        public void onCacheChanged() {
            if (_instance.context == null)
                return;

            stopPlaying();
            ServiceLocator serviceLocator = ServiceLocator.getInstance(_instance.context);

            CacheManager cacheManager = serviceLocator.getService(CacheManager.class);
            Downloader downloader = serviceLocator.getService(Downloader.class);
            if (cacheManager.getCache() != null) {
                List<DownloadRequest> requests = downloader.getAllRequests();
                String newPath = cacheManager.getCache().getPath();
                for (DownloadRequest request : requests) {
                    if (!request.path.startsWith(newPath)) {
                        downloader.cancel(request);
                    }
                }
            }
            downloader.init();
        }
    };

    private static void stopPlaying() {
        if (_instance.context == null)
            return;

        Intent broadcast = new Intent(AdvertisementBus.ACTION);
        broadcast.putExtra(AdvertisementBus.COMMAND, AdvertisementBus.STOP_ALL);
        LocalBroadcastManager.getInstance(_instance.context).sendBroadcast(broadcast);
    }

    @Nullable
    static Context appContext() {
        if (_instance != null)
            return _instance.context;

        return null;
    }
}