package com.vungle.warren;

import android.annotation.TargetApi;
import android.os.Build;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.vungle.warren.model.Cookie;
import com.vungle.warren.model.VisionData;
import com.vungle.warren.model.VisionDataDBAdapter;
import com.vungle.warren.persistence.DatabaseHelper;
import com.vungle.warren.persistence.Repository;
import com.vungle.warren.utility.NetworkProvider;
import com.vungle.warren.vision.VisionAggregationData;
import com.vungle.warren.vision.VisionAggregationInfo;
import com.vungle.warren.vision.VisionConfig;

import java.util.List;
import java.util.concurrent.TimeUnit;

import static android.net.ConnectivityManager.TYPE_BLUETOOTH;
import static android.net.ConnectivityManager.TYPE_ETHERNET;
import static android.net.ConnectivityManager.TYPE_MOBILE;
import static android.net.ConnectivityManager.TYPE_MOBILE_DUN;
import static android.net.ConnectivityManager.TYPE_VPN;
import static android.net.ConnectivityManager.TYPE_WIFI;
import static android.net.ConnectivityManager.TYPE_WIMAX;

class VisionController {
    static final String VISION = "vision";
    static final String VISION_COOKIE = "visionCookie";
    static final String DATA_SCIENCE_CACHE = "data_science_cache";
    static final String AGGREGATE = "aggregate";
    static final String WINDOW = "window";
    static final String LAST_VIEWED_CREATIVE_ID = "last_viewed_creative_id";
    static final String LAST_VIEWED_CAMPAIGN_ID = "last_viewed_campaign_id";
    static final String LAST_VIEWED_ADVERTISER_ID = "last_viewed_advertiser_id";
    static final String TOTAL_VIEW_COUNT = "total_view_count";
    static final String VIEW_COUNT = "view_count";
    static final String LAST_TIME_VIEWED = "last_time_viewed";
    static final String CREATIVE_DETAILS = "creative_details";
    static final String CAMPAIGN_DETAILS = "campaign_details";
    static final String ADVERTISER_DETAILS = "advertiser_details";
    static final String FILTER_ID = "_id";

    @NonNull
    private final Repository repository;
    @NonNull
    private final NetworkProvider networkProvider;
    @NonNull
    private VisionConfig config = new VisionConfig();

    VisionController(@NonNull Repository repository, @NonNull NetworkProvider networkProvider) {
        this.repository = repository;
        this.networkProvider = networkProvider;
    }

    private @Nullable
    String getDataScienceCache() {
        Cookie cookie = repository.load(VISION_COOKIE, Cookie.class).get();
        return cookie == null ? null : cookie.getString(DATA_SCIENCE_CACHE);
    }

    void setDataScienceCache(@Nullable String cache) throws DatabaseHelper.DBException {
        Cookie cookie = new Cookie(VISION_COOKIE);
        if (cache != null) {
            cookie.putValue(DATA_SCIENCE_CACHE, cache);
        }
        repository.save(cookie);
    }

    void setConfig(@NonNull VisionConfig config) throws DatabaseHelper.DBException {
        this.config = config;
        if (config.enabled) {
            repository.trimVisionData(config.viewLimit != null ? config.viewLimit.device : 0);
        }
    }

    void reportData(@NonNull String creative, @NonNull String campaign, @NonNull String advertiser) throws DatabaseHelper.DBException {
        repository.save(new VisionData(System.currentTimeMillis(), creative, campaign, advertiser));
        repository.trimVisionData(config.viewLimit != null ? config.viewLimit.device : 0);
    }

    void clearData() throws DatabaseHelper.DBException {
        repository.trimVisionData(0);
    }

    @TargetApi(Build.VERSION_CODES.LOLLIPOP)
    public @NonNull
    JsonObject getPayload() {
        JsonObject vision = new JsonObject();
        String data_science_cache = getDataScienceCache();
        if(data_science_cache != null) {
            vision.addProperty(DATA_SCIENCE_CACHE, data_science_cache);
        }
        int limit = 0;
        if (config.viewLimit != null) {
            switch (networkProvider.getCurrentNetworkType()) {
                case TYPE_ETHERNET:
                case TYPE_WIFI:
                case TYPE_WIMAX:
                    limit = config.viewLimit.wifi > 0 ? config.viewLimit.wifi : config.viewLimit.device;
                    break;
                case TYPE_BLUETOOTH:
                case TYPE_VPN:
                case TYPE_MOBILE_DUN:
                case TYPE_MOBILE:
                    limit = config.viewLimit.mobile > 0 ? config.viewLimit.mobile : config.viewLimit.device;
                    break;
                default:
                    limit = config.viewLimit.device;
                    break;
            }
        }
        final long start = System.currentTimeMillis();
        JsonArray windows = new JsonArray();
        vision.add(AGGREGATE, windows);
        if (config.aggregationTimeWindows != null) {
            for (int window : config.aggregationTimeWindows) {
                long after = start - TimeUnit.DAYS.toMillis(window);
                VisionAggregationInfo info = repository.getVisionAggregationInfo(after).get();
                JsonObject windowData = new JsonObject();
                windowData.addProperty(WINDOW, window);
                windowData.addProperty(LAST_VIEWED_CREATIVE_ID, info != null ? info.lastCreative : null);
                windowData.addProperty(TOTAL_VIEW_COUNT, info != null ? info.totalCount : 0);
                if (config.aggregationFilters != null) {
                    for (String filter : config.aggregationFilters) {
                        JsonArray details = new JsonArray();
                        windowData.add(filter, details);
                        filter = getFilterName(filter);
                        List<VisionAggregationData> data = repository.getVisionAggregationData(after, limit, filter).get();
                        if (data != null) {
                            for (VisionAggregationData d : data) {
                                JsonObject elem = new JsonObject();
                                elem.addProperty(filter + FILTER_ID, d.id);
                                elem.addProperty(VIEW_COUNT, d.viewCount);
                                elem.addProperty(LAST_TIME_VIEWED, TimeUnit.MILLISECONDS.toSeconds(d.lastTimeStamp));
                                details.add(elem);
                            }
                        }
                    }
                }
                windows.add(windowData);
            }
        }
        return vision;
    }

    @VisibleForTesting
    String getFilterName(String filter) {
        switch (filter) {
            case CREATIVE_DETAILS:
                return VisionDataDBAdapter.VisionDataColumns.COLUMN_CREATIVE;
            case CAMPAIGN_DETAILS:
                return VisionDataDBAdapter.VisionDataColumns.COLUMN_CAMPAIGN;
            case ADVERTISER_DETAILS:
                return VisionDataDBAdapter.VisionDataColumns.COLUMN_ADVERTISER;
            default:
                return null;
        }
    }

    @VisibleForTesting
    VisionConfig getVisionConfig() {
        return this.config;
    }

    boolean isEnabled() {
        return config.enabled;
    }
}
