package com.vungle.warren;

import android.annotation.SuppressLint;
import android.content.Context;
import androidx.annotation.NonNull;
import androidx.annotation.VisibleForTesting;

import com.vungle.warren.analytics.VungleAnalytics;
import com.vungle.warren.downloader.AssetDownloader;
import com.vungle.warren.downloader.CleverCache;
import com.vungle.warren.downloader.Downloader;
import com.vungle.warren.downloader.DownloaderCache;
import com.vungle.warren.downloader.LRUCachePolicy;
import com.vungle.warren.persistence.CacheManager;
import com.vungle.warren.persistence.Designer;
import com.vungle.warren.persistence.GraphicDesigner;
import com.vungle.warren.persistence.Repository;
import com.vungle.warren.tasks.JobCreator;
import com.vungle.warren.tasks.JobRunner;
import com.vungle.warren.tasks.ReconfigJob;
import com.vungle.warren.tasks.VungleJobCreator;
import com.vungle.warren.tasks.utility.JobRunnerThreadPriorityHelper;
import com.vungle.warren.utility.ConcurrencyTimeoutProvider;
import com.vungle.warren.utility.Executors;
import com.vungle.warren.utility.NetworkProvider;
import com.vungle.warren.utility.SDKExecutors;
import com.vungle.warren.utility.TimeoutProvider;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

class ServiceLocator {

    @SuppressLint("StaticFieldLeak")
    private static ServiceLocator INSTANCE;
    private final Context ctx;

    private Map<Class, Creator> creators = new HashMap<>();
    private Map<Class, Object> cache = new HashMap<>();

    private ServiceLocator(@NonNull Context context) {
        this.ctx = context.getApplicationContext();
        buildCreators();
    }

    static synchronized ServiceLocator getInstance(@NonNull Context context) {
        if (INSTANCE == null) {
            INSTANCE = new ServiceLocator(context);
        }

        return INSTANCE;
    }

    static synchronized void deInit() {
        INSTANCE = null;
    }

    synchronized <T> T getService(Class<T> serviceClass) {
        return getOrBuild(serviceClass);
    }

    synchronized <T> boolean isCreated(Class<T> serviceClass) {
        return cache.containsKey(getServiceClass(serviceClass));
    }

    @SuppressWarnings("unchecked")
    private <T> T getOrBuild(@NonNull Class<T> serviceClass) {
        Class target = getServiceClass(serviceClass);
        Object value = cache.get(target);

        if (value == null) {
            Creator<T> creator = creators.get(target);
            if (creator == null)
                throw new IllegalArgumentException("Unknown class");

            value = creator.create();
            if (creator.isSingleton())
                cache.put(target, value);
        }

        return (T) value;
    }


    @NonNull
    @SuppressWarnings("unchecked")
    private Class getServiceClass(@NonNull Class serviceClass) {
        for (Class clazz : creators.keySet()) {
            if (clazz.isAssignableFrom(serviceClass))
                return clazz;
        }

        throw new IllegalArgumentException("Unknown dependency for " + serviceClass);
    }


    @VisibleForTesting
    synchronized <T> void bindService(Class<T> tClass, T service) {
        Class target = getServiceClass(tClass);
        cache.put(target, service);
    }

    @VisibleForTesting
    static final VungleStaticApi VUNGLE_STATIC_API = new VungleStaticApi() {
        @Override
        public boolean isInitialized() {
            return Vungle.isInitialized();
        }

        @Override
        public Collection<String> getValidPlacements() {
            return Vungle.getValidPlacements();
        }
    };

    private void buildCreators() {
        creators.put(JobCreator.class, new Creator() {
            @Override
            public JobCreator create() {
                return new VungleJobCreator(
                        getOrBuild(Repository.class),
                        getOrBuild(Designer.class),
                        getOrBuild(VungleApiClient.class),
                        new VungleAnalytics(getOrBuild(VungleApiClient.class)),
                        RECONFIG_CALL,
                        getOrBuild(AdLoader.class),
                        VUNGLE_STATIC_API);
            }
        });
        creators.put(JobRunner.class, new Creator() {
            @Override
            public JobRunner create() {
                return new VungleJobRunner(
                        getOrBuild(JobCreator.class),
                        getOrBuild(Executors.class).getJobExecutor(),
                        new JobRunnerThreadPriorityHelper(),
                        NetworkProvider.getInstance(ctx)
                );
            }
        });
        creators.put(AdLoader.class, new Creator() {
            @Override
            public AdLoader create() {
                return new AdLoader(getOrBuild(Executors.class),
                        getOrBuild(Repository.class),
                        getOrBuild(VungleApiClient.class),
                        getOrBuild(CacheManager.class),
                        getOrBuild(Downloader.class),
                        getOrBuild(RuntimeValues.class),
                        getOrBuild(VungleStaticApi.class),
                        getOrBuild(VisionController.class),
                        getOrBuild(OperationSequence.class));
            }
        });
        creators.put(Downloader.class, new Creator() {
            @Override
            public Downloader create() {
                return new AssetDownloader(
                        getOrBuild(DownloaderCache.class),
                        AssetDownloader.VERIFICATION_WINDOW,
                        4,
                        NetworkProvider.getInstance(ctx),
                        getOrBuild(Executors.class).getUIExecutor()
                );
            }
        });
        creators.put(VungleApiClient.class, new Creator() {
            @Override
            public VungleApiClient create() {
                return new VungleApiClient(ctx,
                        getOrBuild(CacheManager.class),
                        getOrBuild(Repository.class)
                );
            }
        });
        creators.put(Repository.class, new Creator() {
            @Override
            public Repository create() {
                Executors sdkExecutors = getOrBuild(Executors.class);
                return new Repository(ctx,
                        getOrBuild(Designer.class),
                        sdkExecutors.getIOExecutor(),
                        sdkExecutors.getUIExecutor()
                );
            }
        });
        creators.put(Designer.class, new Creator() {
            @Override
            public Designer create() {
                return new GraphicDesigner(getOrBuild(CacheManager.class));
            }
        });
        creators.put(CacheManager.class, new Creator() {
            @Override
            public CacheManager create() {
                return new CacheManager(ctx);
            }
        });
        creators.put(Executors.class, new Creator() {
            @Override
            public Executors create() {
                return new SDKExecutors();
            }
        });
        creators.put(RuntimeValues.class, new Creator() {
            @Override
            public RuntimeValues create() {
                return new RuntimeValues();
            }
        });
        creators.put(VungleStaticApi.class, new Creator() {
            @Override
            public VungleStaticApi create() {
                return VUNGLE_STATIC_API;
            }
        });
        creators.put(PresentationFactory.class, new Creator() {
            @Override
            public PresentationFactory create() {
                return new AdvertisementPresentationFactory(
                        getOrBuild(AdLoader.class),
                        getOrBuild(VungleStaticApi.class),
                        getOrBuild(Repository.class),
                        getOrBuild(VungleApiClient.class),
                        getOrBuild(JobRunner.class),
                        getOrBuild(RuntimeValues.class));
            }

            @Override
            boolean isSingleton() {
                return false;
            }
        });
        creators.put(DownloaderCache.class, new Creator() {
            @Override
            Object create() {
                CacheManager cacheManager = getOrBuild(CacheManager.class);
                return new CleverCache(
                        cacheManager,
                        new LRUCachePolicy(cacheManager, CleverCache.CC_DIR),
                        new DownloaderSizeProvider(cacheManager,
                                getOrBuild(RuntimeValues.class),
                                0.1f),
                        TimeUnit.DAYS.toMillis(90)
                );
            }
        });
        creators.put(VisionController.class, new Creator() {
            @Override
            public VisionController create() {
                return new VisionController(getOrBuild(Repository.class), NetworkProvider.getInstance(ctx));
            }
        });
        creators.put(TimeoutProvider.class, new Creator() {
            @Override
            public TimeoutProvider create() {
                return new ConcurrencyTimeoutProvider();
            }
        });
        creators.put(OperationSequence.class, new Creator() {
            @Override
            public OperationSequence create() {
                return new OperationSequence();
            }
        });
    }

    private static final ReconfigJob.ReconfigCall RECONFIG_CALL = new ReconfigJob.ReconfigCall() {
        @Override
        public void reConfigVungle() {
            Vungle.reConfigure();
        }
    };

    private abstract class Creator<T> {
        abstract T create();

        boolean isSingleton() {
            return true;
        }
    }
}
