package com.vungle.warren;

import com.vungle.warren.error.VungleException;

/**
 * Callback handler for playing a Vungle advertisement. This is given as a parameter to {@link Vungle#playAd(String, AdConfig, PlayAdCallback)}
 * and is triggered when the advertisement begins to play, when the advertisement ends, and when
 * any errors occur.
 */
public interface PlayAdCallback {

    /**
     * Callback for an advertisement starting. The Vungle SDK has successfully launched the advertisement
     * activity and an advertisement will begin playing momentarily.
     *
     * @param id The Placement ID of the advertisement being played.
     */
    void onAdStart(String id);

    /**
     * Callback for an advertisement ending. The Vungle SDK has finished playing the advertisement and
     * the user has closed the advertisement.
     * @deprecated Replaced by {@link #onAdEnd(String)} and {@link #onAdClick(String)} and
     * {@link #onAdRewarded(String)}
     * @param id        The Placement ID of the advertisement that ended.
     * @param completed Flag that indicates whether or not the user watched the advertisement to completion.
     */
    @Deprecated
    void onAdEnd(String id, boolean completed, boolean isCTAClicked);

    /**
     * Callback for an advertisement ending. The Vungle SDK has finished playing the advertisement and
     * the user has closed the advertisement.
     *
     * @param id        The Placement ID of the advertisement that ended.
     */
    void onAdEnd(String id);

    /**
     * Callback for an advertisement tapped. Sent when the user has tapped on an ad.
     * @param id The Placement ID of the advertisement that tapped.
     */
    void onAdClick(String id);

    /**
     * Callback for the user has watched the advertisement to completion. The Vungle SDK has finished playing the
     * advertisement and the user has closed the advertisement.
     *
     * @param id        The Placement ID of the advertisement that ended.
     */
    void onAdRewarded(String id);

    /**
     * Callback when the user has left the app.
     * @param id The Placement ID of the advertisement that tapped.
     */
    void onAdLeftApplication(String id);

    /**
     * Callback for an error that has occurred while playing an advertisement. If this is called, the
     * error was unrecoverable by the SDK and error handling should happen at the application layer.
     * If this callback is triggered, {@link #onAdStart(String)} and {@link #onAdEnd(String, boolean, boolean)} will
     * not be called. This indicates that the advertisement has finished.
     *
     * @param id        The identifier for the advertisement placement for which the error occurred.
     * @param exception The exception that prevented the advertisement from playing.
     */
    void onError(String id, VungleException exception);
}
