package com.vungle.warren;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.PriorityQueue;
import java.util.concurrent.atomic.AtomicInteger;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

/**
 * Orders {@link com.vungle.warren.AdLoader.Operation} depending on theirs priority.
 * Operation with {@link com.vungle.warren.AdLoader.Priority#HIGHEST} will skip the queue
 * and {@link Callback#onLoadNext(AdLoader.Operation)} will be called immediately.
 *
 * If operation is currently loading for the same Placement new operation will be merged with existing
 * and {@link Callback#onChangePriority(AdLoader.Operation)} can be called.
 */
class OperationSequence {

    interface Callback {

        /**
         * Called when next operation pops from queue.
         * @param op Operation to load next
         */
        void onLoadNext(AdLoader.Operation op);

        /**
         * Called when priority was changed for already loading operation from {@link OperationSequence#loadOperations}
         * @param op Operation
         */
        void onChangePriority(AdLoader.Operation op);
    }

    private static class Entry {
        private final static AtomicInteger seq = new AtomicInteger();
        private final int order = seq.incrementAndGet();

        @NonNull
        AdLoader.Operation operation;

        Entry(@NonNull AdLoader.Operation operation) {
            this.operation = operation;
        }
    }

    private final PriorityQueue<Entry> queue = new PriorityQueue<>(11, new Comparator<Entry>() {
        //keep FIFO for Entry with same priority
        @Override
        public int compare(Entry e1, Entry e2) {
            int result = ((Integer) (e1.operation.priority)).compareTo(e2.operation.priority);
            if (result == 0) {
                return ((Integer) (e1.order)).compareTo(e2.order);
            } else {
                return result;
            }
        }
    });

    private String currentId = null;
    private Map<String, AdLoader.Operation> loadOperations;
    private Callback callback;

    public void init(Callback callback, Map<String, AdLoader.Operation> loadOperations) {
        this.callback = callback;
        this.loadOperations = loadOperations;
    }

    @Nullable
    private Entry get(String id) {
        for (Entry op : queue) {
            if (op.operation.id.equals(id)) {
                return op;
            }
        }
        return null;
    }

    /**
     * Offers operation to queue
     *
     * Check if we already have an operation for a placement_id loading
     * if it merge, merge the running operation with current, otherwise check if the same loadAd is in queue, if so merge the current
     * load ad request with the existing pending ad request
     *
     * If no ad requests are pending for that placement_id, create a new operation
     * for that placement
     *
     * @param op New operation
     */
    synchronized void offer(AdLoader.Operation op) {
        AdLoader.Operation current = loadOperations.get(op.id);
        if (current != null) {
            int oldPriority = current.priority;
            current.merge(op);
            if (current.priority < oldPriority) {
                callback.onChangePriority(current);
            }
        } else {
            Entry next = get(op.id);
            if (next != null) {
                queue.remove(next);
                next.operation.merge(op);
                op = next.operation;
            }

            if (op.priority <= AdLoader.Priority.HIGHEST) {
                callback.onLoadNext(op);
            } else {
                queue.offer(next == null ? new Entry(op) : next);
                reportFinished(null);
            }
        }
    }

    /**
     * Reports that current loading operation was loaded, starts next operation if exist.
     * Wrong id will be ignored.
     * @param finished Id of finished {@link com.vungle.warren.AdLoader.Operation}
     */
    synchronized void reportFinished(String finished) {
        if (currentId == null || currentId.equals(finished)) {
            currentId = null;
            Entry next = queue.poll();
            if (next != null) {
                currentId = next.operation.id;
                callback.onLoadNext(next.operation);
            }
        }
    }

    /**
     * Remove all operations on queue.
     * @return List of operations
     */
    synchronized List<AdLoader.Operation> removeAll() {
        currentId = null;
        ArrayList<AdLoader.Operation> ops = new ArrayList<>();
        while (!queue.isEmpty()) {
            Entry e = queue.poll();
            if (e != null) {
                ops.add(e.operation);
            }
        }
        return ops;
    }

    public synchronized boolean contains(String id) {
        return get(id) != null;
    }
}
