package com.vungle.warren;

import android.content.Context;

import com.vungle.warren.error.VungleException;

import java.util.Collection;

/**
 * Callback handler for Initialization tasks. When the Vungle SDK has successfully initialized and
 * is ready to load and play advertisements, the {@link #onSuccess()} method will be called. If
 * any errors occur during initialization that prevent the SDK from functioning, the {@link #onError(VungleException)}
 * method will be called with a {@link VungleException} that explains the error. In all cases where
 * initialization fails, the SDK will not be operable until {@link Vungle#init(Collection, String, Context, InitCallback)}
 * is called again.
 */
public interface InitCallback {

    /**
     * The success callback for SDK initialization. Once this callback is received, the SDK is ready
     * to load and play ads. Note that the auto-cached placement might still be loading when this
     * callback is received and {@link Vungle#canPlayAd(String)} should still be used before playing
     * ad advertisement.
     */
    void onSuccess();

    /**
     * The failure callback for SDK initialization. Once this callback is received, the SDK has failed
     * to initialize. The throwable parameter will detail the cause. This is most commonly an instance
     * of a {@link VungleException} and can be initialized later. The SDK
     * will not automatically retry.
     *
     * @param exception that prevented the SDK from initializing. Please see the documentation
     *            for handling these errors {@link VungleException}.
     */
    void onError(VungleException exception);

    /**
     * Vungle's SDK will automatically try to cache Ad for only one Placement that
     * is configured on our web dashboard as Auto-cached.
     * As soon as we finish playing an Ad for the auto-cached Placement
     * our SDK will automatically start caching next Ad for this Placement so its readily available
     * for next impression.
     * After successful initialisation, his callback will be available and triggered every time the
     * auto-cached Placement has an Ad available
     *
     * @param placementId auto cached placement id for which the Ad is available
     */
    void onAutoCacheAdAvailable(String placementId);
}
