package com.vungle.warren;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.google.gson.GsonBuilder;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonSyntaxException;
import com.google.gson.annotations.SerializedName;
import com.vungle.warren.model.JsonUtil;

public class CleverCacheSettings {

    final static String KEY_ENABLED = "enabled";
    final static String KEY_TIMESTAMP = "clear_shared_cache_timestamp";
    final static String KEY_CLEVER_CACHE = "clever_cache";

    static final boolean DEFAULT_ENABLED = true;
    static final long DEFAULT_TIMESTAMP = -1;

    @SerializedName(KEY_ENABLED)
    private final boolean enabled;
    @SerializedName(KEY_TIMESTAMP)
    private final long timestamp;


    private CleverCacheSettings(boolean enabled, long timestamp) {
        this.enabled = enabled;
        this.timestamp = timestamp;
    }

    public String serializeToString() {
        JsonObject jsonObject = new JsonObject();
        jsonObject.add(KEY_CLEVER_CACHE, new GsonBuilder().create().toJsonTree(this));
        return jsonObject.toString();
    }

    @Nullable
    public static CleverCacheSettings fromJson(JsonObject jsonObject) {
        if (!JsonUtil.hasNonNull(jsonObject, KEY_CLEVER_CACHE))
            return null;

        long timestamp = DEFAULT_TIMESTAMP;
        boolean enabled = DEFAULT_ENABLED;

        JsonObject ccSettings = jsonObject.getAsJsonObject(KEY_CLEVER_CACHE);
        try {
            if (ccSettings.has(KEY_TIMESTAMP)) {
                timestamp = ccSettings.get(KEY_TIMESTAMP).getAsLong();
            }
        } catch (NumberFormatException ex) {

        }

        if (ccSettings.has(KEY_ENABLED)) {
            JsonElement element = ccSettings.get(KEY_ENABLED);
            if (element.isJsonPrimitive() && "false".equalsIgnoreCase(element.getAsString())) {
                enabled = false;
            }
        }

        return new CleverCacheSettings(enabled, timestamp);
    }

    @Nullable
    static CleverCacheSettings deserializeFromString(String input){
        if (input == null)
            return null;

        JsonObject jsonObject;

        try {
            jsonObject = new GsonBuilder().create().fromJson(input, JsonObject.class);
        } catch (JsonSyntaxException ex) {
            return null;
        }

        return fromJson(jsonObject);
    }

    static CleverCacheSettings getDefault() {
        return new CleverCacheSettings(DEFAULT_ENABLED, DEFAULT_TIMESTAMP);
    }

    public boolean isEnabled() {
        return enabled;
    }

    public long getTimestamp() {
        return timestamp;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        CleverCacheSettings settings = (CleverCacheSettings) o;

        if (enabled != settings.enabled) return false;
        return timestamp == settings.timestamp;
    }

    @Override
    public int hashCode() {
        int result = (enabled ? 1 : 0);
        result = 31 * result + (int) (timestamp ^ (timestamp >>> 32));
        return result;
    }
}
